/* global alert */
(function () {
    'use strict';

    var debounce = require('lodash').debounce;

    var DynamicForm = function (params) {
        this.rules = params.rules;
        this.fieldsState = params.fieldsState;
    };

    var kladrCache = {};

    DynamicForm.prototype = {
        getControlType: function (fieldName) {
            return this.rules[fieldName].type || 'input';
        },
        valueIsSet: function (fieldName, state) {
            var fieldRules = this.rules[fieldName];

            this.fieldsState[fieldName] = state;

            if (typeof fieldRules.onChange === 'function') {
                fieldRules.onChange(state);
            }
        }
    };

    BEM.DOM.decl('b-form-address', {
        onSetMod: {
            js: function () {
                this._paramsValues = this.params.value || {};
                this.fields = new DynamicForm({
                    fieldsState: {},
                    initialState: {},
                    rules: {
                        // depending fields
                        region: {
                            fieldsUnlock: ['district'],
                            onChange: this._onRegionChange.bind(this),
                            type: 'select'
                        },
                        district: {
                            fieldsUnlock: ['city'],
                            onChange: this._onDistrictChange.bind(this),
                            type: 'select'
                        },
                        city: {
                            fieldsUnlock: ['town'],
                            onChange: this._onCityChange.bind(this),
                            type: 'select'
                        },
                        town: {
                            fieldsUnlock: ['street'],
                            onChange: this._onTownChange.bind(this),
                            type: 'select'
                        },
                        // simple fields
                        home: {
                            onChange: this._onSimpleFieldChange.bind(this)
                        },
                        street: {},
                        building: {},
                        construction: {},
                        flat: {},
                        postcode: {}
                    }
                });

                this.levels = ['region', 'district', 'city', 'town'];
                this.values = {};

                var fieldsLoaded = 0;
                var fieldsTotalToLoad = 0;
                var fieldsOptions = this._fieldsOptions = {};

                this.levels.forEach(function (fieldName) {
                    var _this = this;
                    if (this._paramsValues !== null && this._paramsValues[fieldName]) {
                        fieldsTotalToLoad++;
                        // без this.afterCurrentEvent,
                        // если значение достается из кеша, то колбэк сработает сразу и
                        // проверка fieldsLoaded == fieldsTotalToLoad будет всегда true,
                        // после этого блок считает, что все поля загружены и
                        // вешает обработчики на change блоков раньше чем нужно
                        this.afterCurrentEvent(function () {
                            this._getKladrData(this._paramsValues[fieldName], function (data) {
                                fieldsLoaded++;
                                fieldsOptions[fieldName] = data;

                                if (fieldsLoaded == fieldsTotalToLoad) {
                                    _this._onFieldsLoad();
                                }
                            });
                        }, this);
                    }
                }, this);

                if (fieldsTotalToLoad === 0) {
                    this._initBlock();
                }
            }
        },

        _initBlock: function () {
            var updatePostCode = debounce(this._updatePostCode, 500);
            updatePostCode = updatePostCode.bind(this);

            this._getControlBlock('street')
                .on('select', function (e, data) {
                    this.elem('streethidden').val(data.item.params.data[0]);
                    updatePostCode();
                }, this)
                .on('change', function () {
                    this.elem('streethidden').val('');
                    updatePostCode();
                }, this);

            this._getControlBlock('home').on('change', function () {
                updatePostCode();
            }, this);

            if (this._paramsValues) {
                this.findBlockInside('street', 'b-form-input').val(this._paramsValues.streettext || '');
                this.elem('streethidden').val(this._paramsValues.street);
            }

            // setTimeout и 1 принципиальна, т.к. в b-form-select при инициализации
            // тригерится событие change именно через 1 мс
            setTimeout(this._afterBlockInit.bind(this), 1);
        },

        _afterBlockInit: function () {
            this.levels.forEach(function (fieldName) {
                this._getControlBlock(fieldName).on('change', fieldName, function (e) {
                    this._updateField(e.data, e.target.val());
                }, this);
            }, this);
        },

        _onFieldsLoad: function () {
            var options = {};
            var fo = this._fieldsOptions;

            if (fo.region) {
                options.district = fo.region[0];
                options.city = fo.region[1];
                options.town = fo.region[2];
                options.street = fo.region[3];
            }
            if (fo.district) {
                options.city = fo.district[0];
                options.town = fo.district[1];
                options.street = fo.district[2];
            }
            if (fo.city) {
                options.town = fo.city[0];
                options.street = fo.city[1];
            }
            if (fo.town) {
                options.street = fo.town[0];
            }

            this.levels.forEach(function (fieldName) {
                var controlBlock = this._getControlBlock(fieldName);

                if (fieldName in options) {
                    this._updateSelectOptions(controlBlock, options[fieldName]);
                }
                controlBlock.val(this._paramsValues[fieldName]);
                this.values[fieldName] = this._paramsValues[fieldName];
            }, this);

            ['home', 'building', 'construction', 'flat', 'postcode'].forEach(function (fieldName) {
                if (this._paramsValues[fieldName]) {
                    this._getControlBlock(fieldName).val(this._paramsValues[fieldName]);
                    this.values[fieldName] = this._paramsValues[fieldName];
                }
            }, this);

            this._getControlBlock('street').setMod('disabled', options.street ? 'no' : 'yes');
            this._updateStreetUrl();

            this._initBlock();
        },

        _getControlBlock: function (fieldName) {
            return this.findBlockInside(fieldName, 'b-form-' + this.fields.getControlType(fieldName));
        },

        _updateField: function (fieldName, val) {
            if (val !== this.values[fieldName]) {
                this.values[fieldName] = val;
                this.fields.valueIsSet(fieldName, Boolean(val));
            }
        },

        _onFieldChange: function (fieldName, val, fieldsToUpdate) {
            if (val === null) {
                return false;
            }

            if (fieldsToUpdate !== null && typeof fieldsToUpdate !== 'undefined') {
                this._disableFields(fieldsToUpdate, true);
            }

            if (val) {
                this._getKladrData(val, this._afterFieldChange.bind(this, fieldsToUpdate));
            }

            this._updateStreetUrl();
        },

        _afterFieldChange: function (fieldsToUpdate, data) {
            if (fieldsToUpdate !== null && typeof fieldsToUpdate !== 'undefined') {
                fieldsToUpdate.forEach(function (fieldName, i) {
                    this._updateSelectOptions(this._getControlBlock(fieldName), data[i]);
                }, this);
            }

            this._enableAddrFields(
                // data[data.length - 1] == [] or null
                // in case of [] addrField should be enabled
                // in case of null - disabled.
                // Array 'fieldsToUpdate' should contain less elements
                // than 'data'. So the last value of fieldsToUpdate
                // should be street but it doesnt contain any values.
                // For example
                //   fieldsToUpdate = ['region', 'street']
                //   data           = [[...],    [...],    []]
                //                                          ^ this is addrField
                data[(fieldsToUpdate && fieldsToUpdate.length) || 0]
           );
        },

        _updateStreetUrl: function () {
            var street_dp = this._getControlBlock('street').getDataprovider();

            if (!street_dp.params.orig_url) {
                street_dp.params.orig_url = street_dp.params.url;
            }

            street_dp.params.url = street_dp.params.orig_url + '?' +
                this.levels.map(function (v) {
                    return v + '=' + this.findBlockInside(v, 'b-form-select').val();
                }, this).join('&');

            this.elem('streethidden').val('');
        },

        _updateControlOptions: function (getDataFrom, fieldsToSetOptions) {
            var sourceField,
                _this = this;

            do {sourceField = getDataFrom.shift();
                fieldsToSetOptions.unshift(null);
            } while (getDataFrom.length && !_this.values[sourceField]);

            this._getKladrData(this.values[sourceField], this._afterControlUpdate.bind(this, fieldsToSetOptions));
        },

        _afterControlUpdate: function (fieldsToSetOptions, data) {
            fieldsToSetOptions.forEach(function (fieldName, i) {
                if (fieldName) {
                    var control = this._getControlBlock(fieldName);
                    var disabled = (data[i] && (data[i].length || fieldName == 'street')) ? 'no' : 'yes';

                    if (control) {
                        if (control.setOptions) {
                            this._updateSelectOptions(control, data[i]);
                        } else {
                            control.setMod('disabled', disabled);
                            if (fieldName == 'street') {
                                control.val('');
                            }
                        }
                    }
                }
            }, this);
            this._updateStreetUrl();
        },

        _getKladrData: function (value, callback) {
            if (value in kladrCache) {
                callback(kladrCache[value]);
            } else {
                $.ajax({
                    type: 'POST',
                    url: '/kladr/towns',
                    dataType: 'json',
                    data: {code: value}
                }).success(function (data) {
                    if (typeof(data) == 'object' && data.error) {
                        alert(data.error);
                        return;
                    }

                    kladrCache[value] = data;

                    return callback(data);
                }).error(function (data) {
                    alert(data.responseText);
                });
            }
        },

        _updateSelectOptions: function (select, data) {
            var result = $.map(data || [], function (element) {
                return {
                    item: 'option',
                    value: element[0],
                    content: element[1]
                };
            });

            result.unshift({
                item: 'option',
                value: '',
                content: '--------'
            });

            select.setOptions(result);
            select.setMod('disabled', data && data.length ? 'no' : 'yes');
        },

        _enableAddrFields: function (enable) {
            var block = this._getControlBlock('street');
            block.setMod('disabled', enable ? 'no' : 'yes');
            block.val('');
        },

        _updatePostCode: function () {
            var _this = this;

            if (this._getControlBlock('home').val() && this.elem('streethidden').val()) {
                var rec_data = {
                    street: this.elem('streethidden').val(),
                    home: this.findBlockInside('home', 'b-form-input').val()
                };
                this.levels.forEach(function (v) {
                    rec_data[v] = this._getControlBlock(v).val();
                }, this);

                $.ajax({
                    type: 'POST',
                    url: '/kladr/index',
                    dataType: 'json',
                    data: rec_data
                }).success(function (data) {
                    _this._getControlBlock('postcode').val(data || '');
                });
            } else {
                _this._getControlBlock('postcode').val('');
            }
        },

        _disableFields: function (fields) {
            var _this = this;

            $.each(fields,
                function (i, fieldName) {
                    _this
                        ._getControlBlock(fieldName)
                        .setMod('disabled', 'yes')
                        .val('');
                }
           );

            return this;
        },

        _onRegionChange: function (state) {
            if (state) {
                this._onFieldChange('region', this.values.region, ['district', 'city', 'town']);
            } else {
                this._disableFields(['district', 'city', 'town', 'street']);
            }
        },

        _onDistrictChange: function (state) {
            if (state) {
                this._onFieldChange('district', this.values.district, ['city', 'town']);
            } else {
                this._updateControlOptions(['region'], ['city', 'town', 'street']);
            }
        },

        _onCityChange: function (state) {
            if (state) {
                this._onFieldChange('city', this.values.city, ['town']);
            } else {
                this._updateControlOptions(['district', 'region'], ['town', 'street']);
            }
        },

        _onTownChange: function (state) {
            if (state) {
                this._onFieldChange('town', this.values.town);
            } else {
                this._updateControlOptions(['city', 'district', 'region'], ['street']);
            }
        },

        _onSimpleFieldChange: function () {
            this._updatePostCode();
        }
    });
})();
