(function (BEM) {
    'use strict';

    var DATE_FORMAT = 'YYYY-MM-DD';
    var moment = require('moment');
    var ONE_DAY = 1000 * 60 * 60 * 24;

    /**
     * Вычисляет дату по исходной путем прибавления (вычитания) заданного количества дней.
     *
     * @param  {String}  anchor
     * @param  {Number}  delta
     * @param  {Boolean} subtract
     * @return {String}
     */
    function countDate(anchor, delta, subtract) {
        anchor = mdate(anchor);
        anchor = anchor.date(subtract ? anchor.date() - delta : anchor.date() + delta);

        return anchor.format(DATE_FORMAT);
    }

    /**
     * Считает дату в днях.
     *
     * @param  {String} date
     * @return {Number}
     */
    function days(date) {
        return Math.floor(mdate(date).valueOf() / ONE_DAY);
    }

    /**
     * Парсит дату.
     *
     * @param  {String} date
     * @return {Moment}
     */
    function mdate(date) {
        return moment(date, DATE_FORMAT);
    }

    /**
     * Считает разницу дат.
     *
     * @param  {String} to
     * @param  {String} from
     * @return {Number}
     */
    function range(to, from) {
        return days(to) - days(from);
    }

    BEM.DOM.decl('b-form-daterange', {
        onSetMod: {
            js: function () {
                this.fromControl = this.findBlockInside('from', 'b-form-datepicker');
                this.toControl = this.findBlockInside('to', 'b-form-datepicker');
                this.diffFromControl = this.findBlockInside('diffFrom', 'b-form-datepicker');
                this.diffToControl = this.findBlockInside('diffTo', 'b-form-datepicker');
                this.selectControl = this.findBlockInside('period-select', 'b-form-select');
            }
        },

        /**
         * Возвращает bemjson содержимого селекта. Дешевенькая реализация :)
         *
         * @public
         * @return {Array}
         */
        getOptions: function () {
            var current = this.selectControl.val();

            return this.selectControl.elem('option')
                .toArray()
                .map(function (option) {
                    var $option = $(option);
                    var elem = {
                        item: 'option',
                        content: $option.text(),
                        value: $option.attr('value')
                    };

                    elem.value === current && (elem.selected = 1);

                    return elem;
                });
        },

        /**
        Вариант данных для API
        @example
        var data = [
            {
                item: 'optgroup',
                label: 'Russia',
                disabled: 1,    // optional
                content: [
                    { item: 'option', value: '1', content: 'Moscow' },
                    { item: 'option', value: '1', content: 'Saint-Peterburg', disabled: 1 }
                ]
            },
            { item: 'option', value: '3', content: 'Paris', selected: 1 },
            { item: 'option', value: '7', content: 'California', disabled: 1 }
        ];
        */
        /**
         * Устанавливает новое содержимое select-а.
         *
         * @public
         * @param {Array}     data
         * @returns {BEM.DOM}
         */
        setOptions: function (data) {
            this.selectControl.setOptions(data);
            this.selectControl.dropElemCache('option');

            return this;
        },

        /**
         * Сеттит дату при выборе сравниваемых периодов.
         *
         * @protected
         * @param {BEM.b-form-datepicker} source
         */
        _updateValues: function (source) {
            if (this._period !== 'diff') {
                return false;
            }

            var values = this.val();
            var date;

            if (this.diffFromControl === source && values.from && values.to) {
                date = countDate(values.diffFrom, range(values.to, values.from));

                date !== values.diffTo && this.diffToControl.val(date);
            }

            if (this.diffToControl === source && values.from && values.to) {
                date = countDate(values.diffTo, range(values.to, values.from), true);

                date !== values.diffFrom && this.diffFromControl.val(date);
            }
        },

        /**
         * Небольшая проверка для сравнения периодов.
         * Если периоды разной длины, то изменит длину второго периода в соответствии с первым.
         *
         * @public
         */
        fixDiff: function () {
            if (this._period !== 'diff') {
                return false;
            }

            var values = this.val();
            var range1 = range(values.to, values.from);
            var range2 = range(values.diffTo, values.diffFrom);

            range1 !== range2 &&
                this.diffToControl.val(countDate(values.diffFrom, range1));
        },

        /**
         * Меняет / возвращает текущее значение.
         *
         * @public
         * @param  {Object|Undefined} v
         * @return {Object|BEM.DOM}
         */
        val: function (v) {
            if (typeof v !== 'undefined') {
                v.from && this.fromControl.val(v.from);
                v.to && this.toControl.val(v.to);
                v.diffFrom && this.diffFromControl.val(v.diffFrom);
                v.diffTo && this.diffToControl.val(v.diffTo);
                v.period && this.selectControl &&
                    this.selectControl.val(v.period);

                return this;
            } else {
                var result = {
                    from: this.fromControl.val(),
                    to: this.toControl.val(),
                    diffFrom: this.diffFromControl.val(),
                    diffTo: this.diffToControl.val()
                };

                this.selectControl && (result.period = this.selectControl.val());

                return result;
            }
        },

        /**
         * Метод, который информирует о том, что выбрано значение? :О
         *
         * @public
         * @return {Boolean}
         */
        valIsSelected: function () {
            var val = this.val();

            if (val.period) {
                if (val.period === 'diff') {
                    return Boolean(val.from) && Boolean(val.to) && Boolean(val.diffFrom) && Boolean(val.diffTo);
                }

                if (val.period === 'other') {
                    return Boolean(val.from) && Boolean(val.to);
                }
            }

            return true;
        }
    }, {
        live: function () {
            this.liveInitOnBlockInsideEvent('change', 'b-form-datepicker', function (e) {
                this._updateValues(e.block);
                this.trigger('change');
            });

            this.liveInitOnBlockInsideEvent('change', 'b-form-select', function () {
                var val = this.selectControl.val();

                this._period = val;

                this.setMod('hidden-range', val !== 'other' && val !== 'diff' ? 'yes' : '');
                this.setMod('hidden-ranges', val !== 'diff' ? 'yes' : '');
                this.trigger('change');
            });

            return false;
        }
    });

})(BEM);
