(function (BEM, window, document, $, undefined) {

    'use strict';

    var EVT_KEYDOWN = ($.browser.opera && $.browser.version < 12.10) ? 'keypress' : 'keydown';

    var DOM_VK_BACK_SPACE = 8,
        DOM_VK_TAB = 9,
        DOM_VK_RETURN = 13,
        DOM_VK_ESCAPE = 27,
        DOM_VK_SPACE = 32,
        DOM_VK_LEFT = 37,
        DOM_VK_UP = 38,
        DOM_VK_RIGHT = 39,
        DOM_VK_DOWN = 40,
        DOM_VK_DELETE = 46,
        DOM_VK_COMMA = 188;

    function getItem(type, value) {
        return {
            type: type,
            value: value
        };
    }

    function toTag(item) {
        return {
            type: 'tag',
            value: item.value
        };
    }

    function isTag(item) {
        return item.type === 'tag';
    }

    function getCaretPosition(input) {
        var caretPosition = 0, range;

        if (document.selection) {
            range = document.selection.createRange();
            range.moveStart('character', -input.value.length);
            caretPosition = range.text.length;

        } else if (input.selectionStart || input.selectionStart === '0') {
            caretPosition = input.selectionStart;
        }

        return caretPosition;
    }

    function setCaretPosition(input, position) {
        var range;

        if (document.selection) {
            range = document.selection.createRange();
            range.moveStart('character', -input.value.length);
            range.moveStart('character', position);
            range.moveEnd('character', 0);
            range.select();
        } else if (input.selectionStart || input.selectionStart === '0') {
            input.selectionStart = position;
            input.selectionEnd = position;
        }
    }

    /**
     * @param {string} content
     * @param {string} type
     * @param {string} size s|m|l|xl
     * @param {boolean} focused
     */
    function templateItem(content, type, size, focused) {
        return {
            block: 'b-form-input',
            mods: {
                multisuggest: 'yes',
                size: size
            },
            elem: 'multisuggest-item',
            elemMods: {
                focused: focused ? 'yes' : 'no',
                type: type
            },
            content: content
        };
    }

    // function templateItemFocused(content, type) {
    //     return templateItem(content, type, true);
    // }

    function templatePopup(content, id, block, mods) {
        return {
            block: 'i-popup',
            mix: [
                {
                    block: block,
                    elem: 'popup',
                    mods: mods || {},
                    js: {uniqId: id}
                }
            ],
            content: content
        };
    }

    function createPopupBlock(content, id, block, mods) {
        return $(BEM.HTML.build(templatePopup(content, id, block, mods))).bem('i-popup');
    }

    function buildPopupItems(data, suggestVersion) {
        return BEM.HTML.build($.map(data, function (data) {

            var autocompleteItem = {
                    block: 'b-autocomplete-item',
                    data: data,
                    mods: {type: $.isArray(data) ? data[0] : 'text'},
                    suggestVersion: suggestVersion
                },
                prefs;

            ($.isArray(data)) && $.isPlainObject(prefs = data.concat().pop()) && $.extend(autocompleteItem, prefs);

            return autocompleteItem;

        }));
    }

    function nonInput(x) {
        return x && x.type !== 'input';
    }

    function PubSub() {
        this._subscribers = {};
    }

    PubSub.prototype.subscribe = function (event, handler) {
        this._subscribers[event] = this._subscribers[event] || [];
        this._subscribers[event].push(handler);
        return this._subscribers[event].length - 1;
    };

    PubSub.prototype.publish = function (e, data) {
        if (this._subscribers[e]) {
            this._subscribers[e].forEach(function (subscriber) {
                subscriber(data);
            });
        }
    };

    function MultisuggestFieldModel() {
        this.items = [];
        this.focusedItemIndex = -1; // none focused
        this._pubsub = new PubSub();
        this._oldVal = null;
    }

    MultisuggestFieldModel.prototype.clean = function () {
        this.items = [];
        this.focusedItemIndex = -1;
    };

    /**
     * @param {Array} value
     */
    MultisuggestFieldModel.prototype.setValue = function (value) {
        var focusedItem = this.items[this.focusedItemIndex];
        var oldValue = JSON.stringify(this.items.filter(nonInput));
        var newValue = JSON.stringify(value.map(getItem.bind(this, 'tag')));
        if (oldValue !== newValue) {
            this.clean();
            this.addTag(value);

            if (focusedItem && focusedItem.type == 'input') {
                this.focus();
            }
        }
    };

    MultisuggestFieldModel.prototype.focus = function (index) {
        if (index) {
            this.focusedItemIndex = index;
            this._checkChanges();
        } else if (this.items.length === 0 || this.items[this.items.length - 1].type !== 'input') {
            this.focusedItemIndex = this.items.length;
            this.items.push(getItem('input', ''));
            this._checkChanges();
        } else {
            this.focusedItemIndex = this.items.length - 1;
            this._checkChanges();
        }

        if (this.focusedItemIndex > -1 && this.items[this.focusedItemIndex].type === 'input') {
            this._pubsub.publish('focus');
        }
    };

    MultisuggestFieldModel.prototype.focusPrev = function () {
        this.focusedItemIndex = Math.max(this.focusedItemIndex - 1, 0);
        this._checkChanges();
    };

    MultisuggestFieldModel.prototype.focusNext = function () {
        this.focusedItemIndex = Math.min(this.focusedItemIndex + 1, this.items.length);
        this._checkChanges();
    };

    MultisuggestFieldModel.prototype.removeFocused = function () {
        this.items.splice(this.focusedItemIndex, 1);
        this._checkChanges();
    };

    MultisuggestFieldModel.prototype.updateFocusedValue = function (value) {
        this.items[this.focusedItemIndex].value = value;
        this._pubsub.publish('valuechange');
    };

    MultisuggestFieldModel.prototype.convertFocusedToTag = function () {
        var item = this.items[this.focusedItemIndex];
        if (item.type === 'input') {
            this.items[this.focusedItemIndex] = toTag(item);
            this.items.push(getItem('input', ''));
            this.focusedItemIndex = this.items.length - 1;
            this._checkChanges();
        }
    };

    MultisuggestFieldModel.prototype.on = function (event, handler) {
        this._pubsub.subscribe(event, handler);
    };

    MultisuggestFieldModel.prototype.addTag = function (value) {
        var tempInput = null;

        if (this.items.length > 0 && this.items[this.items.length - 1].type === 'input') {
            tempInput = this.items[this.items.length - 1];
            this.items.length = this.items.length - 1;
        }

        function getTag(value) {
            return getItem('tag', value);
        }

        if (Array.isArray(value)) {
            this.items = this.items.concat(value.map(getTag));
        } else {
            this.items.push(getTag(value));
        }

        if (tempInput !== null) {
            this.items.push(tempInput);
        }

        this._checkChanges();
    };

    MultisuggestFieldModel.prototype.removeItem = function (index) {
        this.items.splice(index, 1);
        var newIndex = Math.max(Math.min(this.focusedItemIndex, this.items.length - 1), 0);
        this.focus(newIndex);
    };

    MultisuggestFieldModel.prototype.getFocusedValue = function () {
        if (this.focusedItemIndex < 0) { return ''; }
        if (this.items.length === 0) { return ''; }
        return this.items[this.focusedItemIndex].value;
    };

    MultisuggestFieldModel.prototype._checkChanges = function () {
        var currVal = JSON.stringify(this.items.filter(isTag)) + '#' + this.focusedItemIndex;
        if (currVal !== this._oldVal) {
            this._oldVal = currVal;
            this._pubsub.publish('change');
        }
    };

    BEM.DOM.decl({name: 'b-form-input', modName: 'multisuggest', modVal: 'yes'}, {
        onSetMod: {
            js: function () {
                this._haveBeenFocused = false;
                this._showPopupOnDataRecieved = true;

                this.__base.apply(this, arguments);

                // instance bound methods
                this._update = $.proxy(this._update, this);
                this._onContainerClick = $.proxy(this._onContainerClick, this);
                this._onInputKeydown = $.proxy(this._onInputKeydown, this);
                this._onInputFocus = $.proxy(this._onInputFocus, this);
                this._onInputBlur = $.proxy(this._onInputBlur, this);
                this._onTagKeydown = $.proxy(this._onTagKeydown, this);
                this._onInputKeyup = $.proxy(this._onInputKeyup, this);
                this._onInputClick = $.proxy(this._onInputClick, this);
                this._onTagClick = $.proxy(this._onTagClick, this);
                this._onItemRemoveClick = $.proxy(this._onItemRemoveClick, this);
                this._onPopupShow = $.proxy(this._onPopupShow, this);
                this._onPopupHide = $.proxy(this._onPopupHide, this);
                this._onPopupKeypress = $.proxy(this._onPopupKeypress, this);
                this._onPopupOutsideClick = $.proxy(this._onPopupOutsideClick, this);
                this._onPopupKeyDown = $.proxy(this._onPopupKeyDown, this);
                this._onPopupItemsUpdate = $.proxy(this._onPopupItemsUpdate, this);
                this._updatePopupPos = $.proxy(this._updatePopupPos, this);
                this._onEnterItem = $.proxy(this._onEnterItem, this);
                this._onLeaveItem = $.proxy(this._onLeaveItem, this);
                this._onSelectItem = $.proxy(this._onSelectItem, this);
                this._onItemMouseUp = $.proxy(this._onItemMouseUp, this);
                this._onItemMouseDown = $.proxy(this._onItemMouseDown, this);
                this._onDataReceived = $.proxy(this._onDataReceived, this);
                this._onModelChange = $.proxy(this._onModelChange, this);
                this._onModelValueChange = $.proxy(this._onModelValueChange, this);
                this._onBodyClick = $.proxy(this._onBodyClick, this);

                this._doRequest = $.debounce(this._doRequest, this.params.debounceDelay, this);

                this._model = new MultisuggestFieldModel();

                this._model.on('change', this._onModelChange);
                this._model.on('valuechange', this._onModelValueChange);
                this._model.on('focus', this._onInputFocus);

                this._addTags();

                this.bindTo('multisuggest-container', 'click', this._onContainerClick);

                this.elem('multisuggest-container')
                    .delegate('.b-form-input__multisuggest-item_type_input input', 'focus', this._onInputFocus)
                    .delegate('.b-form-input__multisuggest-item_type_input input', 'blur', this._onInputBlur)
                    .delegate('.b-form-input__multisuggest-item_type_input input', 'keydown', this._onInputKeydown)
                    .delegate('.b-form-input__multisuggest-item_type_input input', 'keyup', this._onInputKeyup)
                    .delegate('.b-form-input__multisuggest-item_type_input input', 'click', this._onInputClick)
                    .delegate('.b-form-input__multisuggest-item_type_tag input', 'keydown', this._onTagKeydown)
                    .delegate('.b-form-input__multisuggest-item_type_tag', 'click', this._onTagClick)
                    .delegate('.b-form-input__multisuggest-item-remove', 'click', this._onItemRemoveClick);
            },

            focused: {
                yes: function () {
                    this._haveBeenFocused = true;
                    this.__base.apply(this, arguments);
                    this._doRequest();
                },
                '': function () {
                    this.__base.apply(this, arguments);
                }
            }
        },

        getDataprovider: function () {
            var url = this.params.dataprovider.url;

            return this._dataprovider || (this._dataprovider = BEM.create(
                this.params.dataprovider.name || this.__self.getName() + '__dataprovider',
                $.extend(this.params.dataprovider, {
                    url: url,
                    callbackCtx: this
                })));
        },

        /**
         * Возвращает/устанавливает текущее значение
         * @param {String} [val] значение
         * @param {Object} [data] дополнительные данные
         * @returns {String|BEM} если передан параметр val, то возвращается сам блок,
         * если не передан -- текущее значение
         */
        val: function (val, data) {
            if (typeof val === 'undefined') {
                return this._val;
            }

            if (this._val != val) {
                var input = this.elem('input');
                input.val() != val && input.val(val);
                this._val = val;
                if (val !== undefined) {
                    this._addTags();
                }
                this.trigger('change', data);
            }

            return this;
        },

        _addTags: function () {
            this._model.setValue(this.elem('input').val().split(/\s*\,\s*/).filter(Boolean));
            this.trigger('change');
        },

        close: function () {
            this._getPopup().hide();
        },

        _onModelChange: function () {
            this._update();

            if (!this._haveBeenFocused) { return; }
            if (this._model.items.length === 0) { return; }
            if (this._model.focusedItemIndex < 0) { return; }

            if (this._model.items[this._model.focusedItemIndex].type === 'input') {
                this._doRequest();
            } else {
                this._getPopup().hide();
            }
        },

        _onModelValueChange: function () {
            this._doRequest();
        },

        _update: function () {
            this.delMod('focused');
            var _this = this, prevInput, currInput, caretPosition,
                size = this.getMod('size');

            function getItemHtml(size, item, index) {
                if (item.type === 'tag') {
                    if (index === _this._model.focusedItemIndex) {
                        return templateItem(item.value, 'tag', size, true);
                    }

                    return templateItem(item.value, 'tag', size);
                }

                if (item.type === 'input') {
                    if (index === _this._model.focusedItemIndex) {
                        return templateItem(item.value, 'input', size, true);
                    }

                    return templateItem(item.value, 'input', size);
                }
            }

            function getItemValue(item) { return item.value; }

            this.elem('input').val(this._model.items.filter(isTag).map(getItemValue).join(','));

            prevInput = this.elem('multisuggest-container')
                .find('.b-form-input__multisuggest-item_focused_yes input').get(0);

            if (prevInput) {
                caretPosition = getCaretPosition(prevInput);
            }

            this.elem('multisuggest-container').html(BEMHTML.apply(
                this._model.items.map(getItemHtml.bind(this, size))
            ));

            currInput = this.elem('multisuggest-container')
                .find('.b-form-input__multisuggest-item_focused_yes input').get(0);
            $(currInput).focus();
            if (currInput) {
                setCaretPosition(currInput, caretPosition);
            }
        },

        _onContainerClick: function () {
            this._model.focus();
        },

        _onInputFocus: function () {
            if (!this.hasMod('focused', 'yes')) {
                this.setMod('focused', 'yes');
            } else {
                this._updatePopupPos();
                this._getPopup().show();
            }
        },

        _onInputBlur: function () {
            this.delMod('focused');
            if (!this._mouseDownOnItem) {
                var focusedItem = this._model.items[this._model.focusedItemIndex];
                if (focusedItem.type === 'input' && focusedItem.value) {
                    this._model.convertFocusedToTag();
                    this._showPopupOnDataRecieved = false;
                }
            }
        },

        _onInputKeydown: function (e) {
            switch (e.which) {
            case DOM_VK_BACK_SPACE:
                if (getCaretPosition(e.target) === 0) {
                    this._model.focusPrev();
                }
                break;

            case DOM_VK_LEFT:
                if (getCaretPosition(e.target) === 0) {
                    this._model.focusPrev();
                }
                break;

            case DOM_VK_COMMA:
            case DOM_VK_SPACE:
            case DOM_VK_RETURN:
                if (this._curPopupItemIndex < 0 && e.target.value !== '') {
                    this._model.convertFocusedToTag();
                    e.preventDefault();
                }
                break;

            case DOM_VK_TAB:
                if (e.target.value !== '') {
                    this._model.convertFocusedToTag();
                    this._showPopupOnDataRecieved = false;
                }
                break;
            }
        },

        _onInputKeyup: function (e) {
            if (e.which != DOM_VK_ESCAPE && e.which != DOM_VK_DOWN && e.which != DOM_VK_UP) {
                this._model.updateFocusedValue(e.target.value);
            }
        },

        _onInputClick: function () {
            this._model.focus();
        },

        _onTagKeydown: function (e) {
            switch (e.which) {
            case DOM_VK_DELETE:
                this._model.removeFocused();
                break;

            case DOM_VK_BACK_SPACE:
                this._model.removeFocused();
                this._model.focusPrev();
                break;

            case DOM_VK_UP:
            case DOM_VK_LEFT:
                this._model.focusPrev();
                break;

            case DOM_VK_DOWN:
            case DOM_VK_RIGHT:
            case DOM_VK_TAB:
                this._model.focusNext();
                e.preventDefault();
                break;
            }
        },

        _onTagClick: function (e) {
            this._model.focus($(e.target).index());
            e.stopPropagation();
        },

        _onItemRemoveClick: function (e) {
            var itemElement = $(e.target).parent('.b-form-input__multisuggest-item');
            var index = itemElement.index();
            this._model.removeItem(index);
            e.stopPropagation();
        },

        // popup

        _doRequest: function () {
            if (!this._haveBeenFocused) { return; }
            var provider = this.getDataprovider(),
                value = this._model.getFocusedValue();

            this.enablePopup();
            this.trigger('data-requested');

            provider.get(value, this._onDataReceived);
        },

        _onDataReceived: function (data) {
            this.trigger('data-received', data);

            var popup = this._getPopup(),
                dataItems = data.items || data;

            if (dataItems.length && this._showPopupOnDataRecieved) {
                this._curPopupItemIndex = -1;
                BEM.DOM.update(
                    popup.elem('items'),
                    buildPopupItems(dataItems, this.params.dataprovider.version),
                    this._onPopupItemsUpdate
                );
            } else {
                this._showPopupOnDataRecieved = true;
                popup.hide();
            }
        },

        _getPopup: function () {
            var _this = this;

            if (this._popup) { return this._popup; }

            var block = this.__self.getName();

            var content = [
                {elem: 'items', tag: 'ul',  mix: [{block: block, elem: 'popup-items'}]},
                {block: 'b-form-input', elem: 'shadow', tag: 'i'}
            ];

            if ((this.params.popupMods || {}).fade == 'yes') {
                content.push({block: block, elem: 'fade'});
            }

            this._popup = createPopupBlock(content, this._uniqId, block, this.params.popupMods);
            this._popup.on('show', this._onPopupShow);
            this._popup.on('outside-click', this._onPopupOutsideClick);
            this._popup.on('hide', this._onPopupHide);

            $.each({
                    mouseover: this._onEnterItem,
                    mouseout: this._onLeaveItem,
                    mouseup: this._onItemMouseUp,
                    mousedown: this._onItemMouseDown
                }, function (e, fn) {
                    BEM.blocks['b-autocomplete-item'].on(_this._popup.domElem, e, function (e) {
                        fn(e.block);
                    });
                });

            BEM.DOM.init(this._popup.domElem);

            return this._popup;
        },

        _onEnterItem: function (item, byKeyboard) {
            if (item.hasMod('enterable', 'no')) { return false; }

            var items = this._popupItems,
                index = this._curPopupItemIndex;

            index > -1 && items[index].delMod('hovered');
            index = this._getPopupItemIndex(item);
            index > -1 && items[index].setMod('hovered', 'yes');

            this._curPopupItemIndex = index;
            this._isCurItemEnteredByKeyboard = Boolean(byKeyboard);

            if (byKeyboard && this.params.updateOnEnter) {
                this._preventRequest = true;
                this
                    .val(
                        item.enter() !== false ? item.val(): this._userVal,
                        {source: 'autocomplete', itemIndex: this._curPopupItemIndex})
                    .del('_preventRequest');
            }
        },

        _onLeaveItem: function (item, byKeyboard) {
            var index = this._curPopupItemIndex;
            if (index > -1 && index == this._getPopupItemIndex(item)) {
                this._popupItems[index].delMod('hovered');
                this._curPopupItemIndex = -1;
            }

            byKeyboard && this.val(this._userVal);
        },

        _onSelectItem: function (item, byKeyboard) {
            var selectResult = item.select(byKeyboard || false),
                needEvent = (typeof selectResult == 'object') && selectResult.needEvent,
                needUpdate = (typeof selectResult == 'object') ?
                                selectResult.needUpdate :
                                selectResult !== false;

            this._preventRequest = true;
            this._preventBlur = true;

            if (needUpdate) {
                this._model.updateFocusedValue(item.val());
                this._model.convertFocusedToTag();
                this._update();
            }

            if (byKeyboard) {
                this.del('_preventRequest');
            } else {
                needUpdate || (this._preventHide = true);
                this.afterCurrentEvent(function () {
                    this.del('_preventRequest', '_preventHide');
                });
            }

            (needUpdate || needEvent) && this.trigger('select', {item: item, byKeyboard: byKeyboard});

            this._getPopup().hide();
        },

        _onItemMouseUp: function (item) {
            if (this._mouseDownOnItem) {
                this._onSelectItem(item, false);
                this.del('_preventBlur');
                this._getPopup().hide();
            }
            this._mouseDownOnItem = false;
        },

        _onItemMouseDown: function () {
            this._mouseDownOnItem = true;
        },

        _onPopupKeypress: function (e) {
            if (e.keyCode == 13) {
                if (this._curPopupItemIndex > -1 && this._isCurItemEnteredByKeyboard) {
                    e.preventDefault();
                    this._onSelectItem(this._popupItems[this._curPopupItemIndex], true);
                }

                this._getPopup().hide();
            }
        },

        _onPopupKeyDown: function (e) {
            var isArrow = e.keyCode == 38 || e.keyCode == 40;

            if (isArrow && !e.shiftKey) {
                e.preventDefault();
                var len = this._popupItems.length,
                    out = false;
                if (len) {
                    var direction = e.keyCode - 39, // пользуемся особенностями кодов клавиш "вверх"/"вниз" ;-)
                        index = this._curPopupItemIndex,
                        i = 0;

                    do {
                        // если выбор перемещается с крайнего элемента вовне списка,
                        // то ставим фокус на инпут и возвращаем в него пользовательское значение
                        out = ((index === 0 && direction == -1) || (index + direction >= len)) &&
                            this._onLeaveItem(this._popupItems[index], true);

                        index += direction;
                        index = index < 0 ? len - 1 : index >= len ? 0 : index;
                    } while (!out && this._onEnterItem(this._popupItems[index], true) === false && ++i < len);
                }
            }
        },

        _updatePopupPos: function () {
            var box = this.elem('multisuggest-container')
                    .find('.b-form-input__multisuggest-item_focused_yes'), // TODO: use this.element
                css;

            if (box.length > 0) {
                css = box.offset();
                css.top += box.outerHeight() + 2;
                this.hasMod(this.elem('popup'), 'fixed') && (css.top -= BEM.DOM.win.scrollTop());
                this._hasPopupFade() && (css.width = box.outerWidth());
                this._preventPopupShow || this._getPopup().show(css);
                box.focus();
            }
        },

        _onPopupShow: function () {
            this.trigger('popup-shown');
            this.bindTo('keypress', this._onPopupKeypress);
            this.bindTo(EVT_KEYDOWN, this._onPopupKeyDown);
            this.bindToWin('resize', this._updatePopupPos);
            this._isPopupShown = true;
        },

        _onPopupOutsideClick: function (e, data) {
            if (this.containsDomElem($(data.domEvent.target))) {
                e.preventDefault();
            }
        },

        _onPopupHide: function () {
            this.trigger('popup-hidden');
            this.unbindFrom('keypress ' + EVT_KEYDOWN);
            this.unbindFromWin('resize');
            this.bindTo('click', this._getHiddenPopup);
            this._curPopupItemIndex = -1;

            this._isPopupShown = false;
        },

        _onPopupItemsUpdate: function () {
            this._updatePopupPos();
            this._popupItems = this._getPopup().findBlocksInside('b-autocomplete-item');
            this.trigger('update-items');
        },

        _getPopupItemIndex: function (item) {
            return $.inArray(item, this._popupItems);
        }
    });

})(BEM, window, document, jQuery);
