(function (BEM, BEMHTML, $) {
    'use strict';

    var ajax = BEM.blocks['i-promises'].ajax;

    var Brands = function () {
        this._brands = [];
    };

    Brands.prototype.getBrandsById = function(bid) {
        return this._brands.filter(function (brand) {
            return Number(brand.bid) === Number(bid);
        });
    };

    Brands.prototype.getData = function() {
        return this._brands.map(function (brand) {
            return {
                bid: brand.bid,
                cpm: (brand.blocked === false) && brand.cpm ? brand.cpm : null,
                blocked: typeof brand.blocked !== 'undefined' ? brand.blocked : true
            };
        });
    };

    Brands.prototype.getInvalidBrands = function(validationFn) {
        return this._brands.filter(validationFn);
    };

    Brands.prototype.add = function(brand) {
        var existingBrands = this.getBrandsById(brand.bid);

        if (existingBrands.length === 0) {
            this._brands.push(brand);
            return brand;
        }

        return null;
    };

    Brands.prototype.removeById = function(bid) {
        this._brands = this._brands.filter(function (brand) {
            return Number(brand.bid) !== Number(bid);
        });
    };

    Brands.prototype.applyById = function(bid, fn, context) {
        this._brands.forEach(function (brand) {
            if (Number(brand.bid) === Number(bid)) {
                fn.call(context, brand);
            }
        });
    };

    BEM.DOM.decl('b-media-brands', {
        onSetMod: {
            js: function () {
                this.input = this.findBlockInside('input', 'b-form-input');
                this.saveButton = this.findBlockInside('save-button', 'b-form-button');
                this.saveSpin = this.findBlockInside('save-spin', 'b-spin');
                this.brandList = this.elem('brand-list');
                this.dataprovider = this.input.getDataprovider();

                this.input.on('select', this.onAutocompleteItemSelected, this);
                this.saveButton.on('click', this.onSaveButtonClicked, this);

                this.brands = new Brands();
                var _this = this;
                if (this.params.brands) {
                    var brands = JSON.parse(this.params.brands);
                    brands.forEach(function (brand) {
                        _this.addBrand(brand);
                    });
                }

                // чтобы по закрытию дочернего попапа родительский не закрывался
                this.input.getPopup().domElem.on('click', function (e) {
                    e.stopPropagation();
                });

                this.popup = this
                    .findBlockOutside('b-popupa')
                    .on('hide', this.rmError, this);
            }
        },

        _buildBrandHtml: function (brand) {
            return BEMHTML.apply({
                block: 'b-media-brand-editor',
                mods: {state: (brand.blocked !== false) ? 'blocked' : 'cpm'},
                name: brand.name,
                cpm: brand.cpm,
                blocked: brand.blocked,
                js: {
                    brand: brand
                }
            });
        },

        addBrand: function (brand) {
            brand = this.brands.add(brand);
            if (brand) {
                var brandEditor = $(this._buildBrandHtml(brand));
                this.brandList.append(brandEditor);
                BEM.DOM.init(brandEditor);
                brand.block = this.findBlockOn(brandEditor, 'b-media-brand-editor');
                brand.block.on('remove', this.onBrandRemoved, this);
                brand.block.on('statechange', this.onBrandStateChanged, this);
                brand.block.on('cpmchange', this.onBrandCpmChanged, this);
            }
        },

        getDataToSend: function() {
            return {
                id: this.params.id,
                atype: this.params.type,
                sign: this.params.sign,
                ajax: 1,
                brands: JSON.stringify(this.brands.getData())
            };
        },

        onBrandRemoved: function (e, data) {
            this.brands.removeById(data.bid);
        },

        onBrandStateChanged: function (e, data) {
            var state = data.value;
            this.brands.applyById(data.bid, function (brand) {
                brand.blocked = state === 'blocked' ? true : false;
            }, this);
        },

        onBrandCpmChanged: function (e, data) {
            var value = data.value;
            this.brands.applyById(data.bid, function (brand) {
                brand.cpm = value;
            }, this);
        },

        onAutocompleteItemSelected: function (e, data) {
            var brand = data.item.params.brand;

            this.input.val('');
            this.addBrand(brand);

        },

        onSaveButtonClicked: function () {
            var invalidNodes = this.brands.getInvalidBrands(function (brand) {
                return brand.blocked === false && (brand.cpm === '' || isNaN(brand.cpm) || Number(brand.cpm) < 0);
            });

            this.rmError();

            if (invalidNodes.length > 0) {
                this.setMod('has-errors', 'yes');

                invalidNodes.forEach(function (brand) {
                    brand.block.setMod('wrong-cpm', 'yes');
                });

                return;
            } else {
                this.delMod('has-errors');
            }

            ajax({
                type: 'POST',
                url: this.params.saveUrl,
                dataType: 'json',
                data: this.getDataToSend()
            })
                .then(this.onBrandsSaved, this)
                .fail(this.onError, this);

            this.disable();
            this.saveSpin.setMod('progress', 'yes');
        },

        onBrandsSaved: function () {
            this.enable();
            this.saveSpin.delMod('progress');

            this.popup.hide();
        },

        onError: function (obj) {
            this.enable();
            this.saveSpin.delMod('progress');

            var error = typeof obj.error === 'string' && obj.error ||
                obj.status + ' ' + obj.statusText;

            BEM.DOM.after(this.elem('error'), BEMHTML.apply({
                block: 'b-media-brands',
                elem: 'server-error',
                tag: 'span',
                content: error
            }));
        },

        disable: function () {
            var blocks = ['b-form-button', 'b-form-input', 'b-form-select'];
            var cache = [];

            blocks.forEach(function (blockName) {
                cache = cache.concat(this.findBlocksInside(blockName));
            }, this);

            cache.forEach(function (block) {
                block.setMod('disabled', 'yes');
            }, this);

            this.blockedControls = cache;
        },

        enable: function () {
            if (this.blockedControls) {
                this.blockedControls.forEach(function (block) {
                    block.delMod('disabled');
                }, this);

                this.blockedControls = null;
            }
        },

        rmError: function () {
            var elem = this.findElem('server-error');

            if (elem.length) {
                elem.remove();
            }
        }
    });
})(BEM, BEMHTML, jQuery);
