(function (BEM, BEMHTML, $) {
    'use strict';

    var ajax = BEM.blocks['i-promises'].ajax;

    var Tree = function (list, userData) {
        this._list = list.slice(0);
        this._tree = this._findAllChildren();
        if (userData) {
            this._applyUserData(userData);
        }
        this._nodeCache = {};
    };

    Tree.prototype._findAllChildren = function (id) {
        var childElms = [];
        id = id || 0;

        childElms = this._list.filter(function (elem) {
            return Number(elem.parent) === Number(id);
        });

        for (var i = 0; i < childElms.length; i++) {
            childElms[i].childElms = this._findAllChildren(childElms[i].id);
        }

        return childElms;
    };

    Tree.prototype._applyUserData = function(userData) {
        this._list.forEach(function (listElem) {
            var matchedElem = userData.filter(function (dataElem) {
                return Number(dataElem.id) === Number(listElem.id);
            })[0] || null;

            if (matchedElem && (matchedElem.cpm || matchedElem.cpm === 0)) {
                listElem.selected = true;
                listElem.cpmValue = matchedElem.cpm;
            }
        });
    };

    Tree.prototype._setBlockToNode = function (block) {
        var id = block.params.id;
        var node = this.getNodeById(id);
        if (node) {
            node.block = block;
        }
    };

    /**
     * Связывает ноды дерева с БЕМ-блоками
     *
     * @param {array:BEM-block} blocks Массив БЕМ-блоков
     **/
    Tree.prototype.setBlocks = function (blocks) {
        var _this = this;
        this._blocks = blocks.slice(0);
        this._blocks.forEach(function (elem) {
            _this._setBlockToNode(elem);
        });
    };

    Tree.prototype.setPropToNodeById = function(prop, id, value) {
        var node = this.getNodeById(id);
        node[prop] = value;
    };

    Tree.prototype.getNodeById = function (id) {
        if (this._nodeCache[id]) {
            return this._nodeCache[id];
        }

        if (Number(id) === 0) {
            return {childElms: this._tree};
        }

        var node = this._list.filter(function (elem) {
            return Number(elem.id) === Number(id);
        })[0] || null;

        this._nodeCache[id] = node;

        return node;
    };

    Tree.prototype.getCpmValueById = function (id) {
        var node = this.getNodeById(id);
        if (!node) {
            return null;
        }

        if (node.cpmValue || node.cpmValue === 0) {
            return node.cpmValue;
        } else if (node.parent) {
            return this.getCpmValueById(node.parent);
        } else {
            return null;
        }
    };

    Tree.prototype.applyToAllChildrenById = function (id, fn, ctx) {
        var node = this.getNodeById(id);
        var child;
        ctx = ctx || this;
        if (node) {
            for (var i = 0; i < node.childElms.length; i++) {
                child = node.childElms[i];
                fn.call(ctx, child);
                if (child.childElms) {
                    this.applyToAllChildrenById(child.id, fn, ctx);
                }
            }
        }
    };

    Tree.prototype.applyToSelectedAndParents = function(id, fn, ctx) {
        id = id || 0;
        ctx = ctx || this;
        var node = this.getNodeById(id);
        var hasSelectedChilds = false;

        if (node.childElms.length > 0) {
            for (var i = 0; i < node.childElms.length; i++) {
                hasSelectedChilds = this.applyToSelectedAndParents(node.childElms[i].id, fn, ctx) || hasSelectedChilds;
            }
        }

        fn.call(ctx, node, hasSelectedChilds);
        return hasSelectedChilds || node.selected;
    };

    /**
     * Возвращает массив невалидных нод
     *
     * @param {function} validationFn Функция, возвращающая true в случае невалидной ноды и false в случае валидной
     **/
    Tree.prototype.getInvalidNodes = function(validationFn) {
        return this._list.filter(validationFn);
    };

    Tree.prototype.getSelectedNodes = function() {
        var result = this._list.filter(function (node) {
            return node.selected === true && node.cpmValue;
        });
        return result.map(function (node) {
            return {
                cpm: node.cpmValue,
                id: node.id
            };
        });
    };

    Tree.prototype.getTemplate = function (tree) {
        tree = tree || this._tree;

        var template = {
            block: 'b-menu',
            mods: {layout: 'vert', 'is-bem': 'yes'},
            content: []

        };

        for (var i = 0; i < tree.length; i++) {
            if (tree[i].childElms.length === 0) {
                template.content.push({
                    elem: 'item',
                    id: tree[i].id,
                    mix: [{block: 'b-media-topics', elem: 'topic'}],
                    content: {
                        block: 'b-media-topic-editor',
                        mods: {editable: tree[i].selected === true ? 'yes' : ''},
                        label: tree[i].name,
                        id: tree[i].id,
                        selected: tree[i].selected,
                        cpm: tree[i].cpmValue
                    }
                });
            } else {
                template.content.push({
                    elem: 'item',
                    id: tree[i].id,
                    mix: [{block: 'b-media-topics', elem: 'topic'}],
                    content: [
                        {
                            block: 'b-link',
                            mods: {pseudo: 'yes'},
                            mix: [{block: 'b-menu', elem: 'trigger'}],
                            content: [
                                {
                                    block: 'b-icon',
                                    mix: [{block: 'b-menu', elem: 'trigger-ico'}]
                                }
                            ]
                        },
                        {
                            block: 'b-media-topic-editor',
                            mods: {editable: tree[i].selected === true ? 'yes' : ''},
                            label: tree[i].name,
                            id: tree[i].id,
                            selected: tree[i].selected,
                            cpm: tree[i].cpmValue
                        },
                        {
                            elem: 'item-content',
                            mix: [{block: 'b-media-topics', elem: 'topic-content'}],
                            content: this.getTemplate(tree[i].childElms)
                        }
                    ]
                });
            }
        }

        return template;
    };

    Tree.prototype.getHTML = function () {
        return BEMHTML.apply(this.getTemplate());
    };

    BEM.DOM.decl('b-media-topics', {
        onSetMod: {
            js: function () {
                this.params.articles = this.params.articles ? JSON.parse(this.params.articles) : null;

                this.formSelect = this.findBlockInside('b-form-select');
                this.saveButton = this.findBlockInside('save-button', 'b-form-button');
                this.saveSpin = this.findBlockInside('save-spin', 'b-spin');
                this.formSelect.on('change', this.onShowParamsChanged, this);
                this.saveButton.on('click', this.onSaveButtonClicked, this);
            },
            'show-selected': function (modName, modVal) {
                var value = modVal === 'yes';
                this.topics.applyToSelectedAndParents(0, function (node, hasSelectedChilds) {
                    if (node.block) {
                        var topic = node.block.domElem.closest(this.buildSelector('topic'));
                        var topicContent = topic.find(this.buildSelector('topic-content'));
                        var topicContentElem = this.findElem(topicContent, 'topic-content');

                        if (!hasSelectedChilds) {
                            this.setMod(topicContentElem, 'all-filtered', (value === true) ? 'yes' : '');
                            if (!node.selected) {
                                value === true ? topic.hide() : topic.show();
                            }
                        }
                    }
                }, this);
            },
            'delay-js': {
                inited: function () {
                    this.getTopics();
                }
            }
        },

        onShowParamsChanged: function (e, data) {
            //var show_all = 0;
            var show_selected = 1;
            var index = data.index;

            this.setMod('show-selected', index !== show_selected ? '' : 'yes');
        },

        onSaveButtonClicked: function () {
            var invalidNodes = this.topics.getInvalidNodes(function (node) {
                if (!node.selected) {
                    return false;
                }
                if (node.selected && !isNaN(node.cpmValue) && Number(node.cpmValue) > 0) {
                    return false;
                }
                return true;
            });

            if (invalidNodes.length > 0) {
                this.setMod('has-errors', 'yes');

                invalidNodes.forEach(function (node) {
                    node.block.setMod('wrong-cpm', 'yes');
                });
                return;
            } else {
                this.delMod('has-errors');
            }

            ajax({
                type: 'POST',
                url: this.params.saveUrl,
                dataType: 'json',
                data: this.getDataToSend()
            })
                .then(this.onUserDataSaved, this)
                .fail(this.onError, this);

            this.saveButton.setMod('disabled', 'yes');
            this.saveSpin.setMod('progress', 'yes');
        },

        onUserDataSaved: function (data) {
            this.saveSpin.delMod('progress');
            this.saveButton.delMod('disabled');
            if (typeof data.error !== 'undefined') {
                throw new Error(data.error);
            } else {
                this.trigger('saved');
            }
        },

        getDataToSend: function () {
            var articles = this.topics.getSelectedNodes();

            return {
                id: this.params.id,
                atype: this.params.type,
                sign: this.params.sign,
                ajax: 1,
                articles: JSON.stringify(articles)
            };
        },

        getTopics: function () {
            ajax({
                url: this.params.getUrl,
                context: this
            })
                .then(this.onTopicsFetched, this)
                .fail(this.onError, this);
        },

        onTopicsFetched: function (data) {
            var _this = this;
            this.topics = new Tree(data, this.params.articles);
            this.elem('tree-container').html(this.topics.getHTML());
            BEM.DOM.init(this.elem('tree-container'));

            var blocks = this.findBlocksInside('b-media-topic-editor');
            this.topics.setBlocks(blocks);

            $.each(blocks, function (index, el) {
                el.on('cpmchange', _this.onCpmChanged, _this);
                el.on('selectedchange', _this.onSelectedChange, _this);
            });
        },

        onError: function (data) {
            window.alert('Server error:' + data);
        },

        onCpmChanged: function (e, data) {
            var node = this.topics.getNodeById(data.id);
            this.topics.setPropToNodeById('cpmValue', data.id, data.value);
            this.topics.applyToAllChildrenById(node.parent, function (node) {
                var topicEditor = node.block;
                var value = this.topics.getCpmValueById(node.id) || '';
                topicEditor.setCpmValue(value);
            }, this);
        },

        onSelectedChange: function (e, data) {
            this.topics.setPropToNodeById('selected', data.id, data.selected);
        }
    });

})(BEM, BEMHTML, jQuery);
