(function (BEM) {
    'use strict';

    var lodash = require('lodash');
    var find = lodash.find;

    /**
     * Implements simple olap logic
     * @param {Array} data     Array of fact objects
     * @param {Object} metadata Metadata for fact objects
     */
    var DataSource = function (data, metadata) {
        this.data = data;
        this.metadata = metadata;
        this.dimNames = metadata.dimNames;
        this.metricName = metadata.metric;
        this.calendarName = metadata.calendar;
        this.metricType = 'number';

        this.groupBy = [];
        this.filters = [];
        this.actualCache = false;
    };

    /**
     * Max count of series for chart to display
     * @type {Number}
     */
    DataSource.prototype.maxSeriesCount = 100;

    /**
     * Max count of categories for chart to display
     * @type {Number}
     */
    DataSource.prototype.maxCategoriesCount = 100;

    /**
     * Generate index key
     * Works like this:
     *     ['date', 'type'] + {date: 'today', type: 1, count: 1337} => 'date:today|type:1'
     *
     * @param  {Array} dims Array of dimensions to iterate
     * @param  {Object} fact
     * @return {String}      Index key
     */
    DataSource.prototype._genIndexKey = function (dims, fact) {
        var indexKey = '';
        if (dims.length === 0) {
            indexKey = this.metricName;
        } else {
            indexKey = dims.reduce(function (key, dim) {
                key += (dim + ':' + fact[dim]);
                if (dims.indexOf(dim) !== dims.length - 1) {
                    key += '|';
                }
                return key;
            }, '');
        }

        return indexKey;
    };

    /**
     * Transform index key to fact object
     * Works like this:
     *     'date:today|type:1' => {date: 'today', type: '1'}
     *
     * @param  {String} indexKey Index key
     * @return {Object}          Fact object
     */
    DataSource.prototype._parseIndexKey = function (indexKey) {
        var fact = {};
        if (indexKey !== this.metricName) {
            var els = indexKey.split('|');
            els.forEach(function (el) {
                var dim = el.split(':')[0];
                var dimEl = el.split(':')[1];
                fact[dim] = dimEl;
            });
        }
        return fact;
    };

    /**
     * Replace dimension ids in index key with its names
     * Works like this:
     *     'date:today|type:1' => 'Error date:today|Error type:1'
     *
     * @param  {String} indexKey Index key with dimension ids
     * @return {String}          Index key with dimension names
     */
    DataSource.prototype._applyNames = function (indexKey) {
        var idsToFormat = ['ErrorType'];
        var maxLength = 20;
        var els = indexKey.split('|');
        var _this = this;
        els = els.map(function (el) {
            var dim = el.split(':')[0];
            var dimEl = el.split(':')[1];
            if (idsToFormat.indexOf(dim) > -1) {
                var formattedName = _this.formatDimElForHTML(dim, dimEl);
                return formattedName.length > maxLength ? formattedName.slice(0, maxLength) + '...' : formattedName;
            } else {
                return _this.getDimNamesById(dim) + (dimEl ? (':' + dimEl) : '');
            }
        });
        return els.join('|');
    };

    /**
     * Hash of fuctions, that implement filters logic
     *
     * @type {Object}
     */
    DataSource.prototype._OpFns = {
        // Some implicit operators logic is driven by b-simple-search's getConfig() method

        '=': function (op1, op2) {
            if (Array.isArray(op2)) {
                return op2.indexOf(op1) > -1;
            } else {
                return op1 === op2;
            }
        },

        '>': function (op1, op2) {
            return Number(op1) > Number(op2);
        },

        '<': function (op1, op2) {
            return Number(op1) < Number(op2);
        },

        IN: function (op1, op2) {
            if (!Array.isArray(op2)) {
                op2 = [op2];
            }
            return op2.indexOf(op1) > -1;
        },

        '<>': function (op1, op2) {
            if (!Array.isArray(op2)) {
                op2 = [op2];
            }
            return op2.indexOf(op1) === -1;
        }
    };

    /**
     * Return function, that implement filter logic
     *
     * @param  {String} operator Filter operator
     * @return {Function}          Function, that implement filter logic
     */
    DataSource.prototype._getOpFn = function (operator) {
        return this._OpFns[operator];
    };

    /**
     * Return true if fact object matches current filters
     *
     * @param  {Object} fact Fact object
     * @return {Boolean}      Matching flag
     */
    DataSource.prototype._matchFilters = function (fact) {
        var _this = this;
        var dim = '';
        var op = '';
        var value;

        return this.filters.every(function (filter) {
            dim = filter[0];
            op = _this._getOpFn(filter[1]);
            value = filter[2];

            return op(fact[dim], value);
        });
    };

    /**
     * Return name of dimension (or array of dimensions) by its id
     *
     * @param  {String|Array} dims Dimensions
     * @return {String|Array}      Names
     */
    DataSource.prototype.getDimNamesById = function (dims) {
        var _this = this;
        if (!Array.isArray(dims)) {
            dims = [dims];
        }

        return dims.map(function (dim) {
            return _this.dimNames[dim];
        });
    };

    /**
     * Format dimension elements value
     *
     * @param  {String} dim   Dimension
     * @param  {String} dimEl Dimension element
     * @return {String}       Formatted dimension element
     */
    DataSource.prototype.formatDimElForHTML = function (dim, dimEl) {
        var dimMetadata = this.metadata.dimensions[dim];
        if (dim === this.metricName) {
            dimMetadata = {type: this.metricType};
        }
        if (dimMetadata.noFormat) {
            return dimEl;
        }

        switch (dimMetadata.type) {
        case 'number':
            return BEMHTML.apply({
                block: 'b-formatter',
                mods: {type: 'number'},
                value: dimEl
            });

        case 'date':
            return BEMHTML.apply({
                block: 'b-formatter',
                mods: {type: 'date'},
                value: {date: dimEl, type: 'day'},
                separator: ' '
            });

        case 'dictionary':
            var formattedEl = '';
            var dimValue = find(dimMetadata.values, function (el) {
                return String(dimEl) === String(el.id);
            });
            if (dimValue) {
                formattedEl = dimEl + '. ' + dimValue.label;
            } else {
                formattedEl = dimEl + '. ' + BEM.I18N('b-olap', 'An unknown error');
            }
            return formattedEl;

        default:
            return dimEl;
        }
    };

    /**
     * Format dimension elements value
     *
     * @param  {String} dim   Dimension
     * @param  {String} dimEl Dimension element
     * @return {String}       Formatted dimension element
     */
    DataSource.prototype.formatDimElForExcel = function (dim, dimEl) {
        var dimMetadata = this.metadata.dimensions[dim];
        if (dim === this.metricName) {
            dimMetadata = {type: this.metricType};
        }
        if (dimMetadata.noFormat) {
            return dimEl;
        }

        switch (dimMetadata.type) {
        case 'dictionary':
            var formattedEl = '';
            if (dimMetadata.values[dimEl]) {
                formattedEl = dimEl + '. ' + dimMetadata.values[dimEl].label;
            } else {
                formattedEl = dimEl + '. ' + BEM.I18N('b-olap', 'An unknown error');
            }
            return formattedEl;

        default:
            return dimEl;
        }
    };

    /**
     * Calculate current metric according to dimension groups and filters
     *
     * @return {Array} Calculated metric
     */
    DataSource.prototype.calculateMetric = function () {
        var _this = this;
        if (this.actualCache === true) {
            return this.metricCache;
        }

        var index = {};
        var data = this.data;
        var groupBy = this.groupBy;
        var metricName = this.metricName;

        data.forEach(function (fact) {
            if (_this._matchFilters(fact)) {
                var indexKey = _this._genIndexKey(groupBy, fact);
                index[indexKey] = index[indexKey] || 0;
                index[indexKey] += Number(fact[metricName]);
            }
        });

        var metric = [];

        var keys = Object.keys(index);
        keys.forEach(function (key) {
            var metricFact = _this._parseIndexKey(key);
            metricFact[metricName] = index[key];
            metric.push(metricFact);
        });

        this.metricCache = metric;
        this.actualCache = true;
        return metric;
    };

    /**
     * Calculate current metric according to dimension groups and filters
     * and formats dimensions element values
     *
     * @param {function} formatter function used to format dimension element value
     * @return {Array} Calculated metric
     */
    DataSource.prototype.calculateAndFormatMetric = function (formatter) {
        var metric = this.calculateMetric();

        return metric.map(function (fact) {
            return Object.keys(fact).reduce(function (result, dim) {
                result[dim] = formatter(dim, fact[dim]);
                return result;
            }, {});
        });
    };

    /**
     * Format for HTML
     *
     * @return {Array} Calculated metric
     */
    DataSource.prototype.calculateMetricForHTML = function () {
        return this.calculateAndFormatMetric(this.formatDimElForHTML.bind(this));
    };

    /**
     * Format for Excel
     *
     * @return {Array} Calculated metric
     */
    DataSource.prototype.calculateMetricForExcel = function () {
        return this.calculateAndFormatMetric(this.formatDimElForExcel.bind(this));
    };

    /**
     * Set array of dimensions to group by
     *
     * @param {Arra} groupBy Array of dimensions
     */
    DataSource.prototype.setGroupBy = function (groupBy) {
        this.groupBy = groupBy;
        this.actualCache = false;
    };

    /**
     * Set data to this data source
     * Data looks like:
     *     [{fact}, {fact}, {fact}, ... , {fact}]
     *
     * Fact has this structure:
     *     {dimension: dimensionElement, dimension: dimensionElement, metric: metricValue}
     *
     * @param {Array} data Data to set
     */
    DataSource.prototype.setData = function (data) {
        this.data = data;
        this.actualCache = false;
    };

    /**
     * Return true if this data source has data
     * @return {Boolean} Has data flag
     */
    DataSource.prototype.hasData = function () {
        return Boolean(this.data && this.data.length > 0);
    };

    /**
     * Clear array of data in this data source
     */
    DataSource.prototype.clearData = function () {
        if (this.data) {
            this.data.length = 0;
        }
    };

    /**
     * Set filters to this data source
     * Data source works with result of b-simple-search's getConfig() method
     *
     * @param {Array} filters Array of filters
     */
    DataSource.prototype.setFilters = function (filters) {
        this.actualCache = false;
        this.filters = filters;
    };

    /**
     * Returns chart type
     *
     * @return {String} Chart type
     */
    DataSource.prototype.getType = function () {
        if (this.groupBy.indexOf(this.calendarName) > -1) {
            return 'stock';
        }

        return undefined;
    };

    /**
     * Generates series for chart
     *
     * @return {Array} Series
     */
    DataSource.prototype.getSeries = function () {
        var _this = this;
        var metric = this.calculateMetric();
        var metricName = this.metricName;
        var calendarName = this.calendarName;
        var series = [];
        var data = [];

        if (this.groupBy.indexOf(this.calendarName) > -1) {
            var index = {};
            metric.forEach(function (fact) {
                var dims = Object.keys(fact).filter(function (key) {
                    return key !== _this.metricName && key !== _this.calendarName;
                });

                var name = _this._genIndexKey(dims, fact);
                if (!index[name]) {
                    index[name] = {
                        name: _this._applyNames(name),
                        yAxis: 0,
                        data: []
                    };
                }

                index[name].data.push({
                    x: Date.parse(fact[calendarName]),
                    y: fact[metricName]
                });
            });

            var keys = Object.keys(index);
            keys.forEach(function (key) {
                series.push(index[key]);
            });

        } else {
            metric.forEach(function (fact, i) {
                data.push({
                    x: i,
                    y: fact[metricName]
                });
            });

            series.push({
                name: this._applyNames(this.metricName),
                type: 'column',
                data: data
            });
        }
        return series.length < this.maxSeriesCount ? series : null;
    };

    /**
     * Generates categories for chart
     *
     * @return {Array} Categories
     */
    DataSource.prototype.getCategories = function () {
        var _this = this;
        var metric = this.calculateMetric();
        var metricName = this.metricName;
        var cats = [];

        metric.forEach(function (el) {
            var catName = '';
            var keys = Object.keys(el).filter(function (key) {
                return key !== metricName;
            });
            catName = _this._genIndexKey(keys, el);
            cats.push(_this._applyNames(catName));
        });

        return cats.length < this.maxCategoriesCount || this.getType() === 'stock' ? cats : null;
    };

    /**
     * Geterates dates map for b-chart (it helps b-chart to make a tooltip)
     *
     * @return {Object} Dates map
     */
    DataSource.prototype.getDatesMap = function () {
        var _this = this;
        var metric = this.calculateMetric();
        var map = {};
        var date;

        metric.forEach(function (fact) {
            date = fact[_this.calendarName];
            map[Date.parse(date)] = date;
        });

        return map;
    };

    BEM.DOM.decl('b-olap', {
        onSetMod: {
            js: function () {
                this.metadata = JSON.parse(this.params.metadata);
                this.chart = this.elem('chart');
                this.table = this.elem('table');
                this.exportForm = this.findBlockInside('export', 'b-export');
            }
        },

        _createDataSource: function () {
            var metadata = this.metadata;
            this.ds = new DataSource(undefined, {
                calendar: metadata.calendar,
                dimensions: metadata.dimensions,
                metric: metadata.metric,
                dimNames: metadata.dim_names
            });
        },

        _buildTable: function () {
            var _this = this;
            var data = this.ds.calculateMetricForHTML();
            if (data.length > 0) {
                var keys = Object.keys(data[0]);
                var headers = keys.map(function (elem) {
                    return {
                        elem: 'header-cell',
                        content: _this.ds.getDimNamesById(elem)
                    };
                });

                var header = {
                    elem: 'header',
                    content: headers
                };

                var rows = data.map(function (row) {
                    var cols = keys.map(function (key) {
                        return {
                            elem: 'cell',
                            content: row[key]
                        };
                    });

                    return {
                        elem: 'row',
                        content: cols
                    };
                });

                var table = {
                    block: 'b-table',
                    js: {
                        rows: rows,
                        rowStep: 10
                    },
                    mods: {theme: 'gray', header: 'float', 'row-limit': 'yes'},
                    content: [
                        header,
                        {
                            elem: 'more-button-row'
                        }
                    ]
                };

                return BEMHTML.apply(table);
            } else {
                return BEMHTML.apply({
                    block: 'b-olap',
                    elem: 'error',
                    content: BEM.I18N('b-olap', 'No data for table')
                });
            }
        },

        _buildChart: function () {
            var data = this.ds.calculateMetric();
            if (data.length === 0) {
                return BEMHTML.apply({
                    block: 'b-olap',
                    elem: 'error',
                    content: BEM.I18N('b-olap', 'No data for chart')
                });
            }

            var series = this.ds.getSeries();
            var categories = this.ds.getCategories();

            if (series === null || categories === null) {
                return BEMHTML.apply({
                    block: 'b-olap',
                    elem: 'error',
                    content: BEM.I18N('b-olap', 'Too much data to display chart')
                });
            } else {
                return BEMHTML.apply({
                    block: 'b-chart',
                    js: {
                        series: series,
                        categories: categories,
                        yAxis: [{
                            axisId: 0,
                            type: this.ds.metricType,
                            label: BEM.I18N('b-olap', 'Pcs.')
                        }],
                        legend: {
                            enabled: true,
                            align: 'top',
                            verticalAlign: 'top'
                        },
                        datesMap: this.ds.getDatesMap(),
                        interval: 'day'
                    },
                    mods: {
                        type: this.ds.getType()
                    }
                });
            }
        },

        _buildExport: function () {
            var _this = this;
            var data = this.ds.calculateMetricForExcel();
            if (data.length > 0) {
                var fields = Object.keys(data[0]);
                fields = fields.map(function (field) {
                    return {
                        id: field,
                        title: _this.ds.getDimNamesById(field)
                    };
                });
                this.exportForm.setProp('fields', fields);
                this.exportForm.setProp('data', data);
                this.exportForm.setProp('filename', 'DSP statistics.xls');
                this.exportForm.delMod('disabled');
            } else {
                this.exportForm.setMod('disabled', 'yes');
            }
        },

        setData: function (data) {
            if (!this.ds) {
                this._createDataSource();
            }
            this.ds.setData(data);
        },

        hasData: function () {
            return this.ds && this.ds.hasData();
        },

        clearData: function () {
            if (this.ds) {
                this.ds.clearData();
            }
        },

        setFilters: function (filters) {
            if (!this.ds) {
                this._createDataSource();
            }
            this.ds.setFilters(filters);
        },

        setGroupBy: function (groupBy) {
            if (!this.ds) {
                this._createDataSource();
            }
            this.ds.setGroupBy(groupBy);
        },

        buildResults: function () {
            if (this.table.length > 0) {
                BEM.DOM.update(this.table, this._buildTable());
            }

            if (this.chart.length > 0) {
                BEM.DOM.update(this.chart, this._buildChart());
            }

            if (this.exportForm) {
                this._buildExport();
            }
        }
    }, {});
})(BEM);
