(function() {

// abstract visual block
function Block(qb, options) {
    this.qb = qb;
    this.paper = qb.paper;
    this.options = $.extend({}, this.options, options);
    this.events = {};
}

Block.prototype = {
    path: null,
    parent: null,
    width: 150,
    height: 30,
    drawn: false,
    x: 0,
    y: 0,
    html: {
        wrapper: '<div class="b-query-builder__paper-block"></div>',
        x: 30,
        y: 5
    },
    options: {
        dragable: true,
        removable: true
    },
    iconPaths: {
        remove: 'M24.778,21.419 19.276,15.917 24.777,10.415 21.949,7.585 16.447,13.087 10.945,7.585 8.117,10.415 13.618,15.917 8.116,21.419 10.946,24.248 16.447,18.746 21.948,24.248z'
    },
    //private
    getPath: function(w, h) {
        var pathArr = this.path(w, h);
            path = '';
        $.each(pathArr, function() {
            path += this[0];
            if( this.length > 1 ) {
                path += this.slice(1).join(',');
            }
        });
        return path;
    },
    startDrag: function(x, y, e) {
        this.dx = this.dy = 0;
    },
    moveDrag: function(dx, dy, x, y, e) {
        this.move(dx - this.dx, dy - this.dy);
        this.dx = dx;
        this.dy = dy;
    },
    endDrag: function(e) {
        this.move(-this.dx, -this.dy);
    },
    on: function(event, handler, scope) {
        if ( typeof handler != 'function' ) {
            return;
        }
        if ( !this.events[event] ) {
            this.events[event] = [];
        }
        this.events[event].push([handler, scope]);
    },
    fire: function(event) {
        if ( !this.events[event] ) {
            return;
        }

        var handlers = this.events[event],
            args = [].slice.call(arguments, 1),
            that = this;

        $.each(this.events[event], function(){
            this[0].apply(this[1] || that, args);
        });
    },
    //public
    draw: function(x, y, minWidth) {
        if ( this.drawn ) {
            return;
        }
        var p = this.paper,
            w = this.width = Math.max(this.getWidth(), minWidth || 0),
            h = this.height = this.getHeight();
        this.node = p.path(this.getPath(w, h));
        this.textNode = p.text(w/2, 15, this.text || '');
        this.textNode.attr('font-size', 14);
        this.nodeSet = p.set();
        this.nodeSet.push(this.node, this.textNode);
        this.setColor(this.color);
        if ( this.options.dragable ) {
            this.nodeSet.drag(this.moveDrag, this.startDrag, this.endDrag, this, this, this);
        }
        if ( this.options.removable ) {
            this.makeRemovable();
        }
        Block.prototype.move.call(this, x, y, true);
        this.drawn = true;
        this.fire('render', this);
        this.setContent(this.htmlContent);
        return this;
    },
    makeRemovable: function() {
        var remove = this.paper.path(this.iconPaths.remove),
            that = this;
        remove.translate(this.width - 30, 0);
        this.nodeSet.push(remove);
        this.icons.remove = remove;
        remove.click(function() {
            that.remove();
        });
        remove.attr({
            fill: 'white',
            'fill-opacity': 0
        });
    },
    remove: function() {
        if ( this.drawn ) {
            this.nodeSet.remove();
            delete this.nodeSet;
            delete this.node;
            delete this.nodeText;
            this.fire('remove', this);
        }
    },
    setText: function(text) {
        this.text = text;
        if ( this.textNode ) {
            this.textNode.attr('text', text);
        }
        return this;
    },
    setColor: function(color) {
        this.color = color;
        if ( this.node && color ) {
            this.node.attr({
                'fill-opacity': .1,
                'stroke': color,
                'fill': color
            });
        }
    },
    move: function(dx, dy, permanent) {
        //TODO: use transform, coz translate is deprecated
        //this.nodeSet.transform("T" + [dx, dy]);
       this.nodeSet.translate(dx, dy);
       if ( this.content ) {
            this.content.css({
                'left': (dx > 0 ? '+=' : '-=') + Math.abs(dx),
                'top': (dy > 0 ? '+=' : '-=') + Math.abs(dy)
            });
        }
        if ( permanent ) {
            this.x += dx;
            this.y += dy;
        }
        return this;
    },
    getHeight: function() {
        return this.height;
    },
    setHeight: function(h) {
        this.height = h;
        if ( this.drawn ) {
            this.node.attr('path', this.getPath(this.getWidth(), h));
        }
        return this;
    },
    getWidth: function() {
        return this.width;
    },
    setWidth: function(w) {
        this.width = w;
        if ( this.drawn ) {
            this.node.attr('path', this.getPath(w, this.getHeight()));
            this.textNode.attr('x', w/2);
        }
        return this;
    },
    setContent: function(html) {
        this.htmlContent = html;
        if ( !this.drawn || !html ) {
            return this;
        }
        if ( !this.content ) {
            this.content = $(this.html.wrapper).appendTo(this.qb.wrapper);
            this.content.css({
                'left': this.x + this.html.x,
                'top': this.y + this.html.y
            });
        }
        this.content.html(html);
        return this;
    }
}

// simple block for terminal statements
function StatementBlock() {
    Block.apply(this, arguments);
    this.icons = {};
}

StatementBlock.prototype = new Block({});

StatementBlock.prototype.gap = {
    offset: 15,
    top: 7,
    width: 10
};

StatementBlock.prototype.path = function(w, h) {
    var gap = this.gap;
    return [
        ['M', 0, 0],
        ['L', gap.offset, 0,
            gap.offset, gap.top,
            gap.offset + gap.width, gap.top,
            gap.offset + gap.width, 0,
            w, 0,
            w, h,
            gap.offset + gap.width, h,
            gap.offset + gap.width, gap.top + h,
            gap.offset, gap.top + h,
            gap.offset, h,
            0, h
        ],
        ['Z']
    ];
}

StatementBlock.prototype.startDrag = function() {
    Block.prototype.startDrag.apply(this, arguments);
    if ( this.parent ) {
        var parent = this.parent;
        this.place = {
            block: parent,
            index: parent.eject(this)
        };
        this.placeholder = new StatementBlock(this.qb, { dragable: false, removable: false });
        this.placeholder.setColor('#ccc');
        parent.insert(this.place.index, this.placeholder);
        this.fire('startDrag', parent, this.place.index);
    }
}

StatementBlock.prototype.moveDrag = function(dx, dy, x, y) {
    Block.prototype.moveDrag.apply(this, arguments);
    if ( this.place ) {
        var newPlace = this.qb.findPlace(this.x + dx, this.y + this.dy);
        if ( this.place.block !== newPlace.block || this.place.index != newPlace.index ) {
            this.place.block.eject(this.placeholder);
            newPlace.block.insert(newPlace.index, this.placeholder);
            this.place = newPlace;
        }
    }
}

StatementBlock.prototype.endDrag = function() {
    Block.prototype.endDrag.apply(this, arguments);
    var parent = this.place.block;
    parent.insert(this.place.index, this);
    if ( this.placeholder ) {
        this.placeholder.remove();
    }
    this.fire('endDrag', parent, this.place.index);
    delete this.place;
    delete this.placeholder;
}

StatementBlock.prototype.remove = function() {
    Block.prototype.remove.apply(this, arguments);
    if ( this.parent ) {
        var parent = this.parent;
        parent.eject(this);
    }
}

// composition block for non-terminal statements
function BracketBlock() {
    Block.apply(this, arguments);
    this.blocks = [];
}

BracketBlock.prototype = new StatementBlock({});

BracketBlock.prototype.bracket = {
    gap: 10,
    th: 30,
    bh: 18,
    emptyHeight: 15
};

BracketBlock.prototype.path = function(w, h) {
    var gap = this.gap,
        bracket = this.bracket;
    return [
        ['M', 0, 0],
        ['L', gap.offset, 0,
            gap.offset, gap.top,
            gap.offset + gap.width, gap.top,
            gap.offset + gap.width, 0,
            w, 0,
            w, bracket.th,
            bracket.gap + gap.offset + gap.width, bracket.th,
            bracket.gap + gap.offset + gap.width, gap.top + bracket.th,
            bracket.gap + gap.offset, gap.top + bracket.th,
            bracket.gap + gap.offset, bracket.th,
            bracket.gap, bracket.th,
            bracket.gap, h - bracket.bh,
            bracket.gap + gap.offset, h - bracket.bh,
            bracket.gap + gap.offset, gap.top + h - bracket.bh,
            bracket.gap + gap.offset + gap.width, gap.top + h - bracket.bh,
            bracket.gap + gap.offset + gap.width, h - bracket.bh,
            w, h - bracket.bh,
            w, h,
            gap.offset + gap.width, h,
            gap.offset + gap.width, gap.top + h,
            gap.offset, gap.top + h,
            gap.offset, h,
            0, h
        ],
        ['Z']
    ];
}

BracketBlock.prototype.getHeight = function(strict) {
    if ( this.drawn && !strict ) {
        return this.height;
    }
    var height = this.bracket.th + this.bracket.bh;
    if ( this.blocks.length ) {
        $.each(this.blocks, function() {
            height += this.getHeight();
        });
    }
    else {
        height += this.bracket.emptyHeight;
    }
    this.height = height;
    return height;
}

BracketBlock.prototype.setWidth = function(w) {
    Block.prototype.setWidth.apply(this, arguments);
    var cw = w - this.bracket.gap;
    $.each(this.blocks, function() {
        this.setWidth(cw);
    });
    return this;
}

BracketBlock.prototype.getWidth = function(strict) {
    if ( this.drawn && !strict ) {
        return this.width;
    }
    var width = this.width - this.bracket.gap;
    $.each(this.blocks, function() {
        width = Math.max(width, this.getWidth());
    });
    this.width = width + this.bracket.gap;
    return this.width;
}

BracketBlock.prototype.draw = function(x, y, minWidth) {
    Block.prototype.draw.apply(this, arguments);
    var cx = x + this.bracket.gap,
        cy = y + this.bracket.th,
        w = this.getWidth() - this.bracket.gap;
    $.each(this.blocks, function() {
        this.draw(cx, cy, w);
        cy += this.getHeight();
    });
}

BracketBlock.prototype.update = function() {
    if ( this.drawn ) {
        var dy = 0,
            x = this.x + this.bracket.gap,
            cy = this.y + this.bracket.th,
            h = this.getHeight(true),
            w = this.getWidth(),
            cw = w - this.bracket.gap;
        $.each(this.blocks, function() {
            if ( !this.drawn ) {
                this.draw(x, cy, cw);
            }
            else {
                this.setWidth(cw);
                this.move(x - this.x, cy - this.y, true);
            }
            cy += this.getHeight();
        });
        this.setHeight(h);
    }
    if ( this.parent ) {
        this.parent.update();
    }
    return this;
}

BracketBlock.prototype.move = function(dx, dy, permanent) {
    Block.prototype.move.apply(this, arguments);
    $.each(this.blocks, function() {
        this.move(dx, dy, permanent);
    });
};

BracketBlock.prototype.insert = function(ind, block) {
    block.parent = this;
    this.blocks.splice(ind, 0, block);
    this.update();
    return this;
};

BracketBlock.prototype.append = function(block) {
    return this.insert(this.blocks.length, block);
};

BracketBlock.prototype.remove = function() {
    $.each(this.blocks.slice(0), function() {
        this.remove();
    });
    StatementBlock.prototype.remove.apply(this, arguments);
};

BracketBlock.prototype.eject = function(block) {
    var ind = $.inArray(block, this.blocks);
    if ( ind >= 0 ) {
        this.blocks.splice(ind, 1);
    }
    block.parent = null;
    this.update();
    return ind;
}

// root filter block
function RootBlock(paper) {
    Block.apply(this, arguments);
}

RootBlock.prototype = new BracketBlock({});

RootBlock.prototype.path = function(w, h) {
    var gap = this.gap,
        bracket = this.bracket;
    return [
        ['M', 0, 0],
        ['L', gap.offset, 0,
            w, 0,
            w, bracket.th,
            bracket.gap + gap.offset + gap.width, bracket.th,
            bracket.gap + gap.offset + gap.width, gap.top + bracket.th,
            bracket.gap + gap.offset, gap.top + bracket.th,
            bracket.gap + gap.offset, bracket.th,
            bracket.gap, bracket.th,
            bracket.gap, h - bracket.bh,
            bracket.gap + gap.offset, h - bracket.bh,
            bracket.gap + gap.offset, gap.top + h - bracket.bh,
            bracket.gap + gap.offset + gap.width, gap.top + h - bracket.bh,
            bracket.gap + gap.offset + gap.width, h - bracket.bh,
            w, h - bracket.bh,
            w, h,
            gap.offset, h,
            0, h
        ],
        ['Z']
    ];
}

/* Expression */

var editors = {
    input: function() {
        return {
            block: 'b-form-input',
            mods: { size: 's' },
            content: { elem: 'input' }
        };
    },
    combobox: function() {
        return {
            block: 'b-form-select',
            mods: { size: 'm', theme: 'grey' },
            content: [
                {
                    elem: 'select',
                    content: []
                }
            ]
        };
    }
};

var types = {
    'string': {
        getEditor: editors.input
    },
    'number': {
        getEditor: editors.input
    },
    'enum': {
        getEditor: function(values) {
            var res = editors.combobox();
            res.content[0].content = $.map(values, function(){
                return {
                    elem: 'option',
                    content: this
                }
            });
            return res;
        }
    },
    'bool': {
        getEditor: function() {
            var res = editors.combobox();
            res.content[0].content = [
                {
                    elem: 'option',
                    content: 'true'
                },
                {
                    elem: 'option',
                    content: 'false'
                }
            ]
            return res;
        }
    }
};

function Environment(env) {
    this.fields = {};
    this.init(env);
}

Environment.prototype = {
    init: function(env) {
        var that = this;
        $.each(env, function(name) {
            var field = new Field(name, this.type);
            that.fields[name] = field;
        });
    },
    getField: function(name) {
        return this.fields[name] || null;
    }
}

function Expression() {}

Expression.prototype = {
    getType: function() {
        throw new Error("Abstract method call");
    },
    getData: function() {
        throw new Error("Abstract method call");
    }
};

function Field(name, type) {
    this.name = name;
    this.type = type;
}

Field.prototype = new Expression();

Field.prototype.getType = function () {
    return this.type;
};

Field.prototype.getData = function () {
    return this.name;
};

function Value(type, value) {
    this.type = type;
    this.value = value;
}

Value.prototype = new Expression();

Value.prototype.getType = function () {
    return this.type;
};

Value.prototype.getData = function () {
    return this.value;
};

function Operation(op) {
    if ( !operations[op] ) {
        throw new Error('Operation do not exists');
    }
    this.op = op;
    this.attrs = operations[op];
    this.arguments = [];
}

Operation.prototype = new Expression();

var operations = {
    'and': {
        term: false,
        arity: ['>', 0],
        color: "hsl(" + [.5, .7, .3] + ")"
    },
    'or': {
        term: false,
        arity: ['>', 0],
        color: "#d54"
    },
    '=': {
        term: true,
        arity: ['=', 2],
        color: "hsl(" + [.6, .5, .5] + ")"
    },
    'like': {
        term: true,
        arity: ['=', 2],
        color: "hsl(" + [.5, .7, .5] + ")"
    },
    'not': {
        term: false,
        arity: ['=', 1],
        color: "hsl(" + [.3, .1, .3] + ")"
    },
    root: {
        term: false,
        arity: ['>', 0]
    }
};

Operation.prototype.getType = function () {
    return 'bool';
};

Operation.prototype.getData = function () {
    var ops = $.map(this.arguments, function(arg) {
        return arg.getData();
    });
    return {
        op: this.op,
        ops: ops
    }
};

Operation.prototype.isFull = function () {
    var arity = this.attrs.arity;
    if ( arity[0] == '=') {
        return this.arguments.length >= arity[1];
    }
    return false;
}

Operation.prototype.isTerm = function () {
    return this.attrs.term;
}

Operation.prototype.addArgument = function (arg, ind) {
    if ( this.isFull() ) {
        throw new Error('Operation is full');
    }
    var argType = arg.getType();
    $.each(this.arguments, function() {
        if ( this.getType() != argType ) {
            throw Error('Incorrect type');
        }
    });
    this.arguments.splice(( typeof ind == 'undefined' ) ? this.arguments.length : ind, 0, arg);
};

Operation.prototype.remove = function(arg) {
    var ind = $.inArray(arg, this.arguments);
    if ( ind >= 0 ) {
        this.arguments.splice(ind, 1);
    }
    return ind;
}

Operation.parse = function (data, env) {
    var op = new Operation(data.op);
    if ( Operation.isTerm(data) ) {
        var first = data.ops[0],
            field = env.getField(first);
        op.addArgument(field);
        $.each(data.ops.slice(1), function() {
            var value = new Value(field.getType(), this);
            op.addArgument(value);
        });
    }
    else {
        $.each(data.ops, function() {
            op.addArgument(Operation.parse(this, env));
        });
    }
    return op;
};

Operation.isTerm = function (data) {
    if (!operations[data.op]) {
        throw new Error('Operation do not exists');
    }
    var op = operations[data.op];
    return op.term;
};

var test = {
    env: {
        fields: {
            'ID': {
                type: 'number'
            },
            'Name': {
                type: 'string'
            }
        }
    },
    data: {
        op: 'and',
        ops: [
            {
                op: 'or',
                ops: [
                    {
                        op: '=',
                        ops: [ 'ID', '101' ]
                    },
                    {
                        op: '=',
                        ops: [ 'ID', '103' ]
                    }
                ]
            },
            {
                op: 'like',
                ops: ['Name', 'user']
            }
        ]
    }
};

function QueryBuilder(wrapper, w, h, env) {
    this.wrapper = wrapper;
    this.paper = Raphael(wrapper.get(0), w, h);
    this.env = new Environment(env.fields);
    this.init();
}

QueryBuilder.prototype = {
    init: function() {
        var that = this;

        this.op = new Operation('root');
        this.op.block = this.root = new RootBlock(this, { dragable: false, removable: false });
        this.root.setText('New Filter');
        this.root.draw(150, 10, 300);
        this.root.expr = this.op;

        //draw collection
        var dy = 10;

        function iter(k) {
            if ( this.term || k == 'root' ) {
                return;
            }
            var st = that.paper.set();
            st.push(that.paper.rect(10, dy, 130, 30, 5).attr('fill', '#eee'));
            st.push(that.paper.text(75, dy + 15, k).attr('font-size', '14'));
            dy += 35;

            st.drag(that.envMove, function() {
                that.dragOp = [k, this, st];
                that.envStart.apply(that, arguments);
            }, that.envEnd, that, that, that);
        }

        $.each(operations, iter);

        dy += 10;

        $.each(this.env.fields, iter);
    },
    envStart: function(x, y) {
        this.x = x - this.wrapper.offset().left;
        this.y = y - this.wrapper.offset().top;
        this.place = this.findPlace(this.x, this.y);
        this.placeholder = new StatementBlock(this, { dragable: false, removable: false });
        this.placeholder.setColor('#ccc');
        this.place.block.insert(this.place.index, this.placeholder);

        this.clone = this.dragOp[2].clone();
        this.clone.dx = 0;
        this.clone.dy = 0;
    },
    envMove: function(dx, dy, x, y) {
        var newPlace = this.findPlace(this.x + dx, this.y + dy);
        this.place = this.place || newPlace;
        if ( this.place.block !== newPlace.block || this.place.index != newPlace.index ) {
            this.place.block.eject(this.placeholder);
            newPlace.block.insert(newPlace.index, this.placeholder);
            this.place = newPlace;
        }

        this.clone.translate(dx - this.clone.dx, dy - this.clone.dy);
        this.clone.dx = dx;
        this.clone.dy = dy;
    },
    envEnd: function() {
        var parent = this.place.block;
        var expr = new Operation(this.dragOp[0]);
        if ( this.placeholder ) {
            this.placeholder.remove();
        }
        this.insert(this.place.index, parent.expr, expr);
        this.clone.remove();
        delete this.place;
        delete this.placeholder;
    },
    insert: function(index, to, expr, notAdd) {
        var block = null,
            that = this;
        to = to || this.op;
        if ( !(expr instanceof Operation) ) {
            expr = Operation.parse(expr, this.env);
        }
        if ( !notAdd ) {
            to.addArgument(expr, index);
        }
        if ( expr.isTerm() ) {
            block = expr.block = new StatementBlock(this);
            block.setText(expr.arguments[0].getData() + " " + expr.op + " " + expr.arguments[1].getData());
        }
        else {
            block = expr.block = new BracketBlock(this);
            $.each(expr.arguments, function() {
                that.append(expr, this, true);
            });
            block.setText(expr.op);
        }
        block.expr = expr;
        block.setColor(expr.attrs.color);
        if ( typeof index != 'undefined' ) {
            to.block.insert(index, block);
        }
        else {
            to.block.append(block);
        }
        block.on('startDrag', function(p) {
            p.expr.remove(expr);
        });
        block.on('endDrag', function(p, ind) {
            p.expr.addArgument(expr, ind);
        });
    },
    append: function(to, expr, notAdd) {
        this.insert(undefined, to, expr, notAdd);
    },
    getData: function() {
        return this.op.getData().ops;
    },
    findPlace: function(x, y, block) {
        block = block || this.root;
        var that = this,
            box = {
                x: block.x,
                y: block.y,
                w: block.getWidth(),
                h: block.getHeight()
            },
            position = {
                block: block,
                index: 0
            };
        if ( y < box.y + block.bracket.th/2 ) {
            return position;
        }
        if ( y > box.y + box.h - block.bracket.bh/2 ) {
            position.index = block.blocks.length;
            return position;
        }
        else {
            var dy = block.y + block.bracket.th,
                find = false;
            $.each(block.blocks, function(i) {
                if ( this instanceof BracketBlock && !this.expr.isFull() ) {
                    if ( y <= dy + this.bracket.th/2 ) {
                        position.index = i;
                        find = true;
                        return false;
                    }
                    else if ( y <= dy + this.getHeight() - this.bracket.bh/2 ) {
                        position = that.findPlace(x, y, this);
                        find = true;
                        return false;
                    }
                }
                else {
                    if ( y < dy + this.getHeight()/2 ) {
                        position.index = i;
                        find = true;
                        return false;
                    }
                }
                dy += this.getHeight();
            });
            if ( !find ) {
                position.index = block.blocks.length;
            }
        }
        return position;
    }
};

BEM.DOM.decl('b-query-builder', {
    onSetMod: {
        js: function() {
            this.qb = new QueryBuilder(this.elem('paper'), 600, 400, test.env);
            this.qb.append(null, test.data);
        }
    }
});

})();
