/*global alert*/
(function(BEM, $) {
    'use strict';

    var Deferred = $.Deferred;

    function where(prop, value) {
        return function(obj) { return Number(obj[prop]) === Number(value); };
    }

    function whereNot(prop, value) {
        return function(obj) { return Number(obj[prop]) !== Number(value); };
    }

    function getRolesById(roles, roleId) {
        var filteredRoles = roles.filter(where('id', roleId));
        return filteredRoles;
    }

    function normalizeRole(role) {
        return {
            id: role.id,
            name: role.label
        };
    }

    /**
     * Role list model
     * @param {string} userId id of user who's roles to track
     * @param {Array} assignedRoles array of assigned roles { id, name }
     * @param {Array} allRoles array of all possible roles { id, name }
     * @param {RoleDataProvider} dataProvider reference to data access layer
     */
    function RoleList(userId, assignedRoles, allRoles, dataProvider) {
        this.userId = userId;
        this.assignedRoles = assignedRoles;
        this.allRoles = allRoles;
        this.dataProvider = dataProvider;
    }

    /**
     * Checks whether assigned roles list contains role with passed id
     */
    RoleList.prototype.hasRole = function (roleId) {
        return getRolesById(this.assignedRoles, roleId).length > 0;
    };

    /**
     * Requests server to add role, adds role to assigned list on success, resolves result to role list
     * @param {string|number} roleId roleId to assign
     * @return {Promise}
     */
    RoleList.prototype.assignRole = function (roleId) {
        var _this = this,
            result = new Deferred();

        function onSuccess(data) {
            if (!data || data.error) { result.reject(); return; }

            _this.assignedRoles = _this.assignedRoles.concat(getRolesById(_this.allRoles, roleId));
            result.resolve(roleId, _this.assignedRoles);
        }

        function onError() {
            result.reject();
        }

        if (!this.hasRole(roleId)) {
            this.dataProvider.sendAssignRequest(this.userId, roleId).then(onSuccess, onError);
        }

        return result.promise();
    };

    /**
     * Requests server for role revokation, removes role from assigned role list on success
     * @param {string|number} roleId roleId to revoke
     * @return {Promise}
     */
    RoleList.prototype.revokeRole = function (roleId) {
        var _this = this,
            result = new Deferred();

        function onSuccess(data) {
            if (!data || data.error) { result.reject(); return; }

            _this.assignedRoles = _this.assignedRoles.filter(whereNot('id', roleId));
            result.resolve(roleId, _this.assignedRoles);
        }

        function onError() {
            result.reject();
        }

        if (this.hasRole(roleId)) {
            this.dataProvider.sendRevokeRequest(this.userId, roleId).then(onSuccess, onError);
        }

        return result.promise();
    };

    /**
     * Data access layer
     */
    function RoleDataProvider(URL, setAction, revokeAction, assignSign, revokeSign) {
        this.URL = URL;
        this.setAction = setAction;
        this.revokeAction = revokeAction;
        this.assignSign = assignSign;
        this.revokeSign = revokeSign;
    }

    /**
     * @return {Deferred}
     */
    RoleDataProvider.prototype.sendAssignRequest = function (userId, roleId) {
        return $.ajax({
            type: 'POST',
            url: this.URL,
            dataType: 'json',
            data: {
                sign: this.assignSign,
                type: this.setAction,
                id: userId,
                role_id: roleId,
                ajax: true
            }
        });
    };

    /**
     * @return {Deferred}
     */
    RoleDataProvider.prototype.sendRevokeRequest = function (userId, roleId) {
        return $.ajax({
            type: 'POST',
            url: this.URL,
            dataType: 'json',
            data: {
                sign: this.revokeSign,
                type: this.revokeAction,
                id: userId,
                roles_id: roleId,
                ajax: true
            }
        });
    };

    /**
     * Presenter
     */
    BEM.DOM.decl('b-roles-editor', {

        onSetMod: {
            js: function() {
                this._initBlock();
                this._addListeners();
            }
        },

        _initBlock: function() {
            var p = this.params,
                assignedRoles = p.roles[0].slice(0),
                allRoles = p.field[0].values.map(normalizeRole),
                dataProvider = new RoleDataProvider(p.URL, p.setAction, p.revokeAction, p.setSign, p.revokeSign);

            this.roles = new RoleList(p.userId, assignedRoles, allRoles, dataProvider);

            // {p-popupa}
            this._bPopup = this.findBlockInside('b-popupa');

            // {array:pi-form-checkbox}
            this._piFormCheckboxList = this.findBlocksOn('item', 'pi-form-checkbox');

            // {function}
            this._onCheckboxChange = $.proxy(this._onCheckboxChange, this);
        },

        _addListeners: function() {
            var _this = this;

            this.bindTo('pseudo-link-content', 'click', $.proxy(this._onListClick, this));
            this._piFormCheckboxList.forEach(function(block) {
                block.on('change', $.proxy(_this._onCheckboxChange, _this, block));
            });
        },

        _onListClick: function() {
            this._bPopup.show(this.elem('pseudo-link-content'));
        },

        /**
         * @param {pi-form-checkbox} block
         **/
        _onCheckboxChange: function(block) {
            var isChecked = block.val(),
                roleId = block.domElem.data('id'),
                onSetRole = $.proxy(this._onSetRole, this, block),
                onRevokeRole = $.proxy(this._onRevokeRole, this, block),
                onError = $.proxy(this._onError, this, block, !isChecked);

            if (isChecked === this.roles.hasRole(roleId)) {
                return;
            }

            if (isChecked) {
                this.roles.assignRole(roleId).then(onSetRole, onError);
            } else {
                this.roles.revokeRole(roleId).then(onRevokeRole, onError);
            }
        },

        /**
         * @param {i-bem} [block] кликнутый блок
         * @param {String} [blockVal] предыдущее значение блока
         */
        _onError: function(block, blockVal) {
            alert(this.params.errorMsg);
            block.val(blockVal);
        },

        /**
         * On set role success
         *
         * @param {pi-form-checkbox} [block]
         * @param {Number|String} [roleId]
         * @param {Number|Object} [assignedRoles] responce from server, 1 if success
         **/
        _onSetRole: function(block, roleId, assignedRoles) {
            var $el = block.domElem;

            this.setMod($el, 'active', 'yes');
            block.val(true);
            this._refreshPseudoLink(assignedRoles);
        },

        /**
         * On revoke role success
         *
         * @param {pi-form-checkbox} [block]
         * @param {number|string} [roleId]
         * @param {number|object} [assignedRoles] responce from server, 1 if seccess
         **/
        _onRevokeRole: function(block, roleId, assignedRoles) {
            var $el = block.domElem;

            this.delMod($el, 'active');
            block.val(false);
            this._refreshPseudoLink(assignedRoles);
        },

        /**
         * Refresh content of the link (roles list)
         **/
        _refreshPseudoLink: function(assignedRoles) {
            this.elem('pseudo-link-content').html($(BEMHTML.apply({
                block: 'b-roles-editor',
                elem: 'pseudo-link-content',
                roles: assignedRoles
            })).html());
        }
    });
})(BEM, jQuery);
