/**
*
* @author Ilya Rezvov <irezvov@yandex-team.ru>
* @date 2012
* @description Module describes block for building simple filter of entities.
*
*/

(function () {
    'use strict';

    var isArray = Array.isArray;
    var isEmptyObject = $.isEmptyObject;

    function ops() {
        return [
            {
                code: 'IN',
                name: BEM.I18N('b-simple-search', 'in list'),
                type: 'number'
            },
            {
                code: '=',
                name: BEM.I18N('b-simple-search', 'equal'),
                type: 'number'
            },
            {
                code: '>',
                name: BEM.I18N('b-simple-search', 'more'),
                type: 'number'
            },
            {
                code: '<',
                name: BEM.I18N('b-simple-search', 'less'),
                type: 'number'
            },
            {
                code: 'LIKE',
                name: BEM.I18N('b-simple-search', 'contain'),
                type: 'text'
            },
            {
                code: 'NOT LIKE',
                name: BEM.I18N('b-simple-search', 'not contain'),
                type: 'text'
            },
            {
                code: '=',
                name: BEM.I18N('b-simple-search', 'equal'),
                type: 'text'
            },
            {
                code: '<>',
                name: BEM.I18N('b-simple-search', 'not equal'),
                type: 'text'
            },
            {
                code: 'IN',
                name: '',
                type: 'enum'
            },
            {
                code: '=',
                name: BEM.I18N('b-simple-search', 'in list'),
                type: 'dictionary'
            },
            {
                code: '<>',
                name: BEM.I18N('b-simple-search', 'not in list'),
                type: 'dictionary'
            },
            {
                code: '=',
                name: BEM.I18N('b-simple-search', 'equal'),
                type: 'contractnumber'
            },
            {
                code: 'IN',
                name: BEM.I18N('b-simple-search', 'in list'),
                type: 'contractnumber'
            },
            {
                code: '=',
                name: BEM.I18N('b-simple-search', 'equal'),
                type: 'publicid'
            },
            {
                code: 'IN',
                name: BEM.I18N('b-simple-search', 'in list'),
                type: 'publicid'
            },
            {
                code: 'IN',
                name: BEM.I18N('b-simple-search', 'in list'),
                type: 'login'
            },
            {
                code: 'NOT IN',
                name: BEM.I18N('b-simple-search', 'not in list'),
                type: 'login'
            },
            {
                code: 'LIKE',
                name: BEM.I18N('b-simple-search', 'contain'),
                type: 'login'
            },
            {
                code: 'NOT LIKE',
                name: BEM.I18N('b-simple-search', 'not contain'),
                type: 'login'
            },
            {
                code: 'LIKE',
                name: BEM.I18N('b-simple-search', 'contain'),
                type: 'domain_mirror'
            },
            {
                code: 'NOT LIKE',
                name: BEM.I18N('b-simple-search', 'not contain'),
                type: 'domain_mirror'
            },
            {
                code: '=',
                name: BEM.I18N('b-simple-search', 'equal'),
                type: 'domain_mirror'
            },
            {
                code: '<>',
                name: BEM.I18N('b-simple-search', 'not equal'),
                type: 'domain_mirror'
            }
        ];
    }

    /**
     * Проверяет значение на пустоту
     *
     * @param {*} value
     * @return {boolean}
     **/
    function isEmpty(value) {
        if (typeof value === 'object') {
            if (value === null) {
                return true;
            } else if (isArray(value)) {
                return value.length === 0;
            } else {
                return isEmptyObject(value);
            }
        } else if (typeof value === 'string') {
            return value.length === 0;
        }
        return false;
    }

    /**
     * Возвращает объект с блоками b-field-editor по лейблу
     *
     * @param {array} editors Список bem-блоков
     * @return {object} {label: editor}
     **/
    function getEditorsByLabel(editors) {
        /**
         * @param {object} result
         * @param {bem-block} editor
         * @return {object}
         **/
        function addEditor(result, editor) {
            result[editor.params.field.label] = editor;
            return result;
        }

        return editors.reduce(addEditor, {});
    }

    /**
    * @block b-simple-search
    *
    * @param {array} all_fields List of all fields for current entity
    * @param {array} search_fields List of searchable fields for current entity in path format
    */
    BEM.DOM.decl('b-simple-search', {
        onSetMod: {
            js: function () {
                this.all_fields = this.params.fields[0];
                this.search_fields = this.params.search_fields[0];
                // из tt2 ожидается
                // hidden_fields: [ {test: 'test_value'} ]
                this.hidden_fields = this.params.hidden_fields && this.params.hidden_fields[0];

                var conf = $.parseJSON(this.params.config);

                this.editors = [];
                this.buildFields();

                if (conf) {
                    this.load(conf);
                }

                if (this.hasMod('onpage', 'yes')) {
                    this.addSearchButton();
                }
            },

            disabled: function (modName, modVal) {
                this.findBlocksInside('b-field-editor').forEach(function (editor) {
                    editor.setMod(modName, modVal);
                });
            }
        },

        // public
        /**
        * @public
        * Get filter conditions for each field
        *
        * @return {array} ['AND', fields]
        */
        getConfig: function () {
            var _this = this;

            function getValue(vals, e) {
                var value = e.val();
                if (value !== undefined &&
                    value !== null &&
                    value !== '' &&
                    value !== false &&
                    !(isArray(value) && !value.length)) {
                    vals.push(_this.makeFilterFromPath(e.field.path, e.getDescription()));
                }
                return vals;
            }

            var vals = this.editors.reduce(getValue, []);

            return ['AND', vals];
        },

        /**
        * @public
        * Check user input
        *
        * @return {bool}
        */
        hasConfig: function () {
            var conf = this.getConfig();
            return Boolean(conf[1].length);
        },

        /**
        * @public
        * Load conditions from config
        *
        * @param {array|object} c Config
        */
        load: function (c) {
            var _this = this;

            function setParams(p) {
                if (_this.isSearchable(p.path)) {
                    var f = _this.getFieldByPath(p.path);
                    f.op(p.op);
                    f.val(p.value);
                }
            }

            var conditionToPath = this.conditionToPath;
            if (isArray(c)) {
                if (c.length != 2 || c[0] != 'AND') {
                    return;
                }
                c = c[1];
                c.forEach(function (c) {
                    setParams(conditionToPath(c));
                });
            } else if (typeof c === 'object') {
                // после добавления продукта загружается список и в качестве
                // конфига приходит объект
                Object.keys(c).forEach(function (name) {
                    setParams({path: name, op: '=', value: c[name]});
                });
            }
        },

        /**
         * @public
         * Returns values of editors.
         * Like getConfig(), but also saves editor's field name, not only field path
         * Used to save user-entered values after entity change.
         */
        getValues: function () {
            return this.editors.map(function (editor) {
                return {
                    label: editor.params.field.label,
                    op: editor.op(),
                    value: editor.val()
                };
            });
        },

        /**
         * @public
         * Sets passed values to editors.
         * Like load(), but also checks name of editor field, not only field path.
         * Used to save user-entered values after entity change.
         *
         * @param {array} values
         * @param {object} [defaultValues] Дефолтные значения для пустых полей {name:value}
         */
        setValues: function (values, defaultValues) {
            if (!isArray(this.editors) || this.editors.length === 0) {
                return;
            }

            /**
             * @param {string} label
             * @return {function}
             **/
            function getLabelFilter(label) {
                /**
                 * @param {object} value
                 * @return {boolean}
                 **/
                return function (value) {
                    return value.label === label;
                };
            }
            /**
             * Устанавливает значени в поле, возвращает список label полей с пустыми значениями
             *
             * @param {array} values
             * @param {array} emptyFieldsLabels
             * @param {bem-block} editor
             * @return {array}
             **/
            function setValue(values, emptyFieldsLabels, editor) {
                var label = editor.params.field.label;
                var field = values.filter(getLabelFilter(label))[0];
                if (typeof field !== 'undefined') {
                    editor.op(field.op);
                    editor.val(field.value);
                    if (isEmpty(field.value)) {
                        emptyFieldsLabels.push(field.label);
                    }
                } else if (isEmpty(editor.val())) {
                    emptyFieldsLabels.push(label);
                }
                return emptyFieldsLabels;
            }

            var emptyFieldsLabels = this.editors.reduce(setValue.bind(null, values), []);

            if (typeof defaultValues === 'object' && defaultValues !== null) {
                this._setDefaultValues(defaultValues, emptyFieldsLabels);
            }
        },

        /**
        * @public
        * Check field is simple, i.e. it hasn't subfields
        *
        * @param {string} f field name
        * @return {bool}
        */
        isFlat: function (f) {
            return this.search_fields.indexOf(f) !== -1;
        },

        /**
        * @protected
        * Transform filter condition to field path and extract value of nestest field
        *
        * @param {array} c Condition
        * @return {object} {path, value, op}
        */
        conditionToPath: function (c) {
            /* magic begin */
            var p = c[0],
                o = c[1],
                v = c[2];
            while (isArray(v) && o != 'IN' && o != '=' && o != 'NOT IN' && o != '<>') {
                p += '.' + v[0];
                o = v[1];
                v = v[2];
            }
            return {
                path: p,
                value: v,
                op: o
            };
            /* magic end */
        },

        /**
        * @public
        * Get field editor by field path
        *
        * @param {string} p path
        * @return {object} b-field-editor
        */
        getFieldByPath: function (p) {
            var r = null;
            $.each(this.editors, function () {
                if (this.field.path == p) {
                    r = this;
                    return false;
                }
            });
            return r;
        },

        // protected
        /**
        * @protected
        * Build filter description for nested fields
        *
        * @param {string} p path
        * @param {object} fc fieldCondition
        * @return {object} Filter description for field
        */
        makeFilterFromPath: function (p, fc, rootFields) {
            if (p == fc[0]) {
                return fc;
            }

            if (!rootFields) {
                rootFields = this.all_fields;
            }

            var parts = p.split('.'),
                cur = parts.shift();

            return [cur, 'MATCH', this.makeFilterFromPath(parts.join('.'), fc, rootFields[cur].subfields)];

        },

        /**
        * @protected
        * Check path exist in searchable fields list
        *
        * @param {string} p path
        */
        isSearchable: function (p) {
            var fs = isArray(this.search_fields[0]) ? [].concat.apply([], this.search_fields) : this.search_fields,
                find = false;

            $.each(fs, function (i, v) {
                return !(find = (v.name == p));
            });

            return find;
        },

        /**
        * @protected
        * Build search_fields from config
        *
        */
        buildFields: function () {
            if (isArray(this.search_fields[0])) {

                if (this.hasMod('single', 'yes')) {
                    this.search_fields = this.search_fields[0].concat(this.search_fields[1] || []);
                    this.buildFields();
                    return;
                }

                this.domElem.html(BEMHTML.apply([
                    {
                        block: 'b-simple-search',
                        elem: 'fields-col-wrap',
                        content: {
                            elem: 'fields-col',
                            tag: 'table'
                        }
                    },
                    {
                        block: 'b-simple-search',
                        elem: 'fields-col-wrap',
                        content: {
                            elem: 'fields-col',
                            tag: 'table'
                        }
                    },
                    {
                        block: 'b-simple-search',
                        elem: 'clear'
                    }
                ]));

                this.renderFields(this.search_fields[0], this.elem('fields-col').eq(0));
                this.renderFields(this.search_fields[1], this.elem('fields-col').eq(1));

            } else {
                if (this.search_fields.length) {
                    this.renderFields(this.search_fields, this.domElem);
                } else {
                    this.domElem.html(BEMHTML.apply([
                        {
                            block: 'b-simple-search',
                            elem: 'clear'
                        }
                    ]));
                }
            }

            this._renderHiddenFields(this.hidden_fields, this.domElem);
        },

        renderFields: function (fs, to) {
            var _this = this;
            fs.forEach(function (v) {
                var html = $(BEMHTML.apply(_this.makeEditor(_this.getByPath(v.name, v.label, v.hint), v)));
                BEM.DOM.append(to, html);

                var e = _this.findBlockInside(html, 'b-field-editor');
                _this.editors.push(e);
            });
        },

        /**
         * Отрисовывает inputs type hidden в начало элемента to
         * Имеет смысл, если стоит мода onpage: 'yes', при которой содержимое оборачивается формой
         *
         * @param {object} fs {name:value}
         * @param {jQuery} to
         */
        _renderHiddenFields: function (fs, to) {
            if (typeof fs === 'undefined') {
                return;
            }

            var content = [];
            for (var name in fs) {
                if (fs.hasOwnProperty(name)) {
                    content.push({
                        tag: 'input',
                        attrs: {
                            type: 'hidden',
                            autocomplete: 'off',
                            name: name,
                            value: fs[name]
                        }
                    });
                }
            }

            BEM.DOM.prepend(to, BEMHTML.apply(content));
        },

        /**
        * @protected
        * Get field by path splited by '.'
        *
        * @param {string} p Path string, ex. "site.domain'
        * @param {string} l Label of field
        * @return {object} h field declaration
        */
        getByPath: function (p, l, h) {
            var parts = p.split('.'),
                cur = parts.shift(),
                name = cur,
                field = this.all_fields[cur];

            /* jshint boss:true */
            while (cur = parts.shift()) {
                field = field.subfields[cur];
                name = cur;
            }

            return $.extend(true, {name: name, path: p}, field, {label: l, hint: h});
        },

        /**
        * @protected
        * Build field editor with label
        *
        * @param {object} field Field description
        * @param {object} params Параметры поля из this.params.search_fields
        * @return {BEMJSON}
        */
        makeEditor: function (field, params) {
            var editor = {
                block: 'b-field-editor',
                tag: 'tr',
                mods: {type: field.type},
                mix: [{block: 'b-simple-search', elem: 'editor'}],
                js: {
                    field: field,
                    operations: ops(),
                    table: true,
                    label: field.label || field.name,
                    hint: field.hint,
                    search: params.login_filter // параметры для location.search
                }
            };

            return editor;
        },

        addSearchButton: function () {
            var htmlForm = BEMHTML.apply({
                tag: 'form',
                attrs: {action: '', method: 'POST'}
            });

            var htmlButton = BEMHTML.apply({
                block: 'b-form-button',
                mix: [{block: 'b-simple-search', elem: 'search-button'}],
                mods: {theme: 'grey-m', size: 'm'},
                name: 'search_json',
                type: 'submit',
                content: BEM.I18N('b-simple-search', 'search')
            });

            this.domElem.children().wrapAll(htmlForm);
            BEM.DOM.append(this.domElem.find('form'), htmlButton);
            this.bindTo(this.domElem.find('form'), 'submit', this.onSearch);
        },

        onSearch: function () {
            if (this.hasConfig()) {
                this.findBlockInside('search-button', 'b-form-button')
                    .elem('input').val(JSON.stringify(this.getConfig()));
            }
            return true;
        },

        /**
         * Проставляет дефолтные значения для переданных полей
         *
         * @param {object} values Дефолтные значения
         * @param {array} fieldsLabels Лэйблы полей, у который нужно проставить дефолтные значения
         **/
        _setDefaultValues: function (values, fieldsLabels) {
            // список полей может меняться
            var editorsByLabel = getEditorsByLabel(this.editors);

            fieldsLabels.forEach(function (label) {
                var editor = editorsByLabel[label];
                var value = values[editor.params.field.name];
                if (typeof value !== 'undefined') {
                    editor.val(value);
                }
            }, this);
        }
    });
})();
