(function () {
    'use strict';

    var chart = require('utils/chart');
    var datas = require('utils/statistic');
    var lodash = require('lodash');

    /**
     * Возвращает список графиков, полученный перемножением полей с деньгами (и метриками)
     * на группировочные поля со всеми возможными значениями группировки.
     *
     * @param  {Array}  valueableFields     Поля значений (деньги + метрики).
     * @param  {Array}  actualCurrencies    Список валют.
     * @param  {Object} fields              Словарь полей.
     * @return {Array}
     */
    function getGraphs(valueableFields, actualCurrencies, fields) {
        return lodash.flatten(valueableFields.map(function (field) {
            var type = fields[field].type;

            if (type === 'money' && actualCurrencies.length) {
                return actualCurrencies.map(function (curId) {
                    return {
                        currency: curId,
                        field: field
                    };
                });
            } else {
                return {field: field};
            }
        }));
    }

    BEM.DOM.decl({
        block: 'b-stat-chart',
        modName: 'type',
        modVal: 'column'
    }, {
        /**
         * Проверяет, является ли количество данных допустимым для построения графиков.
         *
         * @param  {Object}  data
         * @param  {Array}   data.data
         * @param  {Object}  data.currencies
         * @param  {Object}  data.fields
         * @param  {Array}   data.groupingFields
         * @param  {Boolean} data.hasCurrencies
         * @param  {Boolean} data.hasDate
         * @param  {String}  data.interval
         * @param  {Boolean} data.logarithmic
         * @param  {Boolean} data.singleDate
         * @param  {String}  data.size
         * @param  {String}  data.type
         * @param  {Array}   data.valueableFields
         * @return {Object}
         */
        _checkRestrictions: function (data) {
            var dataLength = data.data[0].length;
            var reportsLength = data.data.length;

            if (dataLength === 0) {
                data.empty = true;

                return data;
            }

            if (dataLength * reportsLength > chart.getMaxColumns()) {
                data.excess = true;

                return data;
            }

            if (data.valueableFields.length * reportsLength > chart.getMaxLines()) {
                data.excess = true;

                return data;
            }

            return data;
        },

        /**
         * Считает массивы yAxis, series и categories для типа column.
         *
         * @param  {Object}  data
         * @param  {Array}   data.data
         * @param  {Object}  data.currencies
         * @param  {Boolean} [data.empty]
         * @param  {Boolean} [data.excess]
         * @param  {Object}  data.fields
         * @param  {Array}   data.groupingFields
         * @param  {Boolean} data.hasCurrencies
         * @param  {Boolean} data.hasDate
         * @param  {String}  data.interval
         * @param  {Boolean} data.logarithmic
         * @param  {Boolean} data.singleDate
         * @param  {String}  data.size
         * @param  {String}  data.type
         * @param  {Array}   data.valueableFields
         * @return {Object}
         */
        _updateData: function (data) {
            if (data.empty || data.excess) {
                return data;
            }

            var reportsNum = data.data.length;
            var reports = lodash.range(reportsNum);

            var diff = reportsNum > 1;
            var period = chart.formatPeriod(data.period);

            var fields = data.fields;
            var groupingFields = data.groupingFields;
            var valueableFields = data.valueableFields;

            data.categories = data.data[0].map(function (dataItem) {
                var title = groupingFields.map(function (field) {
                    return field === 'currency_id' ?
                        data.currencies[dataItem[field]].html :
                        fields[field].verbatim ?
                            dataItem[field] :
                            (fields[field].title + ': ' + dataItem[field]);
                }).join(', ');

                return chart.formatTitle(title, data.size === 'small' ? 18 : 30);
            });

            var actualCurrencies = lodash.chain(data.data[0])
                .pluck('currency_id')
                .uniq()
                .filter(Boolean)
                .value();

            var graphs = getGraphs(valueableFields, actualCurrencies, fields);

            data.yAxis = lodash.flatten(graphs.map(function (graph) {
                return reports.map(function (r) {
                    return chart.formatAxis({
                        currency: graph.currency && data.currencies[graph.currency].html,
                        diff: diff,
                        period: period,
                        report: r,
                        title: fields[graph.field].title,
                        type: fields[graph.field].type
                    });
                });
            }));

            var axis = 0;

            data.series = lodash.flatten(graphs.map(function (graph, index) {
                var type = fields[graph.field];

                return reports.map(function (r) {
                    return chart.formatSeries({
                        data: lodash.pluck(data.data[r], graph.field).map(function (val) {
                            return datas.formatValue(val, type);
                        }),
                        diff: diff,
                        index: index,
                        period: period,
                        report: r,
                        title: fields[graph.field].title,
                        type: data.type,
                        yAxis: axis++
                    });
                });
            }));

            return data;
        }
    });
})();
