(function () {
    'use strict';

    var cdate = require('utils/dates').CDate;
    var chart = require('utils/chart');
    var datas = require('utils/statistic');
    var lodash = require('lodash');

    /**
     * Формирует заголовок для графиков.
     *
     * @param  {String}  field               Значимое поле.
     * @param  {Boolean} omitFieldName       О необходимости опустить информацию о значимом поле.
     * @param  {Boolean} addCurrencyInTitle  О необходимости добавить информацию о валюте в заголовок.
     * @param  {Object}  filter              Текущие значения, используемые для группировки.
     * @param  {Array}   groupingFields      Список полей для добавление в заголовок.
     * @param  {Object}  fields              Словарь полей.
     * @param  {Object}  currencies          Словарь валют.
     * @return {String}
     */
    function buildTitle(field, omitFieldName, addCurrencyInTitle, filter, groupingFields, fields, currencies) {
        var title = [];

        if (groupingFields.length === 0 || !omitFieldName) {
            if (addCurrencyInTitle) {
                title.push(fields[field].title + ', ' + currencies[filter.currency_id].html);
            } else {
                title.push(fields[field].title);
            }
        }

        groupingFields.length && groupingFields.forEach(function (groupingField) {
            if (fields[groupingField].verbatim) {
                title.push(filter[groupingField]);
            } else {
                title.push(fields[groupingField].title + ': ' + filter[groupingField]);
            }
        });

        return chart.replaceHTMLCharts(title.join(' | ').replace('&nbsp;', ' '));
    }

    /**
     * Отсеивает поля с булевым типом.
     *
     * @param  {Array}  list        Список полей.
     * @param  {Object} vocabulary
     * @return {Array}              Отфильтрованный список полей.
     */
    function filterBoolean(list, vocabulary) {
        return list.filter(function (field) {
            return vocabulary[field].type !== 'boolean';
        });
    }

    /**
     * Возвращает отфильтрованные данные (если указан критерий).
     *
     * @param  {Array}  data
     * @param  {Object} filter
     * @return {Array}
     */
    function getFilteredData(data, filter) {
        return lodash.isEmpty(filter) ?
            data :
            lodash.filter(data, filter);
    }

    /**
     * Возвращает список графиков, полученный перемножением полей с деньгами (и метриками)
     * на группировочные поля со всеми возможными значениями группировки.
     * Добавляет предварительную группировку по осям.
     * Группировка идет по полям valueableFields + дополнительно по разным валютам.
     *
     * @param  {Array}  valueableFields  Поля значений (деньги + метрики).
     * @param  {Array}  axisList         Группировки.
     * @param  {Object} fields           Словарь полей.
     * @return {Array}
     */
    function getGraphs(valueableFields, axisList, fields) {
        var index = -1;
        var currency;

        return lodash.flatten(valueableFields.map(function (field) {
            var isMoney = fields[field].type === 'money';
            currency = null;
            index++;

            return axisList.map(function (filter) {
                if (isMoney && currency !== filter.currency_id) {
                    if (currency) {
                        index++;
                    }

                    currency = filter.currency_id;
                }

                return {
                    field: field,
                    filter: filter,
                    yAxis: index
                };
            });
        }));
    }

    /**
     * Возвращает время в миллисекундах для точки на графике.
     *
     * @param  {Number} index       Индекс.
     * @param  {String} date        Исходные данные.
     * @param  {String} interval    "day|week|month|year".
     * @param  {Object} datesMap
     * @param  {Array}  cache
     * @return {Number}
     */
    function getPointTime(index, date, interval, datesMap, cache) {
        if (!cache[index]) {
            cache[index] = getTimestamp(date, interval);
            datesMap[cache[index]] = date;
        }

        return cache[index];
    }

    /**
     * Парсит перловые даты и возвращает таймстэмп.
     * Форматы дат:
     *
     * - по дням - 2013-03-05
     * - по неделям - 2011-51
     * - по месяцам - 2010-07
     * - по годам - 2009
     *
     * @param  {String} date        Например, "2013-51".
     * @param  {String} interval    "day|week|month|year".
     * @return {Number}
     */
    function getTimestamp(date, interval) {
        return cdate(date, interval).time();
    }

    /**
     * Возвращает отформатированные оси для графиков.
     *
     * @param  {String} currency
     * @param  {String} type
     * @param  {String} title
     * @return {Array}
     */
    function getYAxis(currency, type, title) {
        return chart.formatAxis({
            currency: currency,
            title: title,
            type: type
        });
    }

    BEM.DOM.decl('b-stat-chart', {
        onSetMod: {
            js: function () {
                var type = this.getMod('type');

                var data = lodash.chain(this.params)
                    .pick([
                        'data',
                        'currencies',
                        'fields',
                        'groupingFields',
                        'hasCurrencies',
                        'hasDate',
                        'interval',
                        'logarithmic',
                        'period',
                        'singleDate',
                        'singleCurrency',
                        'valueableFields'
                    ])
                    .assign({
                        size: this.getMod('size'),
                        type: type === 'column' ? 'column' : null
                    })
                    .value();

                data = this._updateParams(data);
                data = this._checkRestrictions(data);
                data = this._updateData(data);
                this._renderData(data);
            }
        },

        /**
         * Проверяет, является ли количество данных допустимым для построения графиков.
         *
         * @param  {Object}  data
         * @param  {Array}   data.data
         * @param  {Object}  data.currencies
         * @param  {Object}  data.fields
         * @param  {Array}   data.groupingFields
         * @param  {Boolean} data.hasCurrencies
         * @param  {Boolean} data.hasDate
         * @param  {String}  data.interval
         * @param  {Boolean} data.logarithmic
         * @param  {Boolean} data.singleDate
         * @param  {String}  data.size
         * @param  {String}  data.type
         * @param  {Array}   data.valueableFields
         * @return {Object}
         */
        _checkRestrictions: function (data) {
            var dataLength = data.data[0].length;
            var rawAxisLength = data.axisList && data.axisList.length || 0;
            var reportsLength = data.data.length;

            if (dataLength === 0) {
                data.empty = true;

                return data;
            }

            if (data.valueableFields.length * reportsLength * rawAxisLength > chart.getMaxLines()) {
                data.excess = true;

                return data;
            }

            return data;
        },

        /**
         * Обновляет параметры, которые используются для рассчета highcharts-ориентированных полей.
         *
         * @param  {Object}  data
         * @param  {Array}   data.data
         * @param  {Object}  data.currencies
         * @param  {Object}  data.fields
         * @param  {Array}   data.groupingFields
         * @param  {Boolean} data.hasCurrencies
         * @param  {Boolean} data.hasDate
         * @param  {String}  data.interval
         * @param  {Boolean} data.logarithmic
         * @param  {Boolean} data.singleDate
         * @param  {String}  data.size
         * @param  {String}  data.type
         * @param  {Array}   data.valueableFields
         * @return {Object}
         */
        _updateParams: function (data) {
            // Обновляем заголовки полей.
            lodash.each(data.fields, function (desc) {
                desc.title = desc.title || desc.label || desc.caption;
            });

            data.groupingFields = lodash.difference(filterBoolean(data.groupingFields, data.fields), ['date']);

            // Перенос валют в конец массива, чтобы корректно формировать заголовки на графиках.
            var curIndex = data.groupingFields.indexOf('currency_id');
            if (curIndex > -1) {
                data.groupingFields.splice(curIndex, 1);
                data.groupingFields.push('currency_id');
            }

            data.valueableFields = filterBoolean(data.valueableFields, data.fields);

            data.type !== 'column' &&
                (data.axisList = datas.axisList(data.data[0], data.groupingFields));

            return data;
        },

        /**
         * Считает массивы yAxis и series.
         *
         * @param  {Object}  data
         * @param  {Array}   data.data
         * @param  {Array}   data.axisList
         * @param  {Object}  data.currencies
         * @param  {Boolean} [data.empty]
         * @param  {Boolean} [data.excess]
         * @param  {Object}  data.fields
         * @param  {Array}   data.groupingFields
         * @param  {Boolean} data.hasCurrencies
         * @param  {Boolean} data.hasDate
         * @param  {String}  data.interval
         * @param  {Boolean} data.logarithmic
         * @param  {Boolean} data.singleDate
         * @param  {String}  data.size
         * @param  {String}  data.type
         * @param  {Array}   data.valueableFields
         * @return {Object}
         */
        _updateData: function (data) {
            if (data.empty || data.excess) {
                return data;
            }

            var reportsNum = data.data.length;
            var reports = lodash.range(reportsNum);

            var diff = reportsNum > 1; // Есть ли сравнение
            var period = chart.formatPeriod(data.period);

            var fields = data.fields;
            var groupingFields = lodash.difference(data.groupingFields, ['currency_id']);
            var valueableFields = data.valueableFields;

            var addCurrencyInTitle = !data.singleCurrency && data.hasCurrencies && groupingFields.length === 0;
            var omitFieldName = valueableFields.length === 1;

            var axisList = data.axisList;

            data.datesMap = {};

            var graphs = getGraphs(valueableFields, axisList, fields);
            var timeCache = [];

            data.yAxis = lodash.chain(graphs)
                .uniq('yAxis')
                .map(function (graph) {
                    var field = graph.field;
                    var filter = graph.filter;
                    var fieldType = fields[graph.field].type;
                    var currency = null;

                    if (fieldType === 'money' && filter.currency_id) {
                        currency = data.currencies[filter.currency_id].html;
                    }

                    var title = [fields[field].title];

                    if (fieldType === 'money' && filter.currency_id) {
                        title.push(data.currencies[filter.currency_id].html);
                    }

                    return getYAxis(currency, fieldType, title.join(', ').replace('&nbsp;', ' '));
                })
                .flatten()
                .value();

            data.series = lodash.flatten(graphs.map(function (graph, index) {
                var field = graph.field;
                var filter = graph.filter;
                var type = fields[field].type;

                return reports.map(function (r) {
                    var graphPoints = getFilteredData(data.data[r], filter)
                        .map(function (dataItem, i) {
                            return {
                                x: getPointTime(i, dataItem.date, data.interval, data.datesMap, timeCache),
                                y: datas.formatValue(dataItem[field], type)
                            };
                        });

                    return chart.formatSeries({
                        data: graphPoints,
                        diff: diff,
                        index: index,
                        period: period,
                        report: r,
                        title: buildTitle(
                            field,
                            omitFieldName,
                            addCurrencyInTitle,
                            filter,
                            groupingFields,
                            fields,
                            data.currencies
                        ),
                        yAxis: graph.yAxis
                    });
                });
            }));

            return data;
        },

        /**
         * Рисует график.
         *
         * @param  {Object}  data
         * @param  {Boolean} [data.empty]  Нет данных.
         * @param  {Boolean} [data.excess] Слишком много данных.
         */
        _renderData: function (data) {
            var bemjson;

            if (data.empty) {
                bemjson = {
                    elem: 'error-message',
                    content: BEM.I18N('b-stat-chart', 'no-data-to-display')
                };
            } else if (data.excess) {
                bemjson = {
                    elem: 'error-message',
                    content: BEM.I18N('b-stat-chart', 'too-much-data-to-display')
                };
            } else {
                bemjson = {
                    block: 'b-chart',
                    mods: {
                        size: this.getMod('size', 'small') ? 'small' : '',
                        type: data.type !== 'column' ? 'stock' : 'charts'
                    },
                    js: lodash.pick(data, [
                        'categories',
                        'datesMap',
                        'interval',
                        'logarithmic',
                        'series',
                        'yAxis'
                    ])
                };
            }

            BEM.DOM.update(this.elem('container'), BEMHTML.apply(bemjson));
        }
    });
})();
