(function () {
    'use strict';

    var chart = require('utils/chart');
    var datas = require('utils/statistic');
    var lodash = require('lodash');

    /**
     * Порядок сортировки валют.
     * RUB,USD,EUR
     *
     * @type {Object}
     */
    var sortOrder = {
        RUB: 1,
        USD: 2,
        EUR: 3
    };

    /**
     * Функция для сортировки валют.
     *
     * Пример словаря:
     * {
     *     1: {code: "USD"},
     *     2: {code: "RUB"},
     *     3: {code: "EUR"}
     * }
     *
     * @param  {Object}   dictionary   Словарь валют.
     * @return {Function}
     */
    function compareCurrencies(dictionary) {
        return function (a, b) {
            var numericA = sortOrder[dictionary[a].code] || 4;
            var numericB = sortOrder[dictionary[b].code] || 4;

            if (numericA < numericB) {
                return -1;
            }

            if (numericA > numericB) {
                return 1;
            }

            return 0;
        };
    }

    /**
     * Проверяет есть ли достаточно данных, чтобы построить график.
     *
     * Формирует список графиков по исходным данным и группирующим полям (перемножение)
     * и сравнивает полученное количество с количеством исходных отчетов.
     * Если число графиков больше или равно числу отчетов,
     * то считаем что одному графику соответствует один отчет (такое может быть при разных валютах)
     * и в этом случае данных недостаточно для полноценного графика.
     * Решает проблему одной точки у нескольких графиков.
     *
     * Тикет PI-3742.
     *
     * @param  {Object[]} data   Сырые данные.
     * @param  {String[]} fields Список группирующих полей.
     * @return {Boolean}
     */
    function isEnoughData(data, fields) {
        var fieldsWoDate = fields.filter(function (field) {
            return field !== 'date';
        });

        var axisList = datas.axisList(data, fieldsWoDate);

        return axisList.length < data.length;
    }

    BEM.DOM.decl('b-statistic-chartbox', {
        onSetMod: {
            js: function () {
                this._initBlock();
                this._renderCharts();
            }
        },

        _initBlock: function () {
            this.hasDate = this.params.update ?
                this.params.hasDate :
                this._checkDate();
            this.isDateSingle = this.hasDate && this._checkSingleDate();
            this.isTypeColumn = !this.hasDate || this.isDateSingle;

            this.defChartBemjson = {
                block: 'b-stat-chart',
                mods: {},
                js: lodash.pick(this.params, [
                    'currencies',
                    'fields',
                    'groupingFields',
                    'hasCurrencies',
                    'hasDate',
                    'interval',
                    'period',
                    'singleCurrency',
                    'valueableFields'
                ])
            };
            this.chartsBemjson = [];
        },

        _checkDate: function () {
            return this.params.fieldsToRender.dimension_fields.some(function (field) {
                return field.id === 'date';
            });
        },

        _renderCharts: function () {
            var params = this.params;
            var graph;
            var currencies = params.currencies;
            var currencyStr = BEM.I18N('b-statistic-chartbox', 'currency:');
            var title;

            if (this.isTypeColumn && currencies && !this.hasMod('mode', 'single-chart')) {
                // we dont use date as dimension and we have at least one currency
                // so lets draw several charts for each currency

                // lets group data by currency
                var currenciesChartsObj = {};
                lodash.each(currencies, function (currency, id) {
                    currenciesChartsObj[id] = lodash.map(params.data, function (dataItems) {
                        return lodash.filter(dataItems, {currency_id: id});
                    });
                });

                var sortedCurrencies = Object.keys(currenciesChartsObj)
                    .sort(compareCurrencies(currencies));

                sortedCurrencies.forEach(function (curId) {
                    title = currencyStr + ' ' + currencies[curId].html;
                    graph = currenciesChartsObj[curId];

                    if (graph[0].length > 0) {
                        this._addChartBemjson(title, graph, true);
                    }
                }, this);
            } else {
                this._addChartBemjson();
            }

            this._renderChart();
        },

        _renderChart: function () {
            if (this.chartsBemjson.length) {
                this.setMod('rendered', 'yes');
                BEM.DOM.append(this.domElem, BEMHTML.apply(this.chartsBemjson));
            } else {
                this.setMod('rendered', 'no');
            }
        },

        /**
         * Формирует bemjson с графиком и добавляет полученный bemjson в общий массив
         * (графиков может быть несколько - по одному на каждую валюту).
         *
         * Если данных, по которым строится график, недостаточно для полноценного отображения
         * (одна точка к примеру), то bemjson не будет сформирован.
         *
         * @param {string} [title]
         * @param {array}  [chartData]
         * @param {single} [singleCurrency]
         */
        _addChartBemjson: function (title, chartData, singleCurrency) {
            var bemjson = $.extend(true, {}, this.defChartBemjson);
            var params = this.params;
            var length = params.data[0].length;

            this._addMods(bemjson);

            if (!this.isTypeColumn && length > 0 &&
                length <= chart.getMaxLines() && !isEnoughData(params.data[0], params.groupingFields)) {
                return;
            }

            if (!this.hasMod('mode', 'single-chart') && length < 2 || length < 1) {
                return;
            }

            if (this.hasMod('mode', 'islands')) {
                bemjson.mix = [{block: 'b-island'}, {block: 'b-statistic-chartbox', elem: 'island'}];
            }

            bemjson.title = title || bemjson.title;
            bemjson.js.data = chartData || this.params.data;
            bemjson.js.singleDate = this.isDateSingle;

            if (singleCurrency) {
                bemjson.js.singleCurrency = true;
            }

            this.chartsBemjson.push(bemjson);
        },

        _addMods: function (bemjson) {
            var mods = {type: 'slice'};

            if (this.hasMod('size', 'small')) {
                mods.size = 'small';
            }

            if (this.isTypeColumn) {
                mods.type = 'column';
            }

            $.extend(bemjson.mods, mods);
        },

        _checkSingleDate: function () {
            var data = this.params.data[0];
            var dataLength = data.length;
            var isDateSingle = true;

            data.forEach(function (dataItem, i) {
                if (i < (dataLength - 1) && dataItem.date !== data[i + 1].date) {
                    isDateSingle = false;
                }
            });

            return isDateSingle;
        }

    });
})();
