/*jshint sub:true*/
/*globals _,alert*/
(function () {
    'use strict';

    /**
     * Формат вывода даты в тексте.
     *
     * @type {String}
     */
    var DATE_PATTERN = 'DD.MM.YYYY';

    /**
     * Количество строк для рендера.
     *
     * @type {Number}
     */
    var ROWS_PER_ADD = 100;

    /**
     * Ограничение на объем данных.
     *
     * @type {Number}
     */
    var MAX_DATA_COUNT = 10000;

    /**
     * Ограничение на экспорт в эксель.
     *
     * @type {Number}
     */
    var MAX_EXCEL_DATA = 65500;

    var DEFAULT_FILTER_VALUES = {
        is_payable: [2]
    };

    var cdate = require('utils/dates').CDate;
    var dataModule = require('utils/data');
    var datas = require('utils/statistic');
    var lodash = require('lodash');
    var log = require('utils/log');

    /**
    * @private
    * Flat entities tree to flat list with source levels
    *
    * @param {array} tree List of one level of tree
    * @param {number} level Current level of tree flatting
    * @return {array} Floated tree
    */
    function flatTree(tree, level) {
        level = level || 0;
        var l = [];

        $.each(tree, function () {
            this.level = level;
            l.push(this);
            if (this.children) {
                l = l.concat(flatTree(this.children, level + 1));
            }
        });

        return l;
    }

    /**
     * @param {string} id
     * @return {string}
     **/
    function getElemNameFromId(id) {
        if (typeof id === 'string') {
            id = id.replace(/_/g, '-');
        }
        return 'field-' + id;
    }

    // BlockGroup

    function BlockGroup(blocks) {
        this.blocks = blocks;
    }

    function deUniqueId(id) {
        return id.replace(/^id[0-9]+___([A-Za-z0-9_]+)$/, '$1');
    }

    function uniqueId(id) {
        return 'id' + Math.random().toString().substr(2) + '___' + id;
    }

    /**
     * Возвращает значение vat анализируя name полей формы.
     * Понадобилось для корректной обработки случая, когда не
     * пришел vat (without по дефолту), а в названии поля указано _w_nds
     * (т.е. with по факту)
     *
     * @param {array} fields Массив названий полей
     * @return {number} 0 = with, 1 = without, 2 = both
     **/
    function getVatValueFromFieldsNames(fields) {
        var vat = 1;
        if (!fields || !fields.length) {
            return vat;
        }
        var wVat = fields.filter(function (field) {
            return (/_w_nds/).test(field);
        });
        var woVat = fields.filter(function (field) {
            return (/_wo_nds/).test(field);
        });
        if (wVat.length && woVat.length) {
            vat = 2;
        } else if (wVat) {
            vat = 0;
        }
        return vat;
    }

    BlockGroup.prototype = {
        result: function () {
            return this.res;
        },
        size: function () {
            return this.blocks.length;
        },
        getPred: function (args) {
            var pred = args[0],
                a = [].slice.call(args, 1);
            if (typeof pred == 'function') {
                return pred;
            } else {
                return function () {
                    return this[pred].apply(this, a);
                };
            }
        },
        filter: function (pred, ctx) {
            var c = typeof pred == 'string' ? null : ctx;
            pred = this.getPred(arguments);
            var res = [];
            $.each(this.blocks, function () {
                if (pred.call(c || this, this)) {
                    res.push(this);
                }
            });
            return new BlockGroup(res);
        },
        any: function (pred, ctx) {
            var c = typeof pred == 'string' ? null : ctx;
            pred = this.getPred(arguments);
            var res = false;
            $.each(this.blocks, function () {
                return !(res = pred.call(c || this, this));
            });
            return res;
        },
        all: function (pred, ctx) {
            var c = typeof pred == 'string' ? null : ctx;
            pred = this.getPred(arguments);
            var res = true;
            $.each(this.blocks, function () {
                res = pred.call(c || this, this);
            });
            return res;
        },
        m: function (name) {
            var args = [].slice.call(arguments, 1);
            this.res = $.map(this.blocks, function (bl) {
                if (typeof name == 'string') {
                    return bl[name].apply(bl, args);
                } else {
                    return name.apply(bl, args);
                }
            });
            return this;
        },
        on: function () {
            var args = [].slice.call(arguments, 0);

            return this.m.apply(this, ['on'].concat(args));
        },
        change: function () {
            var args = [].slice.call(arguments, 0);

            return this.on.apply(this, ['change'].concat(args));
        }
    };

    BEM.DOM.prototype.findBlocksGroupOn = function (el, bl) {
        return new BlockGroup(this.findBlocksOn(el, bl));
    };

    /**
     * Формирует bemjson элемента option блока b-form-select.
     *
     * @param  {String}   selected  Выбранный период для сравнения.
     * @return {Function}
     */
    function buildOption(selected) {
        return function (option) {
            var bemjson = {
                elem: 'option',
                attrs: {value: option.value},
                content: BEM.I18N('b-statistic', option.caption)
            };

            option.value === selected && (bemjson.attrs.selected = 'yes');

            return bemjson;
        };
    }

    /**
     * Формирует bemjson контрола b-form-select с группировками по дате.
     * Если указан флаг hasWoGrouping, то добавит еще пункт "без группировки" (например, таксопарки).
     *
     * @param  {String}  selectedPeriod     Выбранный период.
     * @param  {Boolean} hasWoGrouping      Добавлять ли пункт "без группировки"?
     * @return {Object}
     */
    function buildSelect(selectedPeriod, hasWoGrouping) {
        var options = [
            {value: 'day', caption: 'by-days'},
            {value: 'week', caption: 'by-weeks'},
            {value: 'month', caption: 'by-months'},
            {value: 'year', caption: 'by-years'}
        ];

        hasWoGrouping && options.unshift({value: 'wo', caption: 'w/o grouping'});

        return {
            block: 'b-form-select',
            mods: {size: 's', theme: 'grey', layout: 'fixed'},
            mix: [{block: 'b-statistic', elem: 'date-group-by'}],
            content: [
                {
                    block: 'b-form-button',
                    type: 'button',
                    mods: {size: 's', theme: 'grey-s', valign: 'middle'},
                    content: ''
                },
                {
                    elem: 'select',
                    attrs: {id: Math.random().toString()},
                    content: options.map(buildOption(selectedPeriod))
                }
            ]
        };
    }

    /**
     * @param  {Object}  fieldsToRender
     * @param  {Boolean} withCurrency
     * @return {Array}
     */
    function fieldsWithDescription(fieldsToRender, withCurrency) {
        var dimension_fields = fieldsToRender.dimension_fields;

        if (withCurrency) {
            dimension_fields = dimension_fields.filter(function (field) {
                return field.id !== 'currency_id';
            });
        }

        var list = dimension_fields
            .concat(withCurrency ? {
                hint: '',
                id: 'currency_id',
                short_title: BEM.I18N('b-statistic', 'currency'),
                title: BEM.I18N('b-statistic', 'currency'),
                type: 'currency'
            } : [])
            .concat(fieldsToRender.entity_fields, fieldsToRender.fields);

        list.forEach(function (field) {
            field.id === 'date' && (field.type = 'date');
            field.title || (field.title = field.label);
        });

        return lodash.uniq(list, 'id');
    }

    /**
     * Возвращает по каким-то признакам тип поля.
     *
     * @param  {Object} field
     * @return {String}
     */
    function getFieldType(field) {
        if (field.verbatim && !field.type) {
            return 'text';
        }

        if (field.type === 'select') {
            return 'date';
        }

        return field.type;
    }

    /**
     * Метод, используемый при сортировке табличных данных.
     * Позволяет получить из сложной структуры данных некоторое значение, по которому можно будет сортировать.
     *
     * @param  {Array} cells    Массив ячеек.
     * @param  {Number} index   Индекс ячейки.
     * @param  {String} type    Тип данных.
     * @return {*}
     */
    function getRowVal(cells, index, type) {
        var val = cells[index];

        switch (type) {
        case 'date':
            val = val.data.date;
            break;
        case 'money':
            val = val.data.value;
            break;
        default:
            val = val.data;
        }

        return val;
    }

    /**
     * Позволяет удалять элемент с полем value в значении "diff".
     *
     * @param  {Boolean}  shouldLock
     * @return {Array}
     */
    function isLockedItem(shouldLock) {
        return function (option) {
            return !shouldLock || option.value !== 'diff';
        };
    }

    /**
     * Отмечает элемент списка в соответствии с переданным значением.
     *
     * @param  {String} selected
     * @return {Array}
     */
    function selectItem(selected) {
        return function (option) {
            option.value === selected && (option.selected = true);

            return option;
        };
    }

    /**
     * Sugar! Небольшая смесь методов map и filter.
     *
     * @param  {Array}    arr
     * @param  {Function} callback
     * @return {Array}
     */
    function partialMap(arr, callback, thisArg) {
        var rest = [];

        arr.forEach(function (item, i) {
            callback.call(thisArg, item, i, rest) === true &&
                rest.push(item);
        });

        return rest;
    }

    BEM.DOM.decl('b-statistic', {
        onSetMod: {
            js: function () {
                this.entitiesTree = this.params.data[0];
                this.entitiesList = flatTree(this.entitiesTree);
                this.entities = [];

                this.loadingPopup = this.findBlockOn('loading-popup', 'b-progress-popup');
                // wait for initialization of nesting blocks
                this.afterCurrentEvent(this.init, this);
            },
            loading: function (modName, modVal) {
                if (modVal == 'yes') {
                    this.submit.setMod('disabled', 'yes');
                    this.loadingPopup.setMod('progress', 'on');
                    this.delMod('table-visible');
                    this.removeTable();
                    this.delMod('too-much-data');
                } else {
                    this._initial = false;
                    this.loadingPopup.delMod('progress');
                    this.submit.delMod('disabled');
                }
            },
            'table-visible': {
                yes: function () {
                    this.setMod('report-controls', 'both');
                },
                '': function () {
                    this.delMod('report-controls');
                }
            }
        },

        init: function () {
            var _this = this;

            // hide for first release
            this.addButton = this.findBlockOn('add-entity', 'b-form-button');
            if (this.addButton) {
                this.addButton.on('click', this.addEntity, this);
            }
            // end

            this.submit = this.findBlockOn('submit', 'b-form-button');
            if (this.submit) {
                this.submit.on('click', this.onSubmitClick, this);
            }

            this.daterange = this.findBlockInside('b-form-daterange');
            if (this.daterange) {
                this.daterange.on('change', this._onDaterangeChange, this);
            }

            // hide for first release
            this.total = this.findBlockOn('total', 'b-form-checkbox');
            if (this.total) {
                this.total.on('change', this.updateFields, this);
            }
            // end

            this._bLinkReducerList = this.findBlocksOn('link-reducer', 'b-link-reducer');
            this._fullLinkList = this.findBlocksOn('full-link', 'b-clipboard-dropdown');

            this.saveReport = this.findBlocksOn('save-report', 'b-form-button');
            if (this.saveReport) {
                $.each(this.saveReport, function () {
                    this.on('click', _this.onSaveReport, _this);
                });
            }
            this.saveReportPopup = this.findBlockOn('save-report-popup', 'b-popupa');
            this.saveReportInput = this.findBlockOn('save-report-input', 'b-form-input');
            this.saveReportButton = this.findBlockOn('save-report-button', 'b-form-button');
            if (this.saveReportButton) {
                this.saveReportButton.on('click', this.onSaveReportSubmit, this);
            }

            this.vatRadio = this.findBlockOn('vat-radio', 'b-form-radio');
            if (this.vatRadio) {
                this.vatRadio.on('change', this.onVatRadioChange, this);
            }

            this.hasRawStat = false;
            this._errorMods = {};

            if (this.params.initial[0]) {
                this.request = this.params.initial[0];
                this._initial = true;
                this.load(this.params.initial[0]);
            } else {
                this.addEntity();
            }
        },

        toggleVat: function (flag) {
            var radio = this.vatRadio;

            if (flag){
                radio.delMod(radio.elem('button'), 'disabled');
            } else {
                radio.setMod(radio.elem('button'), 'disabled', 'yes');
                radio.uncheckAll();
                this.delMod('no-vat');
            }
        },

        onVatRadioChange: function () {
            if (this.hasMod('no-vat', 'yes')) {
                this.delMod('no-vat');
            }
        },

        /**
         * Обработчик клика кнопки "Показать отчет".
         */
        onSubmitClick: function () {
            if (this._showErrors()) {
                return;
            }

            this.daterange.fixDiff(); // Выравнивает периоды.

            var config = this.getConfig();
            this._loadResultFromConf(config);
        },

        _showErrors: function () {
            var _this = this,
                hasErrors = false;

            // Хак для datepicker-a. Прячет открытые попапы.
            lodash.invoke(this.daterange.findBlocksInside('b-form-datepicker'), 'hidePopup');

            if (this.checkIfMoneyFieldSelected() && typeof this.vatVal() === 'undefined') {
                this.setMod('disabled', 'yes');
                this._errorMods['no-vat'] = 'yes';
            } else {
                this.delMod('no-vat');
                delete this._errorMods['no-vat'];
                if (Object.keys(this._errorMods).length === 0) {
                    this.delMod('disabled');
                }
            }

            if (this.hasMod('disabled', 'yes')) {
                hasErrors = true;
                $.each(this._errorMods, function (name, value) {
                    _this.setMod(name, value);
                });
            }
            return hasErrors;
        },

        /**
         * Запрос данных с сервера. Метод POST.
         *
         * @param {Object}       conf
         * @param {Array|String} conf.period
         */
        _loadResultFromConf: function (conf) {
            var dateGroupBy = this.dateGroupBy = this.getDateGroupBy();
            // Не вижу смысла дергать еще раз этот метод, когда результат его работы уже есть в conf.
            var fieldsToRender = this.getFieldsToRender();
            var fieldsOptions = this.getFieldsOptions(fieldsToRender.entity_fields, 'entity');
            this.setMod('loading', 'yes');
            this.delMod('empty');

            var confs;

            if (Array.isArray(conf.period) && conf.period.length === 4) {
                confs = [
                    lodash.assign({}, conf, {period: conf.period.slice(0, 2)}),
                    lodash.assign({}, conf, {period: conf.period.slice(2)})
                ];
            } else {
                confs = [conf];
            }

            dataModule.getStatisticData(confs)
                .then(function (d) {
                    this.delMod('loading');

                    this._onResponse(d, fieldsToRender, fieldsOptions, {
                        dateGroupBy: dateGroupBy,
                        period: conf.period
                    });
                }, this)
                .fail(function (err) {
                    this.delMod('loading');

                    log.error(err);
                }, this);
        },

        /**
         * Обработчик ответа сервера.
         *
         * @param  {Array}  d              Массив с данными, состоящий из двух объектов,
         *                                 которые соответствуют разным периодам.
         * @param  {Object} fieldsToRender
         * @param  {Object} fieldsOptions
         * @param  {Object} opts
         */
        _onResponse: function (d, fieldsToRender, fieldsOptions, opts) {
            var errors = lodash
                .pluck(d, 'error')
                .filter(_.compose(datas.inverse, _.isEmpty));

            if (errors.length) {
                this.removeChart();
                alert(errors.join('\n'));

                return false;
            }

            this._showResult(d, fieldsToRender, fieldsOptions, opts);
        },

        /**
         * Проверка на количество данных.
         *
         * @param  {array}   data
         * @param  {number}  restriction
         * @return {boolean}
         */
        _checkForRestriction: function (data, restriction) {
            return data.some(function (data) {
                return data.data.length > restriction;
            });
        },

        /**
         * Удаляет таблицу и график и показывает сообщение, что данных очень много.
         *
         * @param {array} data
         */
        _showTooMuchData: function (data) {
            this.removeTable();
            this.removeChart();

            this.showTooMuchDataMessage(data[0].data.length);
        },

        /**
         * Дизейблит кнопки формирующие отчет в экселе.
         *
         * @param {boolean} status
         */
        _disableExportButtons: function (status) {
            var value = status ? 'yes' : '';

            this.findBlocksOn(this.elem('export-button'), 'b-form-button').forEach(function (button) {
                button.setMod('disabled', value);
            });
        },

        /**
         * Обрабатывает полученные данные.
         * TODO: Выпилить fieldsOptions.
         *
         * @param  {Array}  d
         * @param  {Object} fieldsToRender
         * @param  {Object} fieldsOptions
         * @param  {Object} opts
         */
        _showResult: function (d, fieldsToRender, fieldsOptions, opts) {
            // Если данных нет, то и показывать нечего :)
            if (!d.some(function (data) {
                return data.data.length > 0;
            })) {
                this.showEmptyResult();

                return;
            }

            if (this._checkForRestriction(d, MAX_EXCEL_DATA)) {
                this._showTooMuchData(d);

                return;
            }

            var actualCurrencies = lodash.chain(d)
                .map(function (data) {
                    return lodash.pluck(data.data, 'currency_id').filter(Boolean);
                })
                .flatten()
                .uniq()
                .value();

            var hasCurrencies = actualCurrencies.length > 0;
            var fields = fieldsWithDescription(fieldsToRender, hasCurrencies);
            var hasDate = lodash.filter(fields, {id: 'date'}).length > 0;

            var valueableFields = datas.valueableFields(fieldsToRender);
            var groupingFields = lodash.difference(datas.fieldsList(fields), valueableFields);

            var params = {
                fields: datas.vocabulary(fields, 'id'),
                groupingFields: groupingFields,
                hasCurrencies: hasCurrencies,
                hasDate: hasDate,
                interval: opts.dateGroupBy,
                period: opts.period,
                singleCurrency: actualCurrencies.length === 1,
                sortOrder: [],
                valueableFields: valueableFields
            };

            // Валютный словарик.
            if (hasCurrencies) {
                params.currencies = datas.vocabulary(d.map(function (report) {
                    return report.currencies;
                }).filter(Boolean)[0], 'id');

                lodash.each(params.currencies, function (obj) {
                    obj.html = datas.formatCurrency(obj.code);
                });
            }

            // totals.
            if (groupingFields.length > 0) {
                var totals = lodash.pluck(d, 'total');
                var sample = lodash.cloneDeep(totals.filter(Boolean)[0]);

                lodash.each(sample, function (group) {
                    lodash.each(group, function (value, field) {
                        group[field] = typeof value === 'number' ? 0 : '0';
                    });
                });

                params.totals = totals.map(function (total) {
                    return total || sample;
                });
            }

            // Хз, что это.
            var hasRawField = params.fields.hasOwnProperty('wo_group_by_date');

            // Предварительная сортировка для фиксации нулей.
            datas.sortByFields(d[0].data, params.groupingFields, params.fields);

            // Фикс данных.
            d = datas.fixData(d, params);

            if (this._checkForRestriction(d, MAX_EXCEL_DATA)) {
                this._showTooMuchData(d);
                this._disableExportButtons(true);

                return;
            } else {
                this._disableExportButtons(false);
            }

            // Сортировочка.
            d.forEach(function (data) {
                datas.sortByFields(data.data, params.groupingFields, params.fields);
            });

            // Обновление заголовка таблицы.
            this._updateTitle(d, params);

            // Обновление полей для экспорта.
            this._updateExports(d, params);

            // Обновляем код в сокращателях ссылок.
            var linkParams;
            if (this._bLinkReducerList) {
                linkParams = this._getLinkParams();
                this._bLinkReducerList.forEach(function (block) {
                    block.setUrlParams(linkParams);
                }, this);
            }

            // Обновляем код в клипбордах.
            var pageLink;
            if (this._fullLinkList) {
                pageLink = this._getLink();
                this._fullLinkList.forEach(function (block) {
                    block.setText(pageLink);
                }, this);
            }

            if (this._checkForRestriction(d, MAX_DATA_COUNT)) {
                this._showTooMuchData(d);

                return;
            }

            // Рисуем графики.
            this.removeChart();

            if (d[0].data.length > 1 &&
                d[0].data.length < 1500 / d.length &&
                params.groupingFields.length > 0 &&
                !hasRawField) {
                this._renderMyChart(d, params);
            }

            // Рисуем таблицу.

            // Сброс сортировки
            this.sortFieldId = null;

            this.shownRowsCount = ROWS_PER_ADD;
            this._convertDataIntoTableForm(d, params);
            this._sortTableData(d, params);
            this._renderMyTable(d, params);

            this.setMod('table-visible', 'yes');

            // Предупреждения.
            var notification = lodash.pluck(d, 'report_notification').filter(Boolean);
            if (notification.length) {
                this.elem('report-notification-text').html(notification[0]);
                this.delMod(this.elem('report-notification-text'), 'hidden');
            } else {
                this.setMod(this.elem('report-notification-text'), 'hidden', 'yes');
            }

            // Заголовок.
            var reportTitle = this.elem('report-title');
            if (reportTitle) {
                var offset = reportTitle.offset();
                $('body,html').animate({scrollTop: offset.top});
            }
        },

        /**
         * Обновляет заголовок таблицы.
         *
         * @param  {Array}        d
         * @param  {Object}       params
         * @param  {Array}        params.currencies
         * @param  {Array}        params.fields
         * @param  {Array}        params.groupingFields
         * @param  {Boolean}      params.hasCurrencies
         * @param  {Boolean}      params.hasDate
         * @param  {String}       params.interval           "day|week|month|year".
         * @param  {Array|String} params.period
         * @param  {Boolean}      params.singleCurrency
         * @param  {Array}        params.valueableFields
         */
        _updateTitle: function (d, params) {
            if (d.length > 1) {
                var period = params.period;

                /* jshint maxlen: false */
                this.elem('report-title-text').text(BEM.I18N('b-statistic', 'Periods comparison %x0 - %x1 and %y0 - %y1')
                    .replace('%x0', cdate(period[0]).format(DATE_PATTERN))
                    .replace('%x1', cdate(period[1]).format(DATE_PATTERN))
                    .replace('%y0', cdate(period[2]).format(DATE_PATTERN))
                    .replace('%y1', cdate(period[3]).format(DATE_PATTERN)));
                /* jshint maxlen: 120 */
            } else {
                this.elem('report-title-text').text(d[0].report_title);
            }
        },

        /**
         * Обновление данных для экспорта в эксель.
         *
         * @param  {Array}        d
         * @param  {Object}       params
         * @param  {Array}        params.currencies
         * @param  {Array}        params.fields
         * @param  {Array}        params.groupingFields
         * @param  {Boolean}      params.hasCurrencies
         * @param  {Boolean}      params.hasDate
         * @param  {String}       params.interval           "day|week|month|year".
         * @param  {Array|String} params.period
         * @param  {Boolean}      params.singleCurrency
         * @param  {Array}        params.valueableFields
         */
        _updateExports: function (d, params, refresh) {
            var reports = lodash.range(d.length);
            var rows = d[0].data;

            if (params.hasDate || params.hasCurrencies || d.length > 1) {
                var modifyDate = params.hasDate && (params.interval !== 'day' && params.interval !== 'month');
                var modify = params.hasCurrencies || modifyDate;

                rows = lodash.chain(rows)
                    .map(function (dateItem, i) {
                        return reports.map(function (r) {
                            var item = modify ?
                                lodash.assign({}, d[r].data[i]) :
                                d[r].data[i];

                            params.hasCurrencies && (item.currency_id = params.currencies[item.currency_id].code);

                            // TODO: Заменить на модуль.
                            modifyDate && (item.date = $(BEMHTML.apply({
                                block: 'b-formatter',
                                mods: {type: 'date'},
                                value: {
                                    date: item.date,
                                    type: params.interval
                                },
                                separator: ' '
                            })).text());

                            return item;
                        });
                    })
                    .flatten()
                    .value();
            }

            this.elem('export').find('input[name="data"]').val(JSON.stringify(rows));

            if (refresh === true) {
                return false;
            }

            var fields = lodash.chain(params.fields)
                .mapValues(function (fieldAttrs) {
                    var title = (fieldAttrs.title || fieldAttrs.label).replace('&nbsp;', ' ');

                    return lodash.assign({}, fieldAttrs, {
                        title: title,
                        label: title
                    });
                })
                .map(function (fieldAttrs, fieldName) {
                    if (fieldName !== 'date') {
                        return lodash.assign({id: fieldName}, fieldAttrs);
                    }

                    var type = 'text';
                    params.interval === 'day' && (type = 'date');
                    params.interval === 'month' && (type = 'date_month');

                    return lodash.assign({id: fieldName}, fieldAttrs, {type: type});
                })
                .value();

            this.elem('export').find('input[name="fields"]')
                .val(JSON.stringify(fields));

            this.elem('export').find('input[name="filename"]').val([
                params.groupingFields.concat(params.valueableFields).join(''),
                ' ',
                cdate().format(DATE_PATTERN),
                '.xls'
            ].join(''));
        },

        /**
         * Преобразует данные к виду, в котором их можно передать
         * БЭМ блоку для отрисовки таблицы.
         *
         * @param  {Array}        d
         * @param  {Object}       params
         * @param  {Array}        params.currencies
         * @param  {Array}        params.fields
         * @param  {Array}        params.groupingFields
         * @param  {Boolean}      params.hasCurrencies
         * @param  {Boolean}      params.hasDate
         * @param  {String}       params.interval           "day|week|month|year".
         * @param  {Array|String} params.period
         * @param  {Boolean}      params.singleCurrency
         * @param  {Array}        params.valueableFields
         */
        _convertDataIntoTableForm: function (d, params) {
            var table = params.table || (params.table = {});
            var colsMap = {};
            var index = 0;

            var fields = params.fields;

            params.singleCurrency && (fields = lodash.omit(params.fields, 'currency_id'));

            table.cols = lodash.map(fields, function (attrs, field) {
                var type = getFieldType(attrs);

                colsMap[field] = {
                    i: index++,
                    numeric: ['money', 'number', 'straight_number']
                        .indexOf(params.fields[field].type) > -1,
                    type: type
                };

                return {
                    colId: field,
                    isSortCol: field === this.sortFieldId,
                    hint: attrs.hint,
                    label: attrs.title || attrs.label || attrs.caption,
                    reversedSort: this.reversedSort,
                    type: type
                };
            }, this);

            table.colsMap = colsMap;

            params.totals && (table.totals = lodash.chain(params.totals)
                .map(function (report, r) {
                    return lodash.map(report, function (val, curId) {
                        var skipped = false;

                        return partialMap(table.cols, function (col, i, result) {
                            if (col.colId in val) {
                                var type = getFieldType(col);
                                var data = val[col.colId];

                                type === 'money' && (data = {
                                    currency: params.currencies[curId].code,
                                    value: data
                                });

                                result.push({
                                    data: data,
                                    type: type,
                                    report: r
                                });
                            } else if (col.colId === 'currency_id') {
                                result.push({
                                    data: BEM.I18N('b-statistic', 'currency-' + params.currencies[curId].code),
                                    type: 'currency',
                                    report: r
                                });
                            } else if (skipped) {
                                result.push({
                                    data: '',
                                    report: r
                                });
                            } else {
                                skipped = datas.inverse(skipped);
                            }
                        });
                    });
                })
                .flatten(true)
                .value());

            table.rows = d[0].data.map(function (row, i) {
                var id = datas.identify();

                return lodash.range(d.length).map(function (r) {
                    var row = d[r].data[i];

                    return {
                        _i: i,
                        _PId: id,
                        report: r,
                        weekendRow: params.hasDate ?
                            cdate(row.date, params.interval).isWeekend() :
                            false,
                        cells: table.cols.map(function (col) {
                            var data;
                            var type = getFieldType(col); // Возможно лишний раз вызывается.

                            switch (type) {
                            case 'money':
                                data = {
                                    currency: params.currencies[row.currency_id].code,
                                    value: row[col.colId]
                                };
                                break;

                            case 'date':
                                data = {
                                    date: row[col.colId],
                                    type: params.interval
                                };
                                break;

                            case 'currency':
                                data = BEM.I18N('b-statistic', 'currency-' + params.currencies[row.currency_id].code);
                                break;

                            default:
                                data = row[col.colId];
                            }

                            return {
                                data: data,
                                type: type
                                // sort_as: field.sort_as,
                            };
                        }, this)
                    };
                }, this);
            }, this);

            return params.table;
        },

        /**
         * Сортирует табличные данные.
         *
         * @param  {Array}        d
         * @param  {Object}       params
         * @param  {Array}        params.currencies
         * @param  {Array}        params.fields
         * @param  {Array}        params.groupingFields
         * @param  {Boolean}      params.hasCurrencies
         * @param  {Boolean}      params.hasDate
         * @param  {String}       params.interval           "day|week|month|year".
         * @param  {Array|String} params.period
         * @param  {Object}       params.table
         * @param  {Array}        params.valueableFields
         */
        _sortTableData: function (d, params) {
            var map = params.table.colsMap;

            if (this.sortFieldId) {
                // Обновляем инфу о сортировке в заголовках.
                lodash.assign(params.table.cols[map[this.sortFieldId].i], {
                    isSortCol: true,
                    reversedSort: this.reversedSort
                });

                // Меняем порядок сортировки.
                var fieldIndex = params.sortOrder.indexOf(this.sortFieldId);
                fieldIndex > -1 && params.sortOrder.splice(fieldIndex, 1);
                params.sortOrder.unshift(this.sortFieldId);
            }

            // Удаляем информацию о предыдущей сортировке.
            params.sortOrder.length > 1 && lodash
                .assign(params.table.cols[map[params.sortOrder[1]].i], {
                    isSortCol: false,
                    reversedSort: this.reversedSort
                });

            var fieldsLength = params.sortOrder.length;
            var reversed = this.reversedSort;

            var field,
                mapped,
                av,
                bv,
                i;

            params.sortOrder.length && params.table.rows.sort(function (a, b) {
                for (i = 0; i < fieldsLength; i++) {
                    field = params.sortOrder[i];
                    mapped = map[field];

                    av = getRowVal(a[0].cells, mapped.i, mapped.type);
                    bv = getRowVal(b[0].cells, mapped.i, mapped.type);

                    if (mapped.numeric) {
                        av = datas.numeric(av);
                        bv = datas.numeric(bv);
                    }

                    if (av < bv) {
                        return reversed ? 1 : -1;
                    }

                    if (av > bv) {
                        return reversed ? -1 : 1;
                    }
                }

                return 0;
            });
        },

        /**
         * Return link for this page with current filters settings
         *
         * @return {string}
         **/
        _getLink: function () {
            // в IE10 нет document.origin
            var url = [document.location.protocol, '//', document.location.host];
            if (this.params.pagePathname) {
                url.push(this.params.pagePathname);
            } else {
                url = [document.location.href.split('?')[0]];
            }
            return encodeURI(url.join('') + '?') + this._getLinkParams();
        },

        _getLinkParams: function () {
            var res = this.request || this.getConfig();
            return encodeURI('request=' + JSON.stringify(res));
        },

        genDataToExport: function (data, currencies) {
            if (data != null && data.length && data[0].currency_id == null) {
                return data;
            }

            var currenciesMap = this.makeCurrenciesMap(currencies),
                dateGroupBy = this.dateGroupBy;

            return data.map(function (dataItem) {
                var resultItem = $.extend({}, dataItem);

                if ('date' in dataItem && dateGroupBy != 'day' && dateGroupBy != 'month') {
                    resultItem.date =
                        $(BEMHTML.apply({
                            block: 'b-formatter',
                            mods: {type: 'date'},
                            value: {
                                date: dataItem.date,
                                type: dateGroupBy
                            },
                            separator: ' '
                        })).text();
                }

                resultItem.currency_id = currenciesMap[dataItem.currency_id];

                return resultItem;
            });
        },

        makeCurrenciesMap: function (currencies) {
            var currenciesMap = {};

            currencies.forEach(function (currency) {
                currenciesMap[currency.id] = currency.code;
            });

            return currenciesMap;
        },

        onSaveReport: function (e) {
            this.saveReportPopup.show(e.target.domElem);
            this.saveReportInput.elem('input').focus();
        },

        onSaveReportSubmit: function () {
            var req = {
                title: this.saveReportInput.val(),
                request: JSON.stringify(this.request)
            };

            this.saveReportPopup.hide();
            return $.post('/statistics/save_report', req, $.proxy(function (d) {
                this.onSaveReportResponse(d, req);
            }, this));
        },

        onSaveReportResponse: function (d, req) {
            var menu = $('.b-menu_partner_yes').bem('b-menu'),
                section = menu.elem('section').eq(1),
                content = BEMHTML.apply({
                    block: 'b-menu',
                    elem: 'layout-vert-cell',
                    tag: 'li',
                    content: {
                        elem: 'item',
                        content: {
                            block: 'b-link',
                            url: '/statistics/?report_id=' + d.id,
                            content: req.title
                        }
                    }
                });

            section.append(content);
        },

        mergeFields: function (fs, newFs) {
            if (!fs.length) {
                return newFs;
            }

            var f = [];
            $.each(fs, function () {
                var cf = this;
                return $.each(newFs, function () {
                    if (this.id == cf.id) {
                        f.push(this);
                        return false;
                    }
                });
            });

            return f;
        },

        unionFields: function (fs, newFs) {
            if (!fs.length) {
                return newFs;
            }

            var f = fs.slice(0);
            $.each(newFs, function () {
                var cf = this,
                    find = false;
                $.each(f, function () {
                    if (this.id == cf.id) {
                        find = true;
                        return false;
                    }
                });
                if (!find) {
                    f.push(cf);
                }
            });

            return f;
        },

        getCommonFields: function () {
            var _this = this,
                es = this.entities.slice(0),
                first = es.shift().entity,
                fs = {
                    dimension: first.dimension_fields,
                    normal: first.fields,
                    entity: first.entity_fields || []
                };

            $.each(es, function () {
                var e = this.entity;

                fs.dimension = _this.mergeFields(fs.dimension, e.dimension_fields);
                fs.normal = _this.mergeFields(fs.normal, e.fields);
                fs.entity = _this.mergeFields(fs.entity, e.entity_fields || []);
            });

            return fs;
        },

        getAllFields: function () {
            var _this = this,
                es = this.entities.slice(0),
                first = es.shift().entity,
                fs = {
                    dimension: first.dimension_fields,
                    normal: first.fields,
                    entity: first.entity_fields || []
                };

            $.each(es, function () {
                var e = this.entity;

                fs.dimension = _this.unionFields(fs.dimension, e.dimension_fields);
                fs.normal = _this.unionFields(fs.normal, e.fields);
                fs.entity = _this.unionFields(fs.entity, e.entity_fields || []);
            });

            /*
            var prio = {
                'number': 1,
                'float': 2,
                'money': 3
            };
            */

            return fs;
        },

        _getCheckedFields: function () {
            var checkedFields = [],
                checkedField,
                i,
                idPart,
                l = this.checkedFields.length,
                wNdsId;

            for (i = 0; i < l; i++) {
                checkedField = this.checkedFields[i];

                idPart = /(.*)(_wo_nds)$/.exec(checkedField);
                if (idPart) {
                    wNdsId = idPart[1] + '_w_nds';
                    if (checkedFields.indexOf(wNdsId) === -1) {
                        checkedFields.push(wNdsId);
                    }
                } else {
                    checkedFields.push(checkedField);
                }
            }

            return checkedFields;
        },

        updateFields: function () {
            var fields = this.addTotal() ? this.getCommonFields() : this.getAllFields();
            var fieldsInConflict = [];
            var _this = this;

            if (this.checkedFields) {
                fieldsInConflict = _this._getConflicts(fields);
            }

            /**
             * getFieldBemjson
             *
             * @param {object} field
             * @param {string} type
             * @return {bemjson}
             */
            function getFieldBemjson(field, type) {
                var checkedFields;
                if (_this.checkedFields) {
                    checkedFields = _this._getCheckedFields();
                }

                var date;
                var entityField;
                var hint;
                var id = uniqueId(field.id);
                var mix = [
                    {block: 'b-statistic', elem: 'entity-field-checkbox', mods: {type: type}},
                    {block: 'b-statistic', elem: getElemNameFromId(field.id)}
                ];
                var mods = {theme: 'grey-m', size: 'm'};
                var title = field.short_title || field.title || field.label;

                if (checkedFields && checkedFields.indexOf(field.id) > -1) {
                    mods.checked = 'yes';
                }

                if (fieldsInConflict.indexOf(field.id) !== -1) {
                    mods.disabled = 'yes';
                }

                if (field.wo_group_by_date) {
                    mix.push({block: 'b-statistic', elem: 'field', mods: {'group-by-date': 'wo'}});
                }

                if (field.short_title) {
                    mods.shorted = 'yes';
                    var idPart = /(.*)(_w_nds|_wo_nds)$/.exec(field.id);
                    if (idPart) {
                        id = idPart[1];
                        if (idPart[2] == '_wo_nds') {
                            return;
                        }
                    }
                }

                if (field.hint) {
                    hint = [
                        title,
                        '&nbsp;',
                        {
                            block: 'b-help-box',
                            content: {
                                elem: 'content-wrapper',
                                mix: [{block: 'b-help-box', elem: 'hint-wrapper'}],
                                content: field.hint
                            }
                        }
                    ];
                } else {
                    hint = title;
                }

                if (field.id === 'date') {
                    date = [
                        '&nbsp;',
                        buildSelect(
                            _this.checkedFieldsExtraParams &&
                            _this.checkedFieldsExtraParams.date &&
                            _this.checkedFieldsExtraParams.date[1],
                            _this.hasRawStat
                        )
                    ];
                } else {
                    date = '';
                }

                entityField = {
                    block: 'b-statistic',
                    elem: 'entity-field',
                    mods: {type: type},
                    content: [
                        {
                            block: 'b-form-checkbox',
                            mods: mods,
                            mix: mix,
                            attrs: {'data-id': field.id},
                            checkboxAttrs: {id: id},
                            content: {
                                elem: 'label',
                                content: hint
                            }
                        },
                        date
                    ]
                };

                return entityField;
            }

            function addFields(type) {
                /**
                 * normalizeProp Возвращает функцию, которая нормализует свойство prop,
                 *               присваивая ему defaultValue, если оно неопределено
                 *
                 * @param {string} prop
                 * @param {*} defaultValue
                 * @return {function}
                 */
                function normalizeProp(prop, defaultValue) {
                    return function (element) {
                        element[prop] = element[prop] || defaultValue;
                        return element;
                    };
                }

                /**
                 * makeIndexByProp Возвращает функцию, которая строит индекс из массива,
                 *                 группируя его элементы по свойству prop
                 *
                 * @param {string} prop
                 * @return {function}
                 */
                function makeIndexByProp(prop) {
                    return function (index, field) {
                        var propValue = field[prop];
                        index[propValue] = index[propValue] || [];
                        index[propValue].push(field);
                        return index;
                    };
                }

                /**
                 * getFieldBemjsonByType Возвращает функцию, которая строит bemjson для поля
                 *                       с типом type
                 *
                 * @param {string} type
                 * @return {function}
                 */
                function getFieldBemjsonByType(type) {
                    return function (field) {
                        return getFieldBemjson(field, type);
                    };
                }

                var fieldIndex = fields[type]
                    .map(normalizeProp('category', 1))
                    .reduce(makeIndexByProp('category'), [])
                    .slice(1);

                if (fieldIndex.length === 1) {
                    return fieldIndex[0].map(getFieldBemjsonByType(type));
                }

                return fieldIndex.map(function (fields) {
                        return {
                            block: 'b-statistic',
                            elem: 'field-group',
                            content: fields.map(getFieldBemjsonByType(type))
                        };
                    });
            }

            BEM.DOM.update(this.elem('fields'), BEMHTML.apply([
                {
                    block: 'b-statistic',
                    elem: 'dimension-fields',
                    dimension: addFields('dimension'),
                    entity: addFields('entity'),
                    normal: addFields('normal')
                }
            ]));

            this.fields = fields;
            this.checkedFields = null;

            this._addFieldsListeners();

            this.buildExampleTable();
        },
        //updateFields

        getGroupFilterValues: function () {
            var simpleSearch = this.findBlockInside('group-filters', 'b-simple-search');
            if (simpleSearch) {
                var config = simpleSearch.getConfig();
                if (config[1].length > 0) {
                    return config;
                } else {
                    return [];
                }
            }
        },

        updateGroupFilters: function () {
            var _this = this;
            var spin = this.findBlockInside('group-filters-spin', 'b-spin');
            var island = this.findElem('group-filters-island');
            var simpleSearch = this.findBlockInside('group-filters', 'b-simple-search');
            var groups = this.getFields('entity').concat(this.getFields('dimension'));
            groups = groups.filter(function (group) {
                return group.filter_values || group.ajax === 1;
            });

            groups.length === 0 ? island.hide() : island.show();

            if (groups.length === 0) {
                this._renderGroupFilters(groups);
                this._debounceFetch && this._debounceFetch(groups);
                spin && spin.delMod('progress');
            } else if (groups.filter(function (group) {
                return group.ajax === 1;
            }).length === 0) {
                this._renderGroupFilters(groups);
            } else {
                simpleSearch && simpleSearch.setMod('disabled', 'yes');
                spin.setMod('progress', 'yes');

                if (!this._debounceFetch) {
                    this._debounceFetch = $.debounce(function (groups) {
                        if (groups.length > 0) {
                            var fieldsToRender = _this.getFieldsToRender();
                            var data = _this.getConfig(fieldsToRender);

                            if (Array.isArray(data.period) && (!data.period[0] || !data.period[1])) {
                                spin.delMod('progress');
                                return;
                            }

                            dataModule.post('/statistics/get_filter_values', data)
                                .then(function (data) {
                                    var spin = _this.findBlockInside('group-filters-spin', 'b-spin');
                                    if (data.error) {
                                        spin.delMod('progress');
                                        return;
                                    }
                                    spin.delMod('progress');
                                    _this._renderGroupFilters(groups, data);
                                })
                                .fail(function (err) {
                                    spin.delMod('progress');

                                    log.error(err);
                                });
                        }
                    }, 1000);
                }

                this._debounceFetch(groups);
            }
        },

        _getSimpleSearchParams: function (groups, values) {
            var result = {
                fields: [],
                search_fields: []
            };

            result.fields.push(groups.reduce(function (result, group) {
                result[group.id] = {
                    type: group.type,
                    values: group.ajax === 1 ? values[group.id] : group.filter_values
                };
                return result;
            }, {}));

            result.search_fields.push([groups.map(function (group) {
                return {
                    label: group.title,
                    name: group.id
                };
            })]);

            return result;
        },

        _renderGroupFilters: function (groups, values) {
            if (groups.length === 0) {
                BEM.DOM.destruct(this.findElem('group-filters'), true);
                return;
            }

            var simpleSearch = this.findBlockInside('group-filters', 'b-simple-search');
            var prevValue = simpleSearch && simpleSearch.getValues();

            BEM.DOM.update(this.findElem('group-filters'), BEMHTML.apply({
                block: 'b-simple-search',
                tag: 'table',
                mods: {single: 'yes'},
                js: this._getSimpleSearchParams(groups, values)
            }));

            simpleSearch = this.findBlockInside('group-filters', 'b-simple-search');

            if (this._groupFiltersConf) {
                simpleSearch.load(this._groupFiltersConf);
                this._groupFiltersConf = null;
            } else {
                prevValue && simpleSearch.setValues(prevValue);
            }
        },

        /**
         * Add change listeners and fire some of them
         **/
        _addFieldsListeners: function () {
            var allFieldsBlocks,
                dateGroupBy,
                blocks;

            blocks = this.findBlocksGroupOn(
                this.findElem(this.elem('fields'), 'entity-field-checkbox', 'type', 'normal'),
                'b-form-checkbox'
           );

            allFieldsBlocks = this.findBlocksGroupOn(
                this.findElem(this.elem('fields'), 'entity-field-checkbox'),
                'b-form-checkbox'
           );

            blocks.change($.proxy(this._onNormalFieldChange, this, blocks), this);
            allFieldsBlocks.change($.proxy(this._onFieldChange, this), this);

            this._onFieldChange();
            this._onNormalFieldChange(blocks);

            // group by date change
            dateGroupBy = this._getBlocksByField('date-group-by');
            dateGroupBy.bFormSelect.on('change', this._onDateGroupByChange, this);
            dateGroupBy.bFormCheckbox.on('change', this._onDateGroupByChange, this);
        },

        _onNormalFieldChange: function (blocks) {
            var checked;

            this.delMod('no-fields');
            delete this._errorMods['no-fields'];

            checked = blocks.filter('hasMod', 'checked', 'yes');
            if (!checked.size()) {
                this.setMod('disabled', 'yes');
                this._errorMods['no-fields'] = 'yes';
                this.toggleVat(false);
            } else {
                if ($.isEmptyObject(this._errorMods)) {
                    this.delMod('disabled');
                }
                this.toggleVat(this.checkIfMoneyFieldSelected());
            }
        },

        _onFieldChange: function () {
            this.findBlocksInside('b-form-checkbox').map(function (checkbox) {
                checkbox.delMod('disabled');
            });

            this.checkedFields = this.getCheckedList();
            this.checkedFields.map(function (field) {
                var block = this.findBlockOn(getElemNameFromId(field), 'b-form-checkbox');
                this._disableConflictsForBlock(block);
            }, this);

            this.updateGroupFilters();
        },

        _onDateGroupByChange: function () {
            var dateGroupBy = this._getBlocksByField('date-group-by'),
                isDisabled = false;

            var val = dateGroupBy.bFormSelect.val();
            var checked = dateGroupBy.bFormCheckbox.isChecked();

            this._togglePeriodLock(val === 'wo' && checked);

            if (val !== 'wo' && checked) {
                isDisabled = true;
            }

            this._toggleDisabled(this.findElem('field', 'group-by-date', 'wo'), isDisabled);
        },

        /**
         * @param {string|jQuery} field
         **/
        _getBlocksByField: function (field) {
            var $el = field;
            if (typeof field === 'string') {
                $el = this.findElem(field);
            }
            return {
                bFormSelect: this.findBlockOn($el, 'b-form-select'),
                bFormCheckbox: this.findBlockOn($el.prev(), 'b-form-checkbox')
            };
        },

        /**
         * @param {jQuery} $field
         * @param {boolean} isDisabled
         **/
        _toggleDisabled: function ($field, isDisabled) {
            var block;

            if (!$field.length) {
                return;
            }

            block = this.findBlockOn($field, 'b-form-checkbox');
            block.toggleMod('disabled', 'yes', '', isDisabled);
            if (isDisabled) {
                block.toggleMod('checked', 'yes', '', false);
            }
        },

        /**
         * Выключает сравнение периодов.
         *
         * @param {Boolean} shouldLock
         */
        _togglePeriodLock: function (shouldLock) {
            if (Boolean(this._lockPeriodDifference) === shouldLock) {
                return;
            }

            this._lockPeriodDifference = shouldLock;

            var current = this.daterange.val().period;
            var options = this._originalOptions || (this._originalOptions = this.daterange.getOptions()
                .map(function (option) {
                    delete option.selected;
                    return option;
                }));

            this.daterange.setOptions(options
                .filter(isLockedItem(shouldLock))
                .map(selectItem(current)));
        },

        getEntity: function (id) {
            var e = null;

            $.each(this.entitiesList, function () {
                if (this.id == id) {
                    e = this;
                    return false;
                }
            });

            return e;
        },

        addEntity: function () {
            var ent = $(BEMHTML.apply([
                    {
                        block: 'b-statistic',
                        elem: 'entity-box',
                        entitiesList: this.entitiesList
                    },
                    {block: 'b-statistic', elem: 'clear'}
                ]));

            BEM.DOM.append(this.elem('entities'), ent);

            var e = {
                list: this.findBlockOn(
                    this.findElem(ent, 'entities-list'), 'b-form-select'
                ),
                filterElem: this.findElem(ent, 'filter')
            };

            e.list.on('change', e, this.onEntityChange, this);

            /* b-simple-search лежит внутри */
            //e.filter = this.findBlockOn(e.filterElem, 'b-simple-search');

            this.entities.push(e);

            return e;
        },

        onEntityChange: function (e) {
            var ent = e.data,
                entity = this.getEntity(ent.list.val()),
                filterBlock = this.findBlockInside(ent.filterElem, 'b-simple-search'),
                filterValues = filterBlock ? filterBlock.getValues() : [],
                filter = this.makeFilters(entity);

            var buttonText = ent.list._button.elem('text');
            buttonText.html($.trim(buttonText.text()));

            function hasFilters(entity) {
                return entity.entity_filter_simple_fields.filter(function (field) {
                    return !Array.isArray(field) || (Array.isArray(field) && field.length !== 0);
                }).length > 0;
            }

            if (!hasFilters(entity)) {
                this.setMod('empty-filters', 'yes');
            } else {
                this.delMod('empty-filters');
                BEM.DOM.update(ent.filterElem, BEMHTML.apply(filter));
                ent.filter = this.findBlockInside(ent.filterElem, 'b-simple-search');

                this.findBlocksInside('b-form-input')
                    .concat(this.findBlocksInside('b-form-select'))
                    .forEach(function (block) {
                        block.on('change', function () {
                            this.updateGroupFilters();
                        }, this);
                    }, this);

                if (ent.filter !== null) {
                    ent.filter.setValues(filterValues, this._initial ? {} : DEFAULT_FILTER_VALUES);
                }
            }

            ent.entity = entity;

            this.hasRawStat = Boolean(entity.has_raw_stat);

            this.updateFields();
            this.trigger('entity-change');
        },

        makeFilters: function (e) {
            if ($.isEmptyObject(e.entity_filter_fields)) {
                return {
                    block: 'b-statistic',
                    elem: 'no-filter',
                    content: [
                        'Нет полей для фильтрации',
                        {
                            elem: 'clear'
                        }
                    ]
                };
            }

            return {
                block: 'b-simple-search',
                tag: 'table',
                mods: {single: 'yes'},
                js: {fields: [e.entity_filter_fields], search_fields: [e.entity_filter_simple_fields]}
            };
        },

        buildExampleTable: function () {
            var content = {
                tag: 'table',
                content: []
            };

            this.elem('example').html(BEMHTML.apply(content));
        },

        showEmptyResult: function () {
            this.removeTable();
            this.table && this.table.empty();

            this.removeChart();

            this.setMod('empty', 'yes');
        },

        removeTable: function () {
            var tableBlock = this.findBlockInside('table', 'b-statistic-table');

            tableBlock &&
                tableBlock.destruct();
        },

        /**
         * Удаляет графики
         */
        removeChart: function () {
            lodash.invoke(this.findBlocksInside(this.findElem('chart'), 'b-statistic-chartbox'), 'destruct');
            this.chart && this.setMod(this.chart, 'visible', 'no');
        },

        checkIfMoneyFieldSelected: function (){
            var moneyFieldSelected = false;

            this.getFields('normal', true).forEach(function (fieldDesc){
                if (fieldDesc.type === 'money') {
                    moneyFieldSelected = true;
                }
            });

            return moneyFieldSelected;
        },

        vatVal: function (value) {
            if (typeof value === 'undefined') {
                var vat = this.vatRadio.val();
                vat = vat.filter(function (elem) {
                    return elem.checked === true;
                });
                if (vat.length > 1) {
                    return 'both';
                } else if (vat.length === 1) {
                    return vat[0].value;
                }
            } else {
                if (value === 'both') {
                    value = ['with', 'without'];
                }

                if (Array.isArray(value)) {
                    this.vatRadio.val(value.map(function (elem){
                        return {
                            value: elem,
                            checked: true
                        };
                    }));
                } else if (typeof value === 'string') {
                    this.vatRadio.val([{value: value, checked: true}]);
                }
            }
        },

        /**
         * @param {string} type
         * @param {boolean} ignoreVAT Используется для получения полей из checkIfMoneyFieldSelected,
         * исключает ситуацию, когда ни один из пунктов НДС не выбран
         */
        getFields: function (type, ignoreVAT) {
            var blocks = this.findBlocksGroupOn(
                    this.findElem(this.elem('fields'), 'entity-field-checkbox', 'type', type),
                    'b-form-checkbox'
               ),
                _this = this,
                checked = blocks.filter('hasMod', 'checked'),
                vat = this.vatVal();

            if (ignoreVAT === true) {
                vat = 'both';
            }

            var addedFields = [],
                fields = checked.m(function () {
                    var id = deUniqueId(this.elem('checkbox').attr('id'));
                    if (this.hasMod('shorted', 'yes')) {
                        if (vat == 'both') {
                            addedFields.push(_this.getField(id + '_w_nds'));
                            id += '_wo_nds';
                        } else {
                            id += ({'with': '_w_nds', without: '_wo_nds'})[vat];
                        }
                    }
                    return _this.getField(id);
                }).result();
            return fields.concat(addedFields);
        },

        getCheckedList: function () {
            return this.getFields('normal')
                .concat(this.getFields('dimension'))
                .concat(this.getFields('entity'))
                .map(function (fieldDesc){ return fieldDesc.id; });
        },

        getField: function (id, fieldType) {
            var r = null,
                fs = this.fields.normal
                    .concat(this.fields.entity)
                    .concat(this.fields.dimension);
            $.each(fieldType ? this.fields[fieldType] : fs, function () {
                if (this.id == id) {
                    r = this;
                    return false;
                }
            });
            return r;
        },

        getEntities: function () {
            var emptyFilters = this.hasMod('empty-filters', 'yes');

            return this.entities.map(function (entity) {
                var e = {id: entity.list.val()};

                if (!emptyFilters && entity.filter && entity.filter.hasConfig()) {
                    e.filter = entity.filter.getConfig();
                }

                return e;
            });
        },

        getDateGroupBy: function () {
            var dateGroupByBlock = this.findBlockOn(
                'date-group-by', 'b-form-select'
           );

            return dateGroupByBlock != null ? dateGroupByBlock.val() : null ;
        },

        mapFields: function (fs) {
            var _this = this;

            return $.map(fs, function (e) {
                var id = deUniqueId(e.id);
                return id === 'date' ?
                    id + '|' + _this.dateGroupBy :
                    id;
            });
        },

        addTotal: function () {
            return false;
        },

        getRequest: function () {
            if (this._showErrors()) {
                return;
            }
            return this.getConfig();
        },

        /**
         * Магия.
         *
         * @param  {Object} fieldsToRender
         * @return {Object}
         */
        getConfig: function (fieldsToRender) {
            this.dateGroupBy = this.getDateGroupBy();

            if (!fieldsToRender) {
                fieldsToRender = this.getFieldsToRender();
            }

            var dateRangeVal = this.daterange.val();

            var res = {
                fields: this.mapFields(fieldsToRender.fields),
                dimension_fields: this.mapFields(fieldsToRender.dimension_fields),
                dimension_filter: this.getGroupFilterValues(),
                entity_fields: this.mapFields(fieldsToRender.entity_fields),
                levels: this.getEntities(),
                total: this.addTotal() ? 1 : 0,
                vat: ['with', 'without', 'both'].indexOf(this.vatVal())
            };

            switch (dateRangeVal.period) {
            case 'diff':
                res.period = [dateRangeVal.from, dateRangeVal.to, dateRangeVal.diffFrom, dateRangeVal.diffTo];
                break;

            case 'other':
                res.period = [dateRangeVal.from, dateRangeVal.to];
                break;

            default:
                res.period = dateRangeVal.period;
            }

            this.request = res;

            return res;
        },

        load: function (conf) {
            var _this = this;
            this._groupFiltersConf = conf.dimension_filter;

            conf.levels.forEach(function (level) {
                var e = this.addEntity();

                e.list.val(level.id);
                if (level.filter) {
                    /* Триггерю событие, чтобы отрисовался b-simple-search
                     * и он попал бы в поле e.filter */
                    e.list.trigger('change');
                    this.afterCurrentEvent(function () {
                        e.filter.load(level.filter);
                    });
                }
            }, this);

            // Выставление периода.
            if (typeof conf.period == 'string') {
                _this.daterange.val({
                    period: conf.period
                });
            } else if (Array.isArray(conf.period)) {
                if (conf.period.length === 2) {
                    _this.daterange.val({
                        period: 'other',
                        from: conf.period[0],
                        to: conf.period[1]
                    });
                } else {
                    _this.daterange.val({
                        period: 'diff',
                        from: conf.period[0],
                        to: conf.period[1],
                        diffFrom: conf.period[2],
                        diffTo: conf.period[3]
                    });
                }
            }

            var fs = conf.fields.concat(conf.entity_fields.concat(conf.dimension_fields));

            this.checkedFieldsExtraParams = {};

            this.checkedFields = $.map(fs, function (v) {
                var parts = v.split('|');

                if (parts.length > 1) {
                    _this.checkedFieldsExtraParams[parts[0]] =
                        parts;
                }

                return parts[0];
            });

            if (typeof conf.vat === 'undefined') {
                conf.vat = getVatValueFromFieldsNames(conf.fields);
            }
            this.vatVal(['with', 'without', 'both'][conf.vat]);

            if (!this.hasMod('filters-only', 'yes')) {
                setTimeout(this._loadResultFromConf.bind(this, conf), 50);
            }
        },

        getFieldsToRender: function () {
            return {
                fields: this.getFields('normal'),
                dimension_fields: this.getFields('dimension'),
                entity_fields: this.getFields('entity')
            };
        },

        /**
         * Рисуем графики.
         *
         * @param  {Array}        d
         * @param  {Object}       params
         * @param  {Array}        params.currencies
         * @param  {Array}        params.fields
         * @param  {Array}        params.groupingFields
         * @param  {Boolean}      params.hasCurrencies
         * @param  {Boolean}      params.hasDate
         * @param  {String}       params.interval           "day|week|month|year".
         * @param  {Array|String} params.period
         * @param  {Boolean}      params.singleCurrency
         * @param  {Array}        params.valueableFields
         */
        _renderMyChart: function (d, params) {
            var bemjson = {
                block: 'b-statistic-chartbox',
                js: lodash.chain(params)
                .pick([
                    'currencies',
                    'fields',
                    'groupingFields',
                    'hasCurrencies',
                    'hasDate',
                    'interval',
                    'period',
                    'singleCurrency',
                    'valueableFields'
                ])
                .assign({
                    data: lodash.pluck(d, 'data'),
                    update: true
                })
                .value()
            };

            this.setMod(this.elem('chart'), 'visible', 'yes');

            BEM.DOM.update(this.elem('chart'), BEMHTML.apply(bemjson));

            var chartbox = this.findBlockInside(this.elem('chart'), 'b-statistic-chartbox');
            // TODO: выпилить.
            if (chartbox.hasMod('rendered', 'yes')) {
                this.setMod(this.chart, 'visible', 'yes');
            } else {
                this.setMod(this.chart, 'visible', 'no');
            }
        },

        /**
         * Формирует в DOM таблицу с данными + добавляет обработчики событий, если нужно.
         *
         * @param  {Array}        d
         * @param  {Object}       params
         * @param  {Array}        params.currencies
         * @param  {jQuery}       [params.ctx]              Узел, относительно которого добавляются строки.
         * @param  {Array}        params.fields
         * @param  {Array}        params.groupingFields
         * @param  {Boolean}      params.hasCurrencies
         * @param  {Boolean}      params.hasDate
         * @param  {String}       params.interval           "day|week|month|year".
         * @param  {Array|String} params.period
         * @param  {Object}       [params.range]            Диапазон рисуемых строк таблицы
         * @param  {Object}       params.table              Сформированные данные для таблицы.
         * @param  {Array}        params.table.cols
         * @param  {Array}        params.table.rows
         * @param  {Array}        params.table.totals
         * @param  {Boolean}      params.singleCurrency
         * @param  {Array}        params.valueableFields
         */
        _renderMyTable: function (d, params) {
            var bemjson;
            var reportsNum = d.length;

            if (params.range) {
                bemjson = lodash.flatten(params.table.rows
                    .slice(params.range.from / reportsNum, params.range.to / reportsNum))
                    .map(function (row) {
                        return {
                            block: 'b-statistic-table',
                            elem: 'row',
                            cells: row.cells,
                            report: row.report,
                            weekendRow: row.weekendRow
                        };
                    });

                BEM.DOM.before(params.ctx, BEMHTML.apply(bemjson));

                this.refreshTableFloatHeader();

                // Прячем showMore, если показывать нечего.
                this.shownRowsCount >= params.table.rows.length * reportsNum &&
                    this.findBlockInside(this.elem('table'), 'b-statistic-table').hideMoreRow();
            } else {
                BEM.DOM.destruct(this.elem('table'), true);

                bemjson = lodash.chain(params.table)
                    .pick([
                        'cols',
                        'totals'
                    ])
                    .assign({
                        block: 'b-statistic-table',
                        js: {
                            showMoreText: this.params.showMoreText
                        },
                        rows: lodash.flatten(params.table.rows
                            .slice(0, this.shownRowsCount / reportsNum)),
                        showMoreButton: params.table.rows.length * reportsNum > this.shownRowsCount
                    })
                    .value();

                BEM.DOM.append(this.elem('table'), BEMHTML.apply(bemjson));

                this.addEventsToMyTable(d, params);
            }
        },

        /**
         * Добавляет обработчики для событий.
         * Хранит данные в замыкании.
         *
         * @param {Object} d                     Исходные данные.
         * @param {Object} params
         * @param {Object} params.fieldsToRender Информация о полях.
         * @param {Object} params.table          Сформированные данные для таблицы.
         * @param {Array}  params.table.cols
         * @param {Array}  params.table.rows
         * @param {Array}  params.table.totals
         */
        addEventsToMyTable: function (d, params) {
            this.findBlockInside('table', 'b-statistic-table')
                .on('sort-column-click', function (e, sortFieldId) {
                    this.reversedSort = this.sortFieldId === sortFieldId && !this.reversedSort;
                    this.sortFieldId = sortFieldId;

                    delete params.ctx;
                    delete params.range;

                    this._sortTableData(d, params);
                    // Перерисовать таблицу.
                    this._renderMyTable(d, params);
                }, this)
                .on('show-more-click', function (e) {
                    // Элемент с кнопкой showMore.
                    // Перед ним добавятся новые строки.
                    var elem = e.target.getMoreRow();
                    var from = this.shownRowsCount;
                    var to = this.shownRowsCount += ROWS_PER_ADD;

                    this._renderMyTable(d, lodash.assign(params, {
                        ctx: elem,
                        range: {
                            from: from,
                            to: to
                        }
                    }));
                }, this);
        },

        getFieldsOptions: function (fields, fieldType) {
            var result = {},
                _this = this;

            $.each(fields, function () {
                result[this.id] = _this.getField(this.id, fieldType);
            });

            return result;
        },

        /**
         * Информирует пользователя о том, что статистика вернула много данных.
         *
         * @param {number} linesCount
         */
        showTooMuchDataMessage: function (linesCount) {
            /*jshint maxlen:false*/
            var msg = BEM.I18N('b-statistic', 'Too much information to display. The number of rows %s. Try to choose a smaller period or change the grouping of data.')
                .replace('%s', linesCount);
            /*jshint maxlen:120*/

            this.elem('too-much-data').text(msg);
            this.setMod('too-much-data', 'yes');
            this.setMod('report-controls', 'top');
        },

        /**
         * Refresh table float header cells (their width)
         **/
        refreshTableFloatHeader: function () {
            var tableBlock = this.findBlockInside('table', 'b-statistic-table');
            tableBlock.refreshFloatHeader();
        },

        /**
         * Обработчик "Выбора периода".
         * Дергает обновление фильтров (полей), которые реализуют какую-то магию.
         */
        _onDaterangeChange: function () {
            var val = this.daterange.val();

            this.updateGroupFilters();

            this.delMod('no-period');
            delete this._errorMods['no-period'];

            if (!val || !this.daterange.valIsSelected()) {
                this._errorMods['no-period'] = 'yes';
                this.setMod('disabled', 'yes');
            } else {
                if ($.isEmptyObject(this._errorMods)) {
                    this.delMod('disabled');
                }
            }
        },

        /**
         * Ищет конфликты и дисейблит/энейблит
         *
         * @param {bem-block} block Поле, для которого ищутся конфликтные поля
         **/
        _disableConflictsForBlock: function (block) {
            if (block === null) {
                return;
            }
            var fieldId = block.domElem.attr('data-id');
            var conflicts = this._getFieldsInConflict(fieldId);
            var isChecked = block.isChecked();

            conflicts.forEach(function (fieldId) {
                this._toggleDisabled(this.findElem(getElemNameFromId(fieldId)), isChecked);
            }, this);
        },

        /**
         * Возвращает конфликты для this.checkedFields
         * В момент вызова this.checkedFields еще не обновлены и
         * нужно проверить есть ли старые поля в новой сущности
         *
         * @param {object} fields Поля текущей entity
         * @return {array}
         **/
        _getConflicts: function (fields) {
            if (!this.checkedFields) {
                return [];
            }

            var result = [];

            var allFields = [];
            allFields = allFields.concat(fields.normal).concat(fields.dimension).concat(fields.entity);
            allFields = allFields.map(function (fieldDesc){
                return fieldDesc && fieldDesc.id;
            });

            this.checkedFields.forEach(function (fieldId) {
                // все отмеченные поля, которые есть в выбранной сущности
                if (allFields.indexOf(fieldId) !== -1) {
                    result = result.concat(this._getFieldsInConflict(fieldId));
                }
            }, this);
            return result;
        },

        /**
         * Ищет конфликты
         * Возвращает список полей в конфликте, исключая поля для которых конфликты искались
         *
         * @param {array} fieldId Поле, для которого ищутся конфликты
         * @return {array}
         **/
        _getFieldsInConflict: function (fieldId) {
            /**
             * Собирает conflict_fields из сущностей (уровней статистики)
             *
             * @param {array} fields
             * @param {object} entity
             **/
            function getConflictFields(fields, entity) {
                fields = fields.concat(entity.entity.conflict_fields);
                return fields;
            }

            /**
             * @param {string} triggerId
             * @param {string} id
             **/
            function isNotTrigger(triggerId, id) {
                return id !== triggerId;
            }

            /**
             * Проверяет есть ли поле в какой-либо группе конфликтующих полей
             * Возвращает поля из конфликтующей группы, кроме переданного поля
             *
             * @param {string} fieldId
             * @param {object} result
             * @param {array} conflictFields Группы конфликтующих полей array:array:string
             * @return {array}
             **/
            function getFieldsInConflictForBlock(fieldId, result, conflictFields) {
                if (conflictFields.indexOf(fieldId) !== -1) {
                    result = result.concat(conflictFields.filter(isNotTrigger.bind(null, fieldId)));
                }
                return result;
            }

            var conflictGroups = this.entities.reduce(getConflictFields, []);
            return conflictGroups.reduce(getFieldsInConflictForBlock.bind(this, fieldId), []);
        }

    });
})();
