/**
 * @param {string:string} this.params.args - {type, name}
 * @param {string} this.params.url
 * @param {object} this.params.editBlockParams
 * @param {object} this.params.args - {type, id}
 **/
(function () {
    'use strict';

    var promises = BEM.blocks['i-promises'];
    var Deferred = promises.Deferred;
    var ajax = promises.ajax;
    var pick = require('lodash').pick;
    var extend = require('lodash').extend;

    function getSign(url) {
        var sign = url.match('sign=([a-zA-Z0-9]+)');
        return (sign && sign[1]) || null;
    }

    /**
     * @param {object} data - responce from server
     **/
    function getMessage(data) {
        var text;

        // id data.error is non empty string
        if (data.error && typeof data.error === 'string') {
            text = data.error;
        } else if (data.error !== undefined || data.responseText !== undefined){
            text = BEM.I18N('b-table-cell-text', 'Changes cannot be saved');
        } else {
            text = BEM.I18N('b-table-cell-text', 'Changes saved');
        }

        return text;
    }

    /**
     * @param {b-form-input|b-form-list|...} editBlocks
     * @param {Array} titles
     **/
    function Element(editBlocks, titles) {
        // {b-form-input|b-form-list|...}
        this._editBlocks = editBlocks;
        // {number|string}
        this._value = this.val();
        // {Array}
        this._titles = titles;

        this._addListeners();
    }

    Element.prototype.val = function (value) {
        if (typeof value === 'undefined') {
            if (Array.isArray(this._editBlocks) && this._editBlocks.length > 1) {
                return this._editBlocks.reduce(function (result, el) {
                    var name = el.name();
                    if (typeof name === 'string') {
                        result[name] = el.val();
                    } else {
                        result = extend(result, el.val());
                    }
                    return result;
                }, {});
            } else {
                return this._editBlocks[0].val();
            }
        } else {
            if (Array.isArray(this._editBlocks) && this._editBlocks.length > 1) {
                this._editBlocks.forEach(function (el) {
                    var name = el.name();
                    if (typeof name === 'string') {
                        el.val(value[name]);
                    } else if (Array.isArray(name)) {
                        el.val(pick(value, name));
                    }
                });
            } else {
                this._editBlocks[0].val(value);
            }
        }
    };

    Element.prototype.refreshValue = function () {
        this.val(this._value);
    };

    Element.prototype.trim = function () {
        this._editBlocks.forEach(function (block) {
            block.trim && block.trim();
        });
        this._value = this.val();
    };

    Element.prototype.afterShow = function () {
        var blocks = this._editBlocks;
        blocks.forEach(function (block) {
            if (block.scrollToSelected) {
                block.scrollToSelected();
            }
        });
    };

    Element.prototype.reset = function () {
        this.val(this._value);
    };

    /**
     * @param {string} url
     * @param {object} data
     **/
    Element.prototype.save = function (url, data) {
        var value = null;

        value = this.val();

        if (typeof value === 'object' && !Array.isArray(value)) {
            $.extend(data, value);
        } else {
            data[this._editBlocks[0].name()] = value;
        }

        var ajaxParams = {
            type: 'POST',
            url: url,
            dataType: 'json',
            data: $.param(data, true)
        };
        return ajax(ajaxParams).then($.proxy(this._success, this), $.proxy(this._error, this));
    };

    Element.prototype.disable = function () {
        var blocks = Array.isArray(this._editBlocks) ? this._editBlocks : [this._editBlocks];
        blocks.forEach(function (block) {
            block.setMod('disabled', 'yes');
        });
    };

    Element.prototype.enable = function () {
        var blocks = Array.isArray(this._editBlocks) ? this._editBlocks : [this._editBlocks];
        blocks.forEach(function (block) {
            block.delMod('disabled');
        });
    };

    /**
     * Return value or selected item text
     *
     * @return {string}
     **/
    Element.prototype.getText = function () {
        var _this = this;

        if (this._editBlocks.length === 1) {
            var block = this._editBlocks[0];
            var text = typeof block.displayVal === 'function' ? block.displayVal() : block.val();
            return typeof text === 'string' ? $.trim(text) : text;
        } else {
            return this._editBlocks.reduce(function (result, block, index) {
                var method = typeof block.displayVal === 'function' ? 'displayVal' : 'val';
                result += (_this._titles[index] + ': ' + $.trim(block[method]()));
                if (index !== _this._editBlocks.length - 1) {
                    result += '; ';
                }
                return result;
            }, '');
        }
    };

    /**
     * Set text value to edit block
     * @param {string} value Value to set
     */
    Element.prototype.setText = function (value) {
        this.val(value);
    };

    Element.prototype._addListeners = function () {
        var blocks = Array.isArray(this._editBlocks) ? this._editBlocks : [this._editBlocks];
        blocks.forEach(function (el) {
            el.on('change', function (e, data) {
                if (data !== undefined && data.initialEvent !== undefined) {
                    data.initialEvent.stopPropagation();
                }
            });
        });
    };

    /**
     * Check errors,
     * refresh value,
     * remove disable mod
     *
     * @param {object} data
     * @return {object} data
     **/
    Element.prototype._success = function (data) {
        if (data.error === undefined) {
            // данные, пришедшие с сервера, могут отличаться от тех, которые содержатся в контролах
            this._editBlocks.forEach(function (el) {
                var name = el.name();
                var val;
                if (typeof name === 'string') {
                    val = data[name];
                    if (typeof val !== 'undefined') {
                        el.val(val);
                    }
                } else if (Array.isArray(name)) {
                    el.val(pick(data, name));
                }
            });
            this._value = this.val();
        } else {
            this._error(data.error);
        }
        return data;
    };

    /**
     * Show error,
     * remove disable mod
     *
     * @param {object} data
     * @return {object}
     **/
    Element.prototype._error = function (data) {
        var blocks = this._editBlocks;
        blocks.forEach(function (block) {
            block.delMod('disabled');
        });
        return data;
    };

    /**
     * Popup
     *
     * @param {b-popupa} bPopup
     * @param {object} params
     * @param {array:b-form-input} params.bformButtonList
     * @param {jQuery} params.$notify
     * @param {string} params.editBlockName
     * @param {object} params.args - {ajax, sign, type, id}
     **/
    function Popup(bPopup, params) {
        // {b-popupa}
        this._bPopup = bPopup;
        // {object}
        this._params = params;
        // {Deferred}
        this._deferred = null;
        // {Element}
        this._element = new Element(bPopup.findBlocksInside(params.editBlockName), params.titles);
        // {}
        this._hideTimeout = null;

        this._addListeners();
    }

    /**
     * Устанавливает значение
     **/
    Popup.prototype.val = function (value) {
        if (typeof value !== 'undefined') {
            this._element.setText(value);
        } else {
            return this._element.val();
        }
    };

    /**
     * Возвращает либо содержание текстового поля
     * либо текст выбранного пункта (не значение)
     **/
    Popup.prototype.getText = function () {
        return this._element.getText();
    };

    Popup.prototype.refreshValue = function () {
        this._element.refreshValue();
    };

    Popup.prototype.destruct = function () {
        if (this._bPopup) {
            this._bPopup.destruct();
        }
    };

    /**
     * @param {jQuery} $elem
     **/
    Popup.prototype.show = function ($elem) {
        if (this._bPopup.isShowed() === true) {
            this._bPopup.hide();
        }
        this._deferred = new Deferred();
        this._bPopup.show($elem);
        setTimeout(this._element.afterShow.bind(this._element), 1);
        return this._deferred;
    };

    Popup.prototype.hide = function () {
        this._bPopup.hide();
    };

    Popup.prototype.disable = function () {
        function disable(block) {
            block.setMod('disabled', 'yes');
        }
        this._element.disable();
        this._params.bFormButtonList.forEach(disable);
    };

    Popup.prototype.enable = function () {
        function enable(block) {
            block.delMod('disabled');
        }
        this._element.enable();
        this._params.bFormButtonList.forEach(enable);
    };

    /**
     * @param {string} text
     **/
    Popup.prototype.notify = function (text) {
        BEM.DOM.update(this._params.$notify, text);
    };

    Popup.prototype._addListeners = function () {
        var _this = this;
        function bindClick(block) {
            block.on('click', _this._onButtonClick, _this);
        }
        this._params.bFormButtonList.forEach(bindClick);

        this._bPopup.on('hide', this._onHide, this);
    };

    Popup.prototype._onButtonClick = function (e) {
        if (e) {
            e.preventDefault();
            e.stopPropagation();
        }
        var type = null;
        if (e.target && (typeof e.target.elem === 'function')) {
            type = e.target.elem('input').attr('type');
        }

        if (type === 'submit') {
            this._onSubmit();
        } else {
            this.hide();
        }
    };

    Popup.prototype._onHide = function (e) {
        if (e) {
            e.preventDefault();
            e.stopPropagation();
        }
        if (this._hideTimeout !== null) {
            clearTimeout(this._hideTimeout);
            this._hideTimeout = null;
        }
        this.enable();
        this.notify('');
        this._element.reset();

        if (this._deferred !== null) {
            this._deferred.fulfill({type: 'hide'});
            this._deferred = null;
        }
    };

    Popup.prototype._onSubmit = function () {
        var camomile = BEMHTML.apply({
            tag: 'span',
            block: 'b-spin',
            js: true,
            mods: {size: '27', theme: 'grey-27', progress: 'yes'},
            mix: [{block: 'b-table-cell-text', elem: 'spin'}]
        });
        this.disable();
        this.notify(camomile);
        this._element.save(this._params.url, this._params.args)
            .then(this._onSaveSuccess.bind(this), this._onSaveError.bind(this));
    };

    /**
     * @param {obejct} data - response from server
     **/
    Popup.prototype._onSaveSuccess = function (data) {
        if (data.error !== undefined) {
            this._onSaveError(data);
            return;
        }

        var text = this._element.getText();
        this.notify(getMessage(data));

        this._hideTimeout = setTimeout(this.hide.bind(this), 500);

        if (this._deferred) {
            this._deferred.notify({type: 'save', text: text, data: data});
        }
    };

    /**
     * @param {obejct} data - response from server
     **/
    Popup.prototype._onSaveError = function (data) {
        this.notify(getMessage(data));
        this.enable();
    };

    // block
    BEM.DOM.decl('b-table-cell-text', {
        onSetMod: {
            js: function () {
                if (this.hasMod('edit') === false) {
                    return;
                }
                this._initBlock();
                if (!this.hasMod('readonly', 'yes')) {
                    this._addListeners();
                } else {
                    this._addListenersReadonly();
                }
            }
        },

        /**
         * Destructs b-popupa as well
         **/
        destruct: function () {
            if (this._popup) {
                this._popup.destruct();
            }
            this.__base.apply(this, arguments);
        },

        /**
         * @return {string}
         **/
        getName: function () {
            return this.params.editBlockParams.name;
        },

        /**
         * Обновляет значение в попапе и в самом блоке
         *
         * @param {string|number} value
         **/
        setValue: function (value) {
            if (value !== this._popup.val()) {
                this._popup.val(value);
                value = this._popup.getText();
            }
            this._value = value;
            BEM.DOM.update(this.elem('text'), this._getFormattedValue(value));
        },

        /**
         * Возвращает форматированное значение, если форматтер для блока
         * был задан
         *
         * @param {string|number} value
         * @return {bemjson|string|number}
         **/
        _getFormattedValue: function (value) {
            var _this = this;
            var formattedValue = value;
            if (this.params.formatterType) {
                var content = Object.keys(this.params.formatterAttrs || {}).reduce(function (result, attr) {
                    result[attr] = _this.params.formatterAttrs[attr];
                    return result;
                }, {});
                content.value = value;

                formattedValue = BEMHTML.apply({
                    block: 'b-formatter',
                    mix: [{block: 'b-table-cell-text', elem: 'formatter'}],
                    mods: {type: this.params.formatterType},
                    value: content
                });
            }
            return formattedValue;
        },

        _initBlock: function () {
            // {string}
            this._editBlockName = this._editBlockName || 'b-form-input';
            // {Popup}
            this._popup = null;
            // {Element}
            this._element = null;

            this._value = this.params.initValue;

            this._onPopupEvent = this._onPopupEvent.bind(this);
        },

        _buildContent: function (text) {
            return BEMHTML.apply([text]);
        },

        _addListeners: function () {
            this.bindTo(this.elem('edit'), 'click', this._onIconClick);
        },

        _addListenersReadonly: function () {
            this.bindTo(this.elem('edit'), 'click', this._onIconClickReadonly);
        },

        _onIconClick: function (e) {
            if (this._popup === null) {
                this._renderPopup();
                this._initPopup();
            }
            var deferred = this._popup.show(this.elem('edit'));
            deferred.then(this._onPopupEvent, null, this._onPopupEvent);
            this._refreshPopupValue();
            this.trigger('edit-start', {initialEvent: e});
        },

        _onIconClickReadonly: function () {
            if (this.hasMod('edit', 'input-icon')) {
                if (this._popup === null) {
                    this._renderPopup();
                    this._popup = this.findBlockOn('popup', 'b-popupa');
                }
                this._popup.show(this.elem('edit'));
            }
        },

        _refreshPopupValue: function () {
            if (!this.hasMod('update-text', 'no')) {
                this._popup.refreshValue();
            }
        },

        /**
         * @param {object} e - {type, data}
         **/
        _onPopupEvent: function (e) {
            if (!e) {
                return;
            }
            if (e.type === 'save') {
                if (!this.hasMod('update-text', 'no')) {
                    this._updateText(e.text);
                }
                if (this.params.editBlockParams && this.params.editBlockParams.trim === true) {
                    this._popup._element.trim();
                }

                this.setMod(this.elem('edit'), 'non-empty', (e.text && e.text.length > 0 ? 'yes' : 'no'));

                this.trigger('change', {
                    data: e.data,
                    block: {
                        name: this.getName(),
                        value: this._popup.val()
                    }
                });
            } else if (e.type === 'hide') {
                this.trigger('edit-stop');
            }
        },

        /**
         * @protected
         * @param {string} value
         **/
        _escapeHtml: function (value) {
            var node = $('<div></div>');
            return node.text(value).html();
        },

        /**
         * @private
         * @param {string|object} text
         **/
        _updateText: function (text) {
            var value = this._buildContent(this._escapeHtml(text));
            BEM.DOM.update(this.elem('text'), this._getFormattedValue(value));
        },

        _renderPopup: function () {
            BEM.DOM.after(this.elem('edit'), BEMHTML.apply({
                block: 'b-table-cell-text',
                mods: {readonly: this.getMod('readonly')},
                elem: 'popup',
                params: this._getElementParams()
            }));
            this.dropElemCache('popup save input form button notify');
        },

        _initPopup: function () {
            var bPopup = this.findBlockOn('popup', 'b-popupa');
            if (this.hasMod('inner', 'yes')) {
                bPopup.domElem.on('click', function (e) {
                    e.stopPropagation();
                });
            }
            this._popup = new Popup(bPopup, {
                titles: this.params.titles,
                url: this.params.url,
                args: this._getAjaxParams(),
                editBlockName: this._editBlockName,
                $notify: this.elem('notify'),
                bFormButtonList: this.findBlocksOn('button', 'b-form-button') || []
            });
        },

        /**
         * @return {object}
         **/
        _getElementParams: function () {
            var params = $.extend({}, this.params);
            delete params.args;
            delete params.url;
            params.elemMods = this._getElemMods();
            return params;
        },

        /**
         * @return {object}
         **/
        _getElemMods: function () {
            var mods = $.extend({}, this.getMods());
            delete mods.js;
            return mods;
        },

        /**
         * @return {object}
         **/
        _getAjaxParams: function () {
            var params = $.extend({}, this.params.args);
            params.ajax = 1;
            if (!params.sing) {
                params.sign = getSign(this.params.url);
            }
            return params;
        }

    });

})();
