(function (BEM) {
    'use strict';

    var promises = BEM.blocks['i-promises'],
        ajax = promises.ajax,

        FIELD_TYPES = ['entity_fields', 'dimension_fields', 'fields'];

    /**
     * [1,2,3].filter(notEqual(2)) -> [1,3]
     */
    function notEqual(value) {
        return function (x) { return x !== value; };
    }

    /**
     * [{foo: 'bar'}, {baz: 'qux'}].filter(hasProp('foo')) -> [{foo: 'bar'}]
     */
    function hasProp(prop) {
        return function (x) { return x.hasOwnProperty(prop); };
    }

    /**
     * [3,4,5,6].filter(notIn([1,2,4,5])) -> [3,6]
     */
    function notIn(arr) {
        return function (x) { return arr.indexOf(x) < 0; };
    }

    /**
     * [{foo:'bar'}, {foo: 'baz'}].map(getProp('foo')) -> ['bar', 'baz']
     */
    function getProp(prop) {
        return function (obj) { return obj[prop]; };
    }

    /**
     * [[1,2],[3,4],[5,6]] -> [1,2,3,4,5,6]
     */
    function flatten(arr) {
        return arr.reduce(function (a, x) {
            return a.concat(x);
        }, []);
    }

    /**
     * num('2') -> 2
     * num('abc') -> 0
     */
    function num(a) {
        var result = Number(a);
        if (isNaN(result)) { return 0; }
        return result;
    }

    /**
     * unique([1,1,2,2]) -> [1,2]
     */
    function unique(arr) {
        return Object.keys(arr.reduce(function (a, x) {
            a[x] = true;
            return a;
        }, {}));
    }

    /**
     * getKeys({foo:'bar', baz: 'qux'}) -> ['foo', 'baz']
     * getKeys('abc') -> []
     */
    function getKeys(obj) {
        if (typeof obj != 'object') { return []; }
        return Object.keys(obj);
    }

    /**
     * @return arguments sum depending on types
     */
    function sum(a, b) {
        switch (typeof a) {
        case 'undefined':
            return b;

        case 'number':
            return sumNumber(a, b);

        case 'object':
            if (Array.isArray(a)) { return sumArray(a, b); }
            return sumObject(a, b);

        case 'string':
            // string may contain a number (i. e. "2.45")
            if (sumNumber(a, b) === 0) {
                return a;
            }

            return sumNumber(a, b);

        default:
            return a;
        }
    }

    function sumNumber(a, b) {
        return num(a) + num(b);
    }

    /**
     * sums up object properties:
     *  - number props results in sum of values
     *  - object props results in recursive call
     *  - string props results in first string
     */
    function sumObject(a, b) {
        var result = {},
            props = unique(getKeys(a).concat(getKeys(b)));

        props.forEach(function (prop){
            result[prop] = sum(a[prop], b[prop]);
        });

        return result;
    }

    /**
     * Sums each item of a with respective item of b
     */
    function sumArray(a, b) {
        var result = [];
        var l = Math.max(a.length, b.length);

        for (var i = 0; i < l; i++) {
            result[i] = sum(a[i], b[i]);
        }

        return result;
    }

    function sendRequest(request) {
        return ajax.postJSON('/statistics/query', {params: JSON.stringify(request)});
    }

    /**
     * Sends specified requests, returns promise which fulfills when all requests completed
     */
    function sendRequests(requests) {
        return promises.all(requests.map(sendRequest));
    }

    function flatTree(nodes) {
        return nodes.reduce(function (result, node) {
            result.push(node);
            return result.concat(flatTree(node.children || []));
        }, []);
    }

    /**
     * {
     *     entity_fields: [{id: 'foo'}, {id: 'qux'}],
     *     dimension_fields: [{id: 'bar'}],             ->   ['foo', 'qux', 'bar', 'baz', 'quux', <children fields>]
     *     fields: [{id: 'baz'}, {id: 'quux'}]
     *     children: [{ ... }]
     * }
     *
     * @param {array} entityTree
     **/
    function EntityList(entityTree) {
        var entities = flatTree(entityTree);
        this.fields = FIELD_TYPES.map(function(fieldsType) {
            return flatten(entities.map(getProp(fieldsType))).map(getProp('id'));
        });

        this.fields = flatten(this.fields);

        this.hasFields = this.hasFields.bind(this);
        this.hasField = this.hasField.bind(this);
        this.hasNoField = this.hasNoField.bind(this);
    }

    /**
     * @param {array:string} fieldList
     **/
    EntityList.prototype.hasFields = function (fieldList) {
        return fieldList.every(this.hasField);
    };

    EntityList.prototype.hasField = function (field) {
        return this.fields.indexOf(field) > -1;
    };

    EntityList.prototype.hasNoField = function (field) {
        return !this.hasField(field);
    };

    /**
     *  params: {
     *      fields: ['downloads', 'partner_wo_nds', 'partner_w_nds'],
     *      dimension_fields: [],
     *      entity_fields: []
     *  }
     */
    function RequestPayload(requestParams) {
        this.params = requestParams;
        this.removeField = this.removeField.bind(this);
    }

    RequestPayload.create = function(requestParams) {
        return new RequestPayload(requestParams);
    };

    /**
     * Removes field from this.params.fields, this.params.dimension_fields
     * or this.params.entity_fields
     */
    RequestPayload.prototype.removeField = function (field) {
        var _this = this;

        FIELD_TYPES.forEach(function (fieldsType) {
            var fields = _this.params[fieldsType],
                fieldIndex = fields.indexOf(field);

            if (fieldIndex > -1) {
                fields.splice(fieldIndex, 1);
            }
        });
    };

    RequestPayload.prototype.isEmpty = function () {
        var params = this.params;
        return FIELD_TYPES.every(function (fieldType) {
            return !params[fieldType] || params[fieldType].length === 0;
        });
    };

    function groupByFieldName(result, item) {
        result = result || {};

        getKeys(item).filter(notEqual('currency_id')).forEach(function(key) {
            result[key] = result[key] || {};
            result[key][item.currency_id] = item[key];
        });

        return result;
    }

    function getRowCreator(data) {

        function getCellData(fieldName) {
            var value = data[fieldName];

            if (value !== undefined) {
                return value;
            }

            return BEM.I18N('b-widget', fieldName) || '';
        }

        function createCell(cellSettings) {
            return {
                type: cellSettings.type,
                name: cellSettings.name,
                data: cellSettings.data || getCellData(cellSettings.name)
            };
        }

        return function (settings) {
            return {cells: settings.cells.map(createCell)};
        };
    }

    function getRowSumsByCurrency(row) {
        return row.cells.map(getProp('data')).reduce(sumObject, {});
    }

    function removeZeroes(row) {
        var datas = row.cells.map(getProp('data')),
            rowSumsByCurrency = getRowSumsByCurrency(row),
            rowCurrencies = getKeys(rowSumsByCurrency),
            notZeroCurrencies = rowCurrencies.filter(function (currency) {
                rowSumsByCurrency[currency] > 0;
            });

        if (notZeroCurrencies.length === 0) {
            rowCurrencies = rowCurrencies.slice(1);
        }

        rowCurrencies.forEach(function(currency) {
            if (rowSumsByCurrency[currency] === 0) {
                datas.forEach(function (data) {
                    if (typeof data == 'object') {
                        delete data[currency];
                    }
                });
            }
        });

        return row;
    }

    function sumRows(result, row) {
        var rowCells = row.cells.map(function(cell) {
            return cell.type && cell.type === 'money' ? cell : {};
        });

        result.cells = sumArray(result.cells, rowCells);
        return result;
    }

    function getTotals(rows) {
        var totalRow = {
            type: 'total',
            cells: [{type: 'text', data: BEM.I18N('b-widget', 'Total')}]
        };

        return rows.reduce(sumRows, totalRow);
    }

    function parseResponses(rowSettings, responses) {
        var data = flatten(responses.map(getProp('data'))).reduce(groupByFieldName, {}),
            rows = rowSettings.map(getRowCreator(data)).map(removeZeroes);

        return {
            currencies: responses.filter(hasProp('currencies'))[0].currencies,
            rows: rows,
            totals: getTotals(rows)
        };
    }

    function getResponseParser(rowSettings) {
        return function (responses) {
            return parseResponses(rowSettings, responses);
        };
    }

    function getRequestFields(params) {
        return flatten(FIELD_TYPES.map(function(prop) {
            return params[prop];
        }));
    }

    function getRequestsFields(requests) {
        return flatten(requests.map(getRequestFields));
    }

    function filterInvisibleRows(rowSettings, entityTree) {
        var entityList = new EntityList(entityTree);

        return rowSettings.filter(function (row) {
            var rowFields = row.cells.filter(hasProp('name')).map(getProp('name'));
            return row.type == 'header' || entityList.hasFields(rowFields);
        });
    }

    function filterUnusedRequests(requests, visibleRows) {
        var payloads = requests.map(RequestPayload.create),
            cells = flatten(visibleRows.map(getProp('cells'))),
            visibleFields = cells.filter(hasProp('name')).map(getProp('name')),
            unavailableFields = getRequestsFields(requests).filter(notIn(visibleFields));

        return payloads.map(function(payload) {
            unavailableFields.forEach(payload.removeField);
            return payload.isEmpty() ? null : payload.params;
        }).filter(notEqual(null));
    }

    function loadData(entityTree, requests, rowSettings) {
        rowSettings = filterInvisibleRows(rowSettings, entityTree);
        requests = filterUnusedRequests(requests, rowSettings);

        return sendRequests(requests).then(getResponseParser(rowSettings), null);
    }

    // block
    BEM.DOM.decl({block: 'b-widget', modName: 'type', modVal: 'products-revenue'}, {
        onSetMod: {
            js: function () {
                this.__base.apply(this, arguments);
                this._show();
            }
        },

        refresh: function () {
            this.__base.apply(this, arguments);
            this._show();
        },

        _initBlock: function () {
            this.__base.apply(this, arguments);

            this._onError = this._onError.bind(this);
            this._renderTemplate = this._renderTemplate.bind(this);
            this._updateContainer = this._updateContainer.bind(this);
            this._clearLoading = this._clearLoading.bind(this);
        },

        _show: function () {
            if (!this.domElem || !this._opts.entities_tree) {
                return;
            }

            this.setMod('state', 'loading');

            loadData(this._opts.entities_tree[0], this._opts.js.requests, this._opts.js.rowSettings)
                .then(this._renderTemplate, null)
                .then(this._updateContainer, this._onError)
                .always(this._clearLoading);
        },

        _clearLoading: function() {
            this.delMod('state');
        },

        _onError: function () {
            this._onEmptyData();
        },

        _renderTemplate: function (data) {
            return BEMHTML.apply({
                block: 'b-widget',
                elem: 'container',
                elemMods: {type: 'products-revenue'},
                currencyList: data.currencies,
                rows: data.rows,
                cols: this._opts.js.cols,
                totals: data.totals
            });
        },

        _updateContainer: function (html) {
            BEM.DOM.init(this.elem('container').replaceWith(html));
            this.dropElemCache('table container');
        }
    });
})(BEM);
