(function (BEM) {

    'use strict';

    var ajax = BEM.blocks['i-promises'].ajax;
    var contains = require('lodash').contains;
    var map = require('lodash').map;
    var chain = require('lodash').chain;
    var isEmpty = require('lodash').isEmpty;
    var isPlainObject = require('lodash').isPlainObject;
    var isArray = require('lodash').isArray;

    var SEARCH_STATUSES = {
        NO_ACCESS: 'no_access',
        NOT_FOUND: 'not_found',
        UNKNOWN: 'unknown',
        OK: 'ok'
    };

    var SEARCH_TYPES = {
        PAGE_ID: 'page_id',
        LOGIN: 'login'
    };

    BEM.DOM.decl({name: 'b-widget', modName: 'type', modVal: 'products-search'}, {

        onSetMod: {
            js: {
                inited: function () {
                    this.__base.apply(this, arguments);

                    this._setConfig();

                    this.radio = this.findBlockInside('b-form-radio');
                    this.radio.bindTo('radio', 'change', this._onRadioChange.bind(this));

                    this.searchInputBlock = this.findBlockInside('b-form-input');
                    this.searchInput = this.searchInputBlock.elem('input');

                    this.elem('content-wrapper').on('scroll', this._onContentScroll.bind(this));
                }
            }
        },

        destruct: function () {
            this.elem('content-wrapper').off('scroll');
            this.__base.apply(this, arguments);
        },

        /**
         * @private
         */
        _setConfig: function () {
            this.baseUrl = '/product/get_product_structure_for_';
            this.searchTypesWithSuggest = ['login'];

            this.searchTypeTranslations = {
                page_id: 'Page ID',
                login: BEM.I18N('b-widget', 'login')
            };

            this._setCampaignsTablesConfig();
        },

        /**
         * @private
         */
        _onContentScroll: function () {
            this.searchInputBlock._popup && this.searchInputBlock._popup.delMod('visibility');
        },

        /**
         * Хардкодим табличные заголовки
         * и поля ячеек, которые будут во всех таблицах(т.е. '#', 'domain' и 'id')
         *
         * @private
         */
        _setCampaignsTablesConfig: function () {
            var i18n = BEM.I18N.bind(BEM, 'b-widget');

            var baseHeaders = ['#', i18n('Domain'), 'Page ID'];
            this.campaignHeaders = {
                base: baseHeaders,
                context_on_site_campaign: baseHeaders.concat([i18n('Direct'), i18n('RTB'), i18n('Stripe'), 'ADFOX', 'Adblock']),
                search_on_site_campaign: baseHeaders.concat([i18n('Direct'), i18n('Premium')]),
                video_an_site: baseHeaders.concat(['Instream', 'InPage']),
                mobile_app_settings: baseHeaders.concat([i18n('RTB')]),
                taxopark: ['#', 'Caption', 'ID'],
                dsp: ['#', 'Caption', 'ID', 'Multistate']
            };

            var baseCols = ['domain', 'id'];
            this.campaignCols = {
                base: baseCols,
                context_on_site_campaign: ['domain', 'page_id'],
                search_on_site_campaign: ['domain', 'page_id'],
                mobile_app_settings: ['store_id', 'context_page_id'],
                taxopark: ['caption', 'id'],
                dsp: ['short_caption', 'id', 'multistate']
            };
        },

        /**
         * Срабатывает при изменении типа поиска
         *
         * @private
         */
        _onRadioChange: function (e) {
            var newSearchType = e.target.value;
            this._toggleInputSuggest(newSearchType);
            this.afterCurrentEvent(this._setInputFocus, this);
        },

        /**
         * Включаем suggest для поиска по логину, иначе отключаем
         *
         * @private
         * @param {string} newSearchType установленный тип поиска
         */
        _toggleInputSuggest: function (newSearchType) {
            var enableSuggest = contains(this.searchTypesWithSuggest, newSearchType);
            var autocompleteValue = enableSuggest ? 'yes' : '';

            this.searchInputBlock.setMod('autocomplete', autocompleteValue);
        },

        /**
         * Устанавливает фокус на поле ввода
         *
         * @private
         */
        _setInputFocus: function () {
            this.searchInput.trigger('focus');
            this.searchInputBlock._onFocus();
        },

        /**
         * При поиске по page_id перекидываем на урл, приехавший от ручки в JSON ответе
         * урл находится в свойстве url в первом объекте в data.result,
         * т.е. 'data: {result: {'some_key': {url: '/url'}}}'
         *
         * @private
         * @param {json} data возвращенный с бекенда json data.result
         */
        _redirectToPage: function (data) {
            var url = chain(data.data)
                .pluck('url')
                .first()
                .valueOf();

            window.location.assign(url);
        },

        /**
         * Обработчик нажатия кнопки "Поиск"
         *
         * @private
         */
        _onSubmit: function () {
            this._toggleLoadingState(true);

            var searchType = this.currentSearchType = this.radio.val();
            var searchValue = this.searchInputBlock.val();

            var searchUrl = [
                this.baseUrl,
                searchType,
                '?',
                searchType,
                '='
            ].join('');

            var url = searchUrl + searchValue;

            this._requestData(url);
        },

        /**
         * Тоггл спиннера
         *
         * @private
         * @param {boolean} isLoading показыват спиннер при true, иначе убирает его
         */
        _toggleLoadingState: function (isLoading) {
            var stateValue = isLoading ? 'loading' : '';
            this.setMod('state', stateValue);
        },

        /**
         * Обращение к ручке и последующий вызов обработчиков поиска/ошибки
         *
         * @private
         * @param {string} url
         */
        _requestData: function (url) {
            ajax.get(url)
                .then(this._onSearch, this._onAjaxError, this)
                .then(this._toggleLoadingState, this);
        },

        /**
         * Рендерим результат выполнения поиска по Логину
         *
         * @private
         * @param {json} result пришедший с бекенда json
         * @return {bemjson} BEMJSON результирующих таблиц при поиске по логину
         */
        _renderCampaignsList: function (result) {
            var campaignsBlocks = map(result, function (item) {
                var model = item.model;
                var campaignHeaders = this.campaignHeaders[model] || this.campaignHeaders.base;
                var campaignCols = this.campaignCols[model] || this.campaignCols.base;

                var data = item.data;
                var hasData = isArray(data);

                var campaignsHeader = {
                    block: 'b-campaigns-header',
                    mix: {block: 'b-campaigns-block', elem: 'header'},
                    content: [
                        {
                            elem: 'url',
                            content: [
                                {
                                    block: 'b-link',
                                    mix: {block: 'b-campaigns-header', elem: 'title'},
                                    url: item.url,
                                    content: item.name
                                },
                                {
                                    elem: 'amount',
                                    content: hasData ? ' (' + data.length + '):' : null
                                }
                            ]
                        }
                    ]
                };

                var campaignsTable;

                if (hasData) {
                    campaignsTable = {
                        block: 'b-campaigns-list',
                        mix: {block: 'b-campaigns-block', elem: 'list'},
                        content: [
                            {
                                elem: 'header',
                                content: {
                                    elem: 'header-row',
                                    content: campaignHeaders.map(function (header) {
                                        return {
                                            elem: 'header-column',
                                            content: header
                                        };
                                    })
                                }
                            },
                            {
                                elem: 'body',
                                content: (data || []).map(function (campaign, campaignIndex) {
                                    var isDeleted = campaign.is_deleted;

                                    var counterCol = {
                                        elem: 'column',
                                        content: campaignIndex + 1
                                    };

                                    var constantCols = (campaignCols || []).map(function (col) {
                                        var isPageIdCol = (col === 'id' || col === 'context_page_id' || col === 'page_id');
                                        var isDomainCol = (col === 'domain' || col === 'store_id');

                                        return {
                                            elem: 'column',
                                            content: isPageIdCol
                                                ? {
                                                    block: 'b-link',
                                                    url: campaign.url,
                                                    content: campaign[col]
                                                } :
                                                campaign[col],
                                            mods: {
                                                type: isDomainCol
                                                    ? 'domain'
                                                    : ''
                                            }
                                        };
                                    });

                                    constantCols = [counterCol].concat(constantCols);

                                    var productCols = (campaign.product || []).map(function (product) {
                                        return {
                                            elem: 'column',
                                            content: {
                                                block: 'b-link',
                                                url: product.url,
                                                content: String(product.count)
                                            }
                                        };
                                    });

                                    var rowCols = constantCols.concat(productCols);

                                    return {
                                        elem: 'row',
                                        mods: {deleted: isDeleted ? 'yes' : ''},
                                        content: rowCols
                                    };
                                })
                            }
                        ]
                    };
                } else {
                    campaignsTable = null;
                }

                return {
                    block: 'b-campaigns-block',
                    content: [
                        campaignsHeader,
                        campaignsTable
                    ]
                };
            }, this);

            BEM.DOM.update(this.elem('result-container'), BEMHTML.apply({
                block: 'b-campaigns-blocks',
                content: campaignsBlocks
            }));
        },

        /**
         * Обработчик данных, пришедших от ручки, если не была возвращена ошибка
         *
         * @private
         * @param {object} data возвращенный с бекенда JSON объект
         */
        _onSearch: function (data) {
            if (data.success === false) {
                this._onAjaxError();
                return false;
            }

            var result = data.result;
            var resultIsEmpty = isEmpty(result);

            var status = result.status || SEARCH_STATUSES.UNKNOWN;
            if (isEmpty(result)) {
                status = SEARCH_STATUSES.NOT_FOUND;
            }

            switch (status) {
            case SEARCH_STATUSES.NO_ACCESS:
                this._onSearchAccessFail();
                return false;

            case SEARCH_STATUSES.NOT_FOUND:
                this._onSearchResultsFail();
                return false;

            case SEARCH_STATUSES.OK:
            case SEARCH_STATUSES.UNKNOWN:
            }

            switch (this.currentSearchType) {
            case SEARCH_TYPES.PAGE_ID:
                return void this._redirectToPage(result);

            case SEARCH_TYPES.LOGIN:
                return void this._renderCampaignsList(result);

            default:
                this._onSearchResultsFail();
                return false;
            }
        },

        /**
         * Хелпер для обработки ошибок
         *
         * @private
         * @param {any} content сообщение об ошибке, которое будет отображено юзеру
         */
        _onError: function (content) {
            BEM.DOM.update(this.elem('result-container'), BEMHTML.apply({
                block: 'b-widget',
                elem: 'error',
                content: content
            }));
        },

        /**
         * Вывод сообщения при ошибке соединения с сервером
         *
         * @private
         */
        _onAjaxError: function () {
            this._onError(BEM.I18N('b-widget', 'Server error'));
        },

        /**
         * Вывод сообщения, если отсутствуют результаты поиска
         *
         * @private
         */
        _onSearchResultsFail: function () {
            switch (this.currentSearchType) {
            case SEARCH_TYPES.LOGIN:
                return void this._onSearchAccessFail();

            default:
                this._onError(BEM.I18N('b-widget', 'Nothing found'));
            }
        },

        /**
         * Вывод сообщения, если поиск с имеющимися правами поиска ничего не выдал
         *
         * @private
         */
        _onSearchAccessFail: function () {
            var searchType = this.searchTypeTranslations[this.currentSearchType];

            this._onError(
                BEM.I18N('b-widget',
                         'This {searchType} hasn\'t been found in products, that are available to you'
                        ).replace('{searchType}', searchType)
            );
        }
    },
    {
        live: function () {
            this.__base.apply(this, arguments);

            this.liveBindTo('search-form', 'submit', function (e) {
                e.preventDefault();
                this._onSubmit();
            });

            return false;
        }

    });

})(BEM);
