(function () {
    'use strict';

    var datas = require('utils/statistic');
    var chain = require('lodash').chain;
    var uniq = require('lodash').uniq;
    var compose = require('lodash').compose;
    var isEmpty = require('lodash').isEmpty;
    var assign = require('lodash').assign;
    var cloneDeep = require('lodash').cloneDeep;
    var forEach = require('lodash').forEach;
    var pluck = require('lodash').pluck;
    var difference = require('lodash').difference;
    var filter = require('lodash').filter;

    /**
     * @param  {Object}  fieldsToRender
     * @param  {Boolean} withCurrency
     * @return {Array}
     */
    function fieldsWithDescription(fieldsToRender, withCurrency) {
        var dimension_fields = fieldsToRender.dimension_fields;

        if (withCurrency) {
            dimension_fields = dimension_fields.filter(function (field) {
                return field.id !== 'currency_id';
            });
        }

        var list = dimension_fields
            .concat(withCurrency ? {
                hint: '',
                id: 'currency_id',
                short_title: BEM.I18N('b-statistic', 'currency'),
                title: BEM.I18N('b-statistic', 'currency'),
                type: 'currency'
            } : [])
            .concat(fieldsToRender.entity_fields, fieldsToRender.fields);

        list.forEach(function (field) {
            field.id === 'date' && (field.type = 'date');
            field.title || (field.title = field.label);
        });

        return uniq(list, 'id');
    }

    BEM.DOM.decl('b-widget', {

        fieldKinds: ['fields', 'entity_fields', 'dimension_fields'],

        onSetMod: {
            js: function () {
                this._initBlock();
                this._addListeners();
            },
            state: {
                loading: function () {
                    var spinBlock = this.findBlockOn('mask-spin', 'b-spin');

                    if (spinBlock && this._showSpin) {
                        spinBlock.setMod('progress', 'yes');
                    }
                },

                '': function () {
                    var spinBlock = this.findBlockOn('mask-spin', 'b-spin');

                    if (spinBlock) {
                        spinBlock.delMod('progress');
                    }
                }
            },
            locked: {
                yes: function () {
                    this._showLock();
                },
                '': function () {
                    this._hideLock();
                }
            },
            width: function () {
                if (this._isInited) {
                    this.afterCurrentEvent(this._onResize);
                }
            }
        },

        /**
         * @return {string}
         **/
        getTitle: function () {
            return this.params.title;
        },

        getType: function () {
            return this.getMod('type');
        },

        /**
         * @return {number}
         **/
        getId: function () {
            return this.params.id;
        },

        refreshSize: function (newBlockSize) {
            this._refreshSize(newBlockSize);
        },

        /**
         * Refresh opts, mods and other elements
         **/
        refresh: function (options) {
            var mod;

            if (!options) {
                return;
            }

            $.extend(this._opts, options);
            if ('mods' in options) {
                for (mod in options.mods) {
                    this.setMod(mod, options.mods[mod]);
                }
            }

            if ('title' in options) {
                this.findElem('title-content').text(options.title);
            }
        },

        /**
         * Needed for b-dashboard
         *
         * @return {string}
         **/
        _initBlock: function () {
            // {jQuery}
            this.$lock = null;
            this.size = {
                block: {
                    height: 0,
                    width: 0,
                    offset: {
                        top: 0,
                        left: 0
                    }
                },
                lock: {
                    height: 0,
                    width: 0
                }
            };

            this._opts = {js: $.extend(true, {}, this.params)};
            this._showSpin = true;

            this._isInited = true;
        },

        /**
         * Events Listeners
         **/
        _addListeners: function () {
            this.bindTo('action-close', 'click', this._onCloseClick);
            this.bindTo('action-setup', 'click', this._onSetupClick);
        },

        _onCloseClick: function (e) {
            e.preventDefault();
            BEM.channel('widget').trigger('close', this.params.id);
        },

        _onLoad: function (ajaxReq, d) {
            var fieldsOptions,
                fieldsToRender = this._getFieldsDesc(ajaxReq.levels[0].id, ajaxReq),
                hasDateField,
                hasFieldsToRender,
                //logarithmic = this._opts && this._opts.js.logarithmic,
                opts = {dateGroupBy: this.dateGroupBy};

            if (!this.domElem) {
                return;
            }
            this.delMod('state');
            if (fieldsToRender) {
                fieldsOptions = this._getExtraOptions(fieldsToRender);
            }

            var errors = chain(d)
                .pluck('error')
                .filter(compose(datas.inverse, isEmpty));

            if (errors.length) {
                this._onEmptyData(errors.join('\n'));

                return false;
            }

            if (d.every(function (data) {
                return data.hasOwnProperty('data');
            })) {
                if (!d.some(function (data) {
                    return data.data.length > 0;
                })) {
                    this._onEmptyData();

                    return false;
                } else {
                    this._refreshTitle(ajaxReq);
                }

                hasDateField = d[0].data[0] && d[0].data[0].date;
                hasFieldsToRender = (fieldsToRender.entity_fields.length && fieldsToRender.dimension_fields.length) ||
                                    fieldsToRender.fields.length;

                if (hasDateField || hasFieldsToRender) {
                    this._renderChart(d, fieldsToRender, fieldsOptions, assign(opts, {period: ajaxReq.period}));
                }
            }
        },

        _onSetupClick: function (e) {
            e.preventDefault();
            BEM.channel('widget').trigger('setup', this.params.id);
        },

        _renderChart: function (d, fieldsToRender, fieldsOptions, opts) {
            var actualCurrencies = chain(d)
                .map(function (data) {
                    return pluck(data.data, 'currency_id').filter(Boolean);
                })
                .flatten()
                .uniq()
                .valueOf();

            var hasCurrencies = actualCurrencies.length > 0;
            var fields = fieldsWithDescription(fieldsToRender, hasCurrencies);
            var hasDate = filter(fields, {id: 'date'}).length > 0;

            var valueableFields = datas.valueableFields(fieldsToRender);
            var groupingFields = difference(datas.fieldsList(fields), valueableFields);

            var params = {
                fields: datas.vocabulary(fields, 'id'),
                groupingFields: groupingFields,
                hasCurrencies: hasCurrencies,
                hasDate: hasDate,
                interval: opts.dateGroupBy,
                period: opts.period,
                singleCurrency: actualCurrencies.length === 1,
                sortOrder: [],
                valueableFields: valueableFields
            };

            // Валютный словарик.
            if (hasCurrencies) {
                params.currencies = datas.vocabulary(d.map(function (report) {
                    return report.currencies;
                }).filter(Boolean)[0], 'id');

                forEach(params.currencies, function (obj) {
                    obj.html = datas.formatCurrency(obj.code);
                });
            }

            // totals.
            if (groupingFields.length > 0) {
                var totals = pluck(d, 'total');
                var sample = cloneDeep(totals.filter(Boolean)[0]);

                forEach(sample, function (val) {
                    forEach(val, function (v, k) {
                        val[k] = typeof v === 'number' ? 0 : '0';
                    });
                });

                params.totals = totals.map(function (total) {
                    return total || sample;
                });
            }

            // Предварительная сортировка для фиксации нулей.
            datas.sortByFields(d[0].data, params.groupingFields, params.fields);

            d = datas.fixData(d, params);

            d.forEach(function (data) {
                datas.sortByFields(data.data, params.groupingFields, params.fields);
            });

            var bemjson = {
                block: 'b-statistic-chartbox',
                mods: {
                    size: 'small',
                    mode: 'single-chart'
                },
                js: chain(params)
                    .pick([
                        'currencies',
                        'fields',
                        'groupingFields',
                        'hasCurrencies',
                        'hasDate',
                        'interval',
                        'period',
                        'valueableFields'
                    ])
                    .assign({
                        data: pluck(d, 'data'),
                        update: true
                    })
                    .valueOf()
            };

            BEM.DOM.update(this.elem('container'), BEMHTML.apply(bemjson));
        },

        /**
         * Refresh block and lock sizes
         *
         * @param {object} newBlockSize
         **/
        _refreshSize: function (newBlockSize) {
            var $el = this.domElem;

            this.size = {
                block: {
                    height: $el.height(),
                    width: $el.width(),
                    offset: $el.offset()
                },
                lock: this._getLockSize()
            };

            if (newBlockSize) {
                $.extend(this.size.block, newBlockSize);
            }
        },

        _onResize: function () {
            this._refreshSize();
            BEM.channel('widget').trigger('resize');
        },

        /**
         * Return loch sizes
         *
         * @return {string:number} - {height, width}
         **/
        _getLockSize: function () {
            var wrapper = this.elem('content-wrapper');
            return {
                height: wrapper.outerHeight(),
                width: wrapper.outerWidth()
            };
        },

        /**
         * Render lock for edit mode
         **/
        _showLock: function () {
            if (!this.$lock || !this.$lock.length) {
                this.$lock = this.elem('lock');
            }

            this.$lock.css({
                right: this.size.block.width - this.size.lock.width,
                bottom: this.size.block.height - this.size.lock.height
            }).show();
        },

        /**
         * Remove lock
         **/
        _hideLock: function () {
            this.$lock.hide();
        },

        /**
         * TODO: write description
         *
         *
         **/
        _getData: function (list, id, filterData) {
            var childData,
                fieldKind,
                i,
                j,
                l = list.length,
                node,
                result = {};

            for (i = 0; i < l; i++) {
                node = list[i];

                if (node.id == id) {
                    result = {title: node.title};

                    for (j = 0; j < 3; j++) {
                        fieldKind = this.fieldKinds[j];

                        result[fieldKind] = this._filterFields(
                            node[fieldKind],
                            filterData[fieldKind]
                       );
                    }
                    return result;
                } else if (node.children !== null && node.children !== undefined) {
                    childData = this._getData(node.children, id, filterData);

                    if (childData !== null) {
                        return childData;
                    }
                }
            }

            return null;
        },

        /**
         * TODO: write description
         *
         *
         **/
        _filterFields: function (fields, filterSet) {
            var i, fieldsObj = {};

            for (i = 0; i < fields.length; i++) {
                var field = fields[i];

                fieldsObj[field.id] = field;
            }

            var result = [];

            for (i = 0; i < filterSet.length; i++) {
                var fieldParts = filterSet[i].split('|');

                result.push(fieldsObj[
                    fieldParts[0]
               ]);

                if (fieldParts[0] == 'date') {
                    this.dateGroupBy = fieldParts[1];
                }
            }

            return result;
        },

        /**
         * Покажет сообщение, что данных нет,
         * либо сообщение об ошибке.
         *
         * @param  {String} errDesc
         */
        _onEmptyData: function (errDesc) {
            BEM.DOM.update(this.elem('container'), BEMHTML.apply({
                block: 'b-widget',
                elem: 'empty-result-msg',
                content: errDesc || BEM.I18N('b-widget', 'No data')
            }));
        }
    });
})();
