// Various utility functions

(function(BEM, window, undefined) {

    'use strict';

    /**
     * Debounces function.
     * Sequential call of returned funtion will
     * result in one call of passed function <fn>
     * in <timeout> milliseconds after last call.
     *
     * @param {function} fn function to decorate
     * @param {number} timeout debouncing timeout in ms
     * @returns decorated function
     */
    function debounce(fn, timeout) {
        var args, timer = null;

        return function() {
            var _this = this;

            args = arguments;

            if (timer !== null) { window.clearTimeout(timer); }

            timer = window.setTimeout(function() {
                fn.apply(_this, args);
                timer = null;
            }, timeout);
        };
    }

    /**
     * Throttles function.
     * Sequential call of returned function will
     * result in one call of passed function <fn>
     * in <timeout> milliseconds.
     *
     * @param {function} fn function to decorate
     * @param {number} timeout throttling timeout in ms
     * @returns decorated function
     */
    function throttle(fn, timeout) {
        var args = null,
            timer = null;

        return function() {
            var _this = this;

            args = arguments;

            function f() {
                if (args) {
                    fn.apply(_this, args);
                    args = null;
                    timer = window.setTimeout(f, timeout);
                } else {
                    timer = null;
                }
            }

            if (timer === null) {
                f();
            }
        };
    }

    BEM.DOM.decl('i-utils', {}, {
        debounce: debounce,
        throttle: throttle
    });

})(BEM, window);
