'use strict';

var fs = require('fs');

var buildConfig = require('../buildConfig');
var allLevels = buildConfig.allLevels;
var testLevels = buildConfig.testLevels;
var allTestJsLevels = buildConfig.allTestJsLevels;
var allTestDepsLevels = buildConfig.allTestDepsLevels;
var devMode = buildConfig.devMode;

var use = require('./techs');

module.exports = function (config) {
    config.mode('production', function () {
        config.setLanguages(buildConfig.languages);

        config.node('.tmp/priv', function (nodeConfig) {
            nodeConfig.addTechs([
                use('levels', {levels: allLevels}),

                // Базовые технологии
                use('bemdecl-from-bemjson'),
                use('deps-old'),
                use('files'),

                // bemjson из tt2
                use('tt2bemjson', {
                    target: '?.bemjson.js',
                    sources: buildConfig.allTemplates
                }),

                // Серверный bemhtml и priv.js
                use('bemhtml', {devMode: devMode}),
                use('i18n-merge-keysets', {lang: 'all'}),
                use('i18n-merge-keysets', {lang: '{lang}'}),
                use('i18n-lang-js', {lang: 'all'}),
                use('i18n-lang-js', {lang: '{lang}'}),
                use('priv-js'),
                use('priv-js-i18n-all', {langTargets: ['all'].concat(config.getLanguages()).map(getLangTarget)})
            ]);

            nodeConfig.addTargets([
                '?.all.priv.js'
            ]);
        });

        config.node('.tmp/common', function (nodeConfig) {
            var techs = getClientTechs(buildConfig.commonLevels, buildConfig.allTemplates);
            var targets = getClientTargets();

            nodeConfig.addTechs(techs);
            nodeConfig.addTargets(targets);
        });

        config.node('.tmp/promo', function (nodeConfig) {
            var techs = getClientTechs(buildConfig.promoLevels, buildConfig.promoTemplates, 'promo');
            var targets = getClientTargets();

            nodeConfig.addTechs(techs);
            nodeConfig.addTargets(targets);
        });
    });

    /**
     * Конфиг для сборки примеров
     */
    config.mode('examples', function () {
        /* Языки для примеров */
        config.setLanguages(['ru', 'en']);

        /* Добавление набора нод в сборку */
        config.nodes('blocks.*/*/*.examples/*');

        /* Конфигурирование нод */
        config.nodeMask(/\.examples\/[\w\-]+$/, function (nodeConfig) {
            /* Зависимости для статики */
            nodeConfig.addTechs([
                use('levels', '__?', {levels: getDevLevels(nodeConfig)}),
                use('file-provider', {target: '?.bemjson.js'}),
                use('bemdecl-from-bemjson', '__?', {sourceTarget: '?.bemjson.js'}),
                use('deps-old', '__?'),
                use('files', '__?')
            ]);

            /* Зависимости для клиентских шаблонов */
            nodeConfig.addTechs([
                use('bemdecl-from-deps-by-tech', '__?.bh', {filesTarget: '__?.files'}),
                use('deps-old', '__?.bh', {levelsTarget: '__?.levels'}),
                use('files', '__?.bh', {levelsTarget: '__?.levels'})
            ]);

            /* Объединенные зависимости */
            nodeConfig.addTechs([
                use('deps-merge', '__?.all', {depsSources: ['__?.deps.js', '__?.bh.deps.js']}),
                use('files', '__?.all', {levelsTarget: '__?.levels'})
            ]);

            /* Переводы */
            nodeConfig.addTechs([
                use('i18n-merge-keysets', '__?.all', {
                    target: '__?.keysets.all.js',
                    lang: 'all'
                }),
                use('i18n-merge-keysets', '__?.all', {
                    target: '__?.keysets.{lang}.js',
                    lang: '{lang}'
                }),
                use('i18n-lang-js', '__?', {
                    target: '__?.lang.all.js',
                    keysetsTarget: '__?.keysets.all.js',
                    lang: 'all'
                }),
                use('i18n-lang-js', '__?', {lang: '{lang}'})
            ]);

            /* Серверный bemhtml */
            nodeConfig.addTechs([
                use('bemhtml', '__?'),

                use('lmd-config', {
                    target: '__?.temp.lmd.json',
                    modules: ['__?.bemhtml.js'],
                    main: '__?.bemhtml.js',
                    bemhtml: '__?.bemhtml.js',
                    lmd: '_?.bemhtml.js',
                    base: '.lmd/bemhtml.for.examples.lmd.json'
                }),
                use('lmd', {
                    target: '_?.bemhtml.js',
                    modules: ['__?.bemhtml.js'],
                    config: '__?.temp.lmd.json'
                })
            ]);

            /* Полный CSS */
            nodeConfig.addTech(use('css', '_?', {filesTarget: '__?.all.files'}));

            /* Клиентский browser-js = lmd { js + bh.js } */
            nodeConfig.addTechs([
                use('js', '__?'),
                use('bemhtml', '__?.bh'),
                use('file-merge', {
                    target: '__?.merged.js',
                    sources: ['__?.bh.bemhtml.js', '__?.js'],
                    divider: '\n\n/* Client bemhtml.js end */\n/* browser.js beginning */\n\n'
                }),

                use('lmd-config', {
                    target: '__?.bro.lmd.json',
                    modules: ['__?.merged.js'],
                    main: '__?.merged.js',
                    bemhtml: '__?.bh.bemhtml.js',
                    lmd: '_?.browser.js',
                    base: '.lmd/common.lmd.json'
                }),
                use('lmd', {
                    target: '_?.browser.js',
                    modules: ['__?.merged.js'],
                    config: '__?.bro.lmd.json'
                })
            ]);

            /* html странички */
            nodeConfig.addTech(use('html-from-bemjson-i18n', '_?', {
                bemjsonTarget: '?.bemjson.js',
                langAllTarget: '__?.lang.all.js',
                langTarget: '__?.lang.{lang}.js'
            }));

            /* Цели сборки */
            nodeConfig.addTargets([
                '_?.css',
                '_?.browser.js',
                '_?.{lang}.html'
            ]);
        });
    });

    /* Сборка тестов */
    config.mode('test', function (config) {
        config.node('.cov/test', function (nodeConfig) {
            nodeConfig.addTechs([
                use('bemdecl-from-levels', {levels: testLevels, target: '_?.bemdecl.js'}),
                use('levels', '_?.alljs', {levels: allTestJsLevels}),
                use('levels', '_?.alldeps', {levels: allTestDepsLevels}),
                use('deps-old', {
                    bemdeclTarget: '_?.bemdecl.js',
                    levelsTarget: '_?.alldeps.levels',
                    depsTarget: '_?.all.deps.js'
                }),
                use('files', '_?.alljs', {depsTarget: '_?.all.deps.js'}),
                use('files', '_?.alldeps', {depsTarget: '_?.all.deps.js'}),
                use('js', '_?.alljs', {target: '?.all.js'}),
                use('levels', '_?.test', {levels: testLevels}),

                use('css', '_?.alljs', {target: '?.css', depsTarget: '_?.all.deps.js'}),

                use('deps-old', '_?.test', {bemdeclTarget: '_?.bemdecl.js'}),
                use('files', '_?.test'),
                use('test-js', {filesTarget: '_?.test.files', target: '?.test.js'}),
                use('bemhtml', {filesTarget: '_?.alldeps.files'}),
                use('file-provider', {target: '?.bemjson.js'}),
                use('html-from-bemjson')
            ]);

            nodeConfig.addTargets([
                '?.css',
                '?.test.js',
                '?.all.js',
                '?.bemhtml.js',
                '?.html'
            ]);
        });
    });

    config.setEnv({
        PRJ_ROOT: config.resolvePath()
    });
};

function getLangTarget(x) {
    return '?.lang.' + x + '.js';
}

function getClientTechs(levels, templates, cacheName) {
    var res = [
        use('levels', {levels: levels}),

        // Базовые технологии
        use('bemdecl-from-bemjson'),
        use('deps-old'),
        use('files'),

        // bemjson из tt2
        use('tt2bemjson', {
            target: '?.bemjson.js',
            cacheName: cacheName,
            sources: templates
        }),

        // Клиентский bemdecl из client.blocks.bem.tt2
        use('tt2bemjson', {
            target: '?.tt2.bemjson.js',
            cacheName: 'client',
            sources: buildConfig.clientTemplates
        }),
        use('bemdecl-from-bemjson', '?.tt2'),

        // Клиентский bemdecl из deps
        use('bemdecl-from-deps-by-tech', '?.deps', {filesTarget: '?.files'}),

        // Клиентский bemhtml
        use('bemdecl-merge', {
            bemdeclSources: ['?.tt2.bemdecl.js', '?.deps.bemdecl.js'],
            bemdeclTarget: '?.client.bemdecl.js'
        }),
        use('deps-old', {
            bemdeclTarget: '?.client.bemdecl.js',
            depsTarget: '?.client.deps.js'
        }),
        use('files', '?.client', {levelsTarget: '?.levels'}),
        use('bemhtml', {
            devMode: devMode,
            target: '?.bemhtml.js',
            filesTarget: '?.client.files'
        }),

        // Клиентский js
        use('i18n-merge-keysets', {lang: 'all'}),
        use('i18n-merge-keysets', {lang: '{lang}'}),
        use('i18n-lang-js', {lang: 'all'}),
        use('i18n-lang-js', {lang: '{lang}'}),
        use('js-i18n', {lang: '{lang}', target: '?.{lang}.client.js'}),
        use('js'),

        // Клиентский css
        use('css'),
        use('css-ie6', {sourceSuffixes: ['css', 'ie.css', 'ie6.css']}),
        use('css-ie7', {sourceSuffixes: ['css', 'ie.css', 'ie7.css']}),
        use('css-ie8', {sourceSuffixes: ['css', 'ie.css', 'ie8.css']}),
        use('css-ie9', {sourceSuffixes: ['css', 'ie9.css']})
    ].concat(buildConfig.languages.map(getLangJsMergeTech));

    return res;
}

function getLangJsMergeTech(lang) {
    return [require('enb/techs/file-merge'), {
        sources: ['?.bemhtml.js', '?.' + lang + '.client.js'],
        target: '?.' + lang + '.js'
    }];
}

function getClientTargets() {
    return [
        '?.{lang}.js',
        '?.css',
        '?.ie6.css',
        '?.ie7.css',
        '?.ie8.css',
        '?.ie9.css'
    ];
}

function getDevLevels(nodeConfig) {
    var levels = buildConfig.commonLevels;

    /* b-form-button.examples/submit/submit.blocks/
     * блоки, специфичные для примера */
    var path = nodeConfig.resolvePath(),
        localLevel = [path, /[\w\-]+$/i.exec(path) + '.blocks'].join('/');

    fs.existsSync(localLevel) &&
        (levels = levels.concat(localLevel));

    return levels;
}
