'use strict';

var vow = require('vow');
var glob = require('glob');

module.exports = require('enb/lib/build-flow').create()
    .name('bemdecl-from-levels')
    .target('target', '?.bemdecl.js')
    .defineOption('levels', [])
    .builder(function () {
        var levels = this.getOption('levels');

        return vow.all(levels.map(dirlist))
            .then(resolvePaths(levels))
            .then(flatten)
            .then(toBlocks)
            .then(toBemdecl);
    })
    .createTech();

function dirlist(level) {
    var promise = vow.promise();
    glob(level + '/**/*.{bemhtml,js}', function (err, files) {
        promise.fulfill(files);
    });
    return promise;
}

function resolvePaths(levelNames) {
    return function (levels) {
        return levels.map(function (level, index) {
            var name = levelNames[index];
            return level.map(function (dir) {
                return dir.replace(name + '/', '');
            });
        });
    };
}

function flatten(arr) {
    return arr.reduce(function (rs, arr) {
        return rs.concat(arr);
    }, []);
}

function toBlocks(dirs) {
    var result = blocks(dirs)
        .concat(blocksWithElems(dirs))
        .concat(blocksWithMods(dirs))
        .concat(blocksWithElemMods(dirs));

    return result;
}

function blocks(dirs) {
    return dirs
        .map(function (dir) {
            return dir.split('/')[0];
        })
        .reduce(function (result, dir) {
            if (result.indexOf(dir) === -1) {
                result.push(dir);
            }
            return result;
        }, [])
        .map(function (dir) {
            return {block: dir};
        });
}

function filename(dir) {
    return dir.split('/').pop();
}

function matchRegexp(regexp) {
    return function(str) {
        return regexp.test(str);
    };
}

function blocksWithModsAndElems(dirs, regexp) {
    return dirs
        .map(filename)
        .filter(matchRegexp(regexp))
        .map(function (file) {
            var groups = [].slice.apply(regexp.exec(file));
            return blockGenerator.apply(null, groups);
        });
}

function blockGenerator(/*filename, args*/) {
    var args = [].slice.call(arguments, 1);
    if (args.length === 2) {
        return {
            block: args[0],
            elem: args[1]
        };
    } else if (args.length === 3) {
        return {
            block: args[0],
            mod: args[1],
            val: args[2]
        };
    } else if (args.length === 4) {
        return {
            block: args[0],
            elem: args[1],
            mod: args[2],
            val: args[3]
        };
    }
}

function blocksWithElems(dirs) {
    var regexp = /^([^._]+)__([^._]+)\.(?:js|bemhtml)$/;
    return blocksWithModsAndElems(dirs, regexp);
}

function blocksWithMods(dirs) {
    var regexp = /^([^._]+)_([^._]+)_([^._]+)\.(?:js|bemhtml)$/;
    return blocksWithModsAndElems(dirs, regexp);
}

function blocksWithElemMods(dirs) {
    var regexp = /^([^._]+)__([^._]+)_([^._]+)_([^._]+)\.(?:js|bemhtml)$/;
    return blocksWithModsAndElems(dirs, regexp);
}

function toBemdecl(data) {
    return 'exports.deps = ' + JSON.stringify(data, null, 4) + ';';
}
