/**
 * lmd-config
 * =====
 *
 * Создает конфиг файл для lmd
 *
 * **Опции**
 * * *String* **base** - Ссылка на базовый конфиг. Необязательная. Указывается путь относительно корневой папки проекта.
 * * *Array* **modules** - Список модулей. По умолчанию - `['?.js']`.
 * * *String* **lmd** - Результирующий таргет технологии lmd, формирует поле `output` в конфиге.
 * * *String* **bemhtml** - Маска для bemhtml таргетов. По умолчанию - `?.bemhtml.js`.
 * * *String* **target** - Результирующий конфиг. По умолчанию - `?.lmd.js`.
 *
 * **Пример**
 *
 * ```javascript
 * nodeConfig.addTech('./techs/lmd-config')
 * ```
 */
'use strict';

var path = require('path');
var fs = require('vow-fs');

module.exports = require('enb/lib/build-flow').create()
    .name('lmd-config')
    .target('target', '?.lmd.json')
    .methods({
        /**
         * @param {String} [pathname]
         * @return {String}
         */
        resolveFromRoot: function (pathname) {
            return path.join(this.node.getRootDir(), pathname);
        },
        /**
         * @param {Object} [to]
         * @param {Array} [from]
         */
        extendModules: function (to, from) {
            var mask = this.node.unmaskTargetName(this.getOption('bemhtml', '?.bemhtml.js'));

            from.forEach(function (file) {
                file = path.basename(file);
                var desc;

                if (file === mask) {
                    desc = {
                        path: file,
                        exports: '(this.BEMHTML = BEMHTML)'
                    };
                } else {
                    desc = file;
                }

                to[file] = desc;
            });
        }
    })
    .useSourceListFilenames('modules', ['?.js']) // В аргументы билдера придут абсолютные пути к файлам.
    .needRebuild(function (cache) {
        var node = this.node;

        var files = this.getOption('modules', ['?.js'])
            .map(function (file) {
                return node.resolvePath(node.unmaskTargetName(file));
            });

        var base = this.getOption('base');
        base && files.push(path.join(node.getRootDir(), base));

        return cache.needRebuildFileList('lmd-config', files);
    })
    .saveCache(function (cache) {
        var node = this.node;

        var files = this.getOption('modules', ['?.js'])
            .map(function (file) {
                return node.resolvePath(node.unmaskTargetName(file));
            });

        var base = this.getOption('base');
        base && files.push(path.join(node.getRootDir(), base));

        cache.cacheFileList('lmd-config', files);
    })
    .builder(function (files) {
        var _this = this;

        var base = _this.getOption('base');
        var lmd = _this.getOption('lmd', '?.lmd.js');
        var main = _this.getOption('main', '?.js');

        var node = _this.node;

        if (base) {
            var basePath = _this.resolveFromRoot(base);
            return fs.read(basePath).then(function (config) {
                config = JSON.parse(config);

                var baseRoot = config.root || config.path;
                baseRoot = path.join(path.dirname(basePath), baseRoot);

                config.root = node.resolvePath('');
                delete config.path;

                // Переписываем пути наследуемых модулей
                config.modules || (config.modules = {});
                Object.keys(config.modules).forEach(function (key) {
                    var value = config.modules[key];

                    if (typeof value === 'string' && value.charAt(0) !== '@') {
                        config.modules[key] = node.relativePath(path.join(baseRoot, value));
                    } else if (config.modules[key].path) {
                        value.path = node.relativePath(path.join(baseRoot, value.path));
                    }
                });

                // Добавляем свои модули
                _this.extendModules(config.modules, files);

                config.main = node.unmaskTargetName(main);
                config.output = node.unmaskTargetName(lmd);

                return JSON.stringify(config, null, 4);
            });
        } else {
            var config = {
                root: node.resolvePath(''),
                name: node.getTargetName(),
                modules: {},
                main: node.unmaskTargetName(main),
                output: node.unmaskTargetName(lmd)
            };

            _this.extendModules(config.modules, files);

            return JSON.stringify(config, null, 4);
        }
    })
    .createTech();
