'use strict';

var vow = require('vow');
var vowFs = require('vow-fs');
var fs = require('fs');
var FileList = require('enb/lib/file-list');

module.exports = require('enb/lib/build-flow').create()
    .name('tt2-to-bemjson')
    .target('target', '?.bemjson')
    .defineRequiredOption('sources')
    .defineOption('suffixes', 'bem.tt2')
    .defineOption('cacheName', 'all')
    .needRebuild(function (cache) {
        this._tt2FileList = getFileList(this._sources);
        return cache.needRebuildFileList(
            'tt2bemjson__' + this._cacheName,
            this._tt2FileList.items
        );
    })
    .saveCache(function (cache) {
        cache.cacheFileList(
            'tt2bemjson__' + this._cacheName,
            this._tt2FileList.items
        );
    })
    .builder(function () {
        return vow.all(this._tt2FileList.getBySuffix(this._suffixes).map(readAndTransform(tt2bemjson))).then(join);
    })
    .createTech();

function getFileList(sources) {
    var fileList = new FileList();

    sources = Array.isArray(sources) ? sources : [sources];

    sources.forEach(function (source) {
        var stat = fs.statSync(source);

        if (stat.isDirectory()) {
            fileList.loadFromDirSync(source, true);
        } else if (stat.isFile()) {
            var filename = source.split('/');
            filename = filename[filename.length - 1];

            fileList.addFiles([{
                name: filename,
                fullname: source,
                suffix: getSuffix(filename),
                mtime: stat.mtime.getTime()
            }]);
        }
    });
    return fileList;
}

function join(fragments) {
    return '([' + fragments.join(',\n') + '])';
}

function readAndTransform(transformer) {
    transformer = transformer || function (x) { return x; };
    return function (file) {
        return vowFs.read(file.fullname, 'utf8').then(transformer);
    };
}

function tt2bemjson(s) {
    s = s.replace(/\s*\[%(.|\n)*?%\]\s*/g, '');

    if (/block\s*:\s*'b-page'/.test(s)) {
        return s;
    }

    return '({block : \'b-page\', content : [' + s +  ']})';
}

function getSuffix(filename) {
    return filename.split('.').slice(1).join('.');
}
