'use strict';

var path = require('path');

var buildConfig = require('./buildConfig');
var allLevels = buildConfig.allLevels;

var MINIMIZE = buildConfig.minimize;

function append(suffix) {
    return function (s) {
        return path.join(s, suffix);
    };
}

function includeOnlyByMasks(masks) {
    function prepend(prefix) {
        return function (x) { return prefix + x; };
    }

    function include(items) {
        return items.map(prepend('--include='));
    }

    function exclude(items) {
        return items.map(prepend('--exclude='));
    }

    return exclude(['*/examples/*', '*/*.examples/*', '*.i18n'])
        .concat(include(['*/'].concat(masks)))
        .concat(exclude(['*']));
}

var addTrailingSlash = append('/');

var rsyncArgs = ['--quiet', '--copy-links', '--copy-dirlinks'];

var outPathCss = buildConfig.outPath.css;
var outPathJs = buildConfig.outPath.js;

module.exports = function (grunt) {
    require('time-grunt')(grunt);

    grunt.initConfig({
        pkg: grunt.file.readJSON('package.json'),

        clean: {
            enb: {
                src: ['.tmp/']
            },

            build: {
                src: ['build/']
            },

            test: {
                src: ['.cov/']
            }
        },

        mkdir: {
            all: {
                options: {
                    create: [
                        '.tmp/priv',
                        '.tmp/common',
                        '.tmp/promo',
                        '.tmp/lmd',
                        'build/data/css',
                        'build/data/js',
                        'build/data/i/blocks'
                    ]
                }
            }
        },

        enb: {
            all: {
                env: {
                    XJST_ASYNCIFY: 'yes'
                },
                targets: ['.tmp/priv', '.tmp/common', '.tmp/promo']
            },
            options: {
                noLog: false
            },
            test: {
                env: {
                    YENV: 'test'
                },
                targets: ['.cov/test']
            }
        },

        lmd: {
            priv: {build: 'priv'},

            //common
            pub_en: {build: 'pub_en'},
            pub_ru: {build: 'pub_ru'},
            pub_tr: {build: 'pub_tr'},

            //promo
            pub_promo_en: {build: 'pub_promo_en'},
            pub_promo_ru: {build: 'pub_promo_ru'},
            pub_promo_tr: {build: 'pub_promo_tr'}
        },

        borschik: {
            options: {
                minimize: MINIMIZE
            },

            // common
            css: {src: '.tmp/common/common.css', dest: 'build/data/css/common.css'},

            css_ie:  {src: '.tmp/common/common.ie.css',  dest: 'build/data/css/common.ie.css'},
            css_ie6: {src: '.tmp/common/common.ie6.css', dest: 'build/data/css/common.ie6.css'},
            css_ie7: {src: '.tmp/common/common.ie7.css', dest: 'build/data/css/common.ie7.css'},
            css_ie8: {src: '.tmp/common/common.ie8.css', dest: 'build/data/css/common.ie8.css'},
            css_ie9: {src: '.tmp/common/common.ie9.css', dest: 'build/data/css/common.ie9.css'},

            js_en: {src: '.tmp/lmd/common.en.js', dest: 'build/data/js/_common.en.js'},
            js_ru: {src: '.tmp/lmd/common.ru.js', dest: 'build/data/js/_common.ru.js'},
            js_tr: {src: '.tmp/lmd/common.tr.js', dest: 'build/data/js/_common.tr.js'},

            // promo
            promo_css: {src: '.tmp/promo/promo.css', dest: 'build/data/css/promo.css'},

            promo_css_ie:  {src: '.tmp/promo/promo.ie.css',  dest: 'build/data/css/promo.ie.css'},
            promo_css_ie6: {src: '.tmp/promo/promo.ie6.css', dest: 'build/data/css/promo.ie6.css'},
            promo_css_ie7: {src: '.tmp/promo/promo.ie7.css', dest: 'build/data/css/promo.ie7.css'},
            promo_css_ie8: {src: '.tmp/promo/promo.ie8.css', dest: 'build/data/css/promo.ie8.css'},
            promo_css_ie9: {src: '.tmp/promo/promo.ie9.css', dest: 'build/data/css/promo.ie9.css'},

            promo_js_en: {src: '.tmp/lmd/promo.en.js', dest: 'build/data/js/_promo.en.js'},
            promo_js_ru: {src: '.tmp/lmd/promo.ru.js', dest: 'build/data/js/_promo.ru.js'},
            promo_js_tr: {src: '.tmp/lmd/promo.tr.js', dest: 'build/data/js/_promo.tr.js'}
        },

        copy: {
            priv: {
                files: [{
                    src: '.tmp/lmd/priv.all.priv.js',
                    dest: 'build/common.priv.js'
                }]
            },
            out_priv: {
                files: [{
                    src: 'build/common.priv.js',
                    dest: path.join(buildConfig.outPath.priv, 'common.priv.js')
                }]
            },
            out_css: {
                files: [
                    {src: 'build/data/css/common.css',     dest: path.join(outPathCss, '_common.css')},
                    {src: 'build/data/css/common.ie.css',  dest: path.join(outPathCss, '_common.ie.css')},
                    {src: 'build/data/css/common.ie6.css', dest: path.join(outPathCss, '_common.ie6.css')},
                    {src: 'build/data/css/common.ie7.css', dest: path.join(outPathCss, '_common.ie7.css')},
                    {src: 'build/data/css/common.ie8.css', dest: path.join(outPathCss, '_common.ie8.css')},
                    {src: 'build/data/css/common.ie9.css', dest: path.join(outPathCss, '_common.ie9.css')}
                ]
            },
            out_js: {
                files: [
                    {src: 'build/data/js/_common.en.js', dest: path.join(outPathJs, '_common.en.js')},
                    {src: 'build/data/js/_common.ru.js', dest: path.join(outPathJs, '_common.ru.js')},
                    {src: 'build/data/js/_common.tr.js', dest: path.join(outPathJs, '_common.tr.js')}
                ]
            },
            out_promo_css: {
                files: [
                    {src: 'build/data/css/promo.css',     dest: path.join(outPathCss, '_promo.css')},
                    {src: 'build/data/css/promo.ie.css',  dest: path.join(outPathCss, '_promo.ie.css')},
                    {src: 'build/data/css/promo.ie6.css', dest: path.join(outPathCss, '_promo.ie6.css')},
                    {src: 'build/data/css/promo.ie7.css', dest: path.join(outPathCss, '_promo.ie7.css')},
                    {src: 'build/data/css/promo.ie8.css', dest: path.join(outPathCss, '_promo.ie8.css')},
                    {src: 'build/data/css/promo.ie9.css', dest: path.join(outPathCss, '_promo.ie9.css')}
                ]
            },
            out_promo_js: {
                files: [
                    {src: 'build/data/js/_promo.en.js', dest: path.join(outPathJs, '_promo.en.js')},
                    {src: 'build/data/js/_promo.ru.js', dest: path.join(outPathJs, '_promo.ru.js')},
                    {src: 'build/data/js/_promo.tr.js', dest: path.join(outPathJs, '_promo.tr.js')}
                ]
            }
        },

        rsync: {
            options: {
                args: rsyncArgs,
                recursive: true
            },
            img: {
                options: {
                    args: rsyncArgs.concat(includeOnlyByMasks([
                        '*.gif',
                        '*.png',
                        '*.jpg',
                        '*.jpeg',
                        '*.ttf',
                        '*.woff',
                        '*.eot',
                        '*.swf',
                        '*.svg'
                    ])),
                    src: allLevels.map(addTrailingSlash),
                    dest: buildConfig.outPath.img,
                    syncDest: true
                }
            }
        },

        concat: {
            dist: {
                src: 'test/tests/*.js',
                dest: '.cov/test/test.concat.js'
            }
        },

        watch: {
            test: {
                files: [
                    'src/**/*.js',
                    'models/**/*.js'
                    // 'test/**/*.js'
                ],
                tasks: ['test']
            },
            build: {
                files: [
                    'src/**/*.js',
                    'blocks.lego/**/*.bemhtml',
                    'blocks.lego/**/*.js',
                    'blocks.lego/**/*.css',
                    'blocks.pi/**/*.bemhtml',
                    'blocks.pi/**/*.js',
                    'blocks.pi/**/*.css',
                    'blocks.promo/**/*.bemhtml',
                    'blocks.promo/**/*.js',
                    'blocks.promo/**/*.css',
                    '../blocks/**/*.bemhtml',
                    '../blocks/**/*.js',
                    '../blocks/**/*.css',
                    '../templates/**/*.bem.tt2'
                ],
                tasks: ['build']
            }
        },

        concurrent: {
            lmd: [
                'lmd:priv',
                'lmd:pub_en', 'lmd:pub_ru', 'lmd:pub_tr',
                'lmd:pub_promo_en', 'lmd:pub_promo_ru', 'lmd:pub_promo_tr'
            ],
            borschik: [
                'borschik:css',
                'borschik:css_ie', 'borschik:css_ie6', 'borschik:css_ie7',
                'borschik:css_ie8', 'borschik:css_ie9',
                'borschik:js_en', 'borschik:js_ru', 'borschik:js_tr',

                'borschik:promo_css',
                'borschik:promo_css_ie', 'borschik:promo_css_ie6', 'borschik:promo_css_ie7',
                'borschik:promo_css_ie8', 'borschik:promo_css_ie9',
                'borschik:promo_js_en', 'borschik:promo_js_ru', 'borschik:promo_js_tr'
            ]
        }
    });

    grunt.loadNpmTasks('grunt-concurrent');
    grunt.loadNpmTasks('grunt-enb');
    grunt.loadNpmTasks('grunt-mkdir');
    grunt.loadNpmTasks('grunt-rsync');
    grunt.loadNpmTasks('grunt-lmd');
    grunt.loadNpmTasks('grunt-borschik');
    grunt.loadNpmTasks('grunt-contrib-clean');
    grunt.loadNpmTasks('grunt-contrib-copy');
    grunt.loadNpmTasks('grunt-contrib-concat');
    grunt.loadNpmTasks('grunt-contrib-watch');

    // Tasks

    grunt.registerTask('nop', function () {});

    grunt.registerTask('test', []);

    grunt.registerTask('copy-out', [
        'copy:out_js',
        'copy:out_promo_js',
        'copy:out_css',
        'copy:out_promo_css',
        'copy:out_priv'
    ]);

    grunt.registerTask('build', [
        'mkdir',
        'enb:all',
        'concurrent:lmd',
        'concurrent:borschik',
        'copy:priv',
        'test',
        'copy-out',
        'rsync:img'
    ]);

    grunt.registerTask('default', 'build');
};
