(function (BEM) {
    'use strict';
    /**
     * Блок `header2` — островная шапка в белом дизайне.
     * API блока позволяет управлять выдвигающимися панелями. В любой момент времени может быть открыта только одна панель.
     */
    BEM.DOM.decl('header2', {

        /**
         * Генерируется, когда изменяется текущая открытая панель
         * (открывается новая, закрывается открытая или пероисходит переключение с одной панели на другую).
         *
         * @event header2#panel-switch
         * @param {Object}      data          Данные события.
         * @param {String|null} data.closing  Имя закрывающейся панели или null если до события не было открытой панели.
         * @param {String|null} data.opening  Имя открывающейся панели или null если новая панель не открывается.
         */

        /**
         * @private
         */
        onSetMod: {
            js: {
                inited: function() {
                    this._openedPanel = null;

                    this.on('panel-switch', this._onPanelSwitch, this);
                    this.on('action-change', this._onActionChange, this);

                    this._ajaxForm = this.findBlockInside('a-form-ajax');
                }
            }
        },

        /**
         * @private
         */
        _setOpenedPanel: function(newPanel) {
            var currPanel = this._openedPanel;
            if(currPanel !== newPanel) {
                this._openedPanel = newPanel;
                this.trigger('panel-switch', {closing: currPanel, opening: newPanel});
            }
        },

        /**
         * Получить имя текущей открытой панели.
         *
         * @public
         * @returns {String|null} Имя открытой панели или null если все панели закрыты.
         */
        getOpenedPanel: function() {
            return this._openedPanel;
        },

        /**
         * Открыть панель. Если уже открыта другая панель, она будет закрыта.
         *
         * @public
         * @param {String} name Имя открываемой панели.
         * @returns {BEM.DOM} this
         */
        openPanel: function(name) {
            this._setOpenedPanel(name);
            return this;
        },

        /**
         * Закрыть панель.
         *
         * @public
         * @param {String} [name] Имя закрываемой панели. Если имя не указано, будет закрыта текущая открытая панель.
         * @returns {BEM.DOM} this
         */
        closePanel: function(name) {
            if(!name || name === this._openedPanel) {
                this._setOpenedPanel(null);
            }
            return this;
        },

        /**
         * Переключить панель в заданное состояние.
         *
         * @public
         * @param {String} name Имя панели.
         * @param {Boolean} [state] Состояние в которое переключается панель.
         * Если параметр не указан, состояние панели сменится на противоположное.
         * @returns {BEM.DOM} this
         */
        togglePanel: function(name, state) {
            if(name === this._openedPanel && !state) {
                this._setOpenedPanel(null);
            } else if(name !== this._openedPanel && state || arguments.length < 2) {
                this._setOpenedPanel(name);
            }
            return this;
        },

        /**
         * @private
         */
        _onPanelSwitch: function (e, data) {
            var SETTINGS_PANEL = 'panel-settings';

            var isOpening = data.opening === SETTINGS_PANEL;
            var isClosing = data.closing === SETTINGS_PANEL;

            if (isOpening || isClosing) {
                this.toggleMod(this.elem(SETTINGS_PANEL), 'opened', 'yes', '', isOpening);

                this['_' + (isOpening ? 'open': 'close') + 'SettingsPanel']();
            }
        },

        /**
         * @private
         * При открытии панели загружаем данные формы аяксом
         * @param {object} e Объект события(Event Object)
         * @param {object} data BEM JSON
         */
        _openSettingsPanel: function () {
            if (this._ajaxForm) {
                this._ajaxForm.hasMod('loaded', 'no') &&
                    this._ajaxForm.load();
            }

            var page = $('.b-page').bem('b-page');
            var popups = page.findBlocksInside('b-popupa');
            popups.forEach(function (popup) {
                popup.isShowed() && popup.hide();
            });
        },

        /**
         * @private
         * При закрытии панели очищаем форму
         */
        _closeSettingsPanel: function () {
            this._ajaxForm && this._ajaxForm.hasMod('loaded', 'yes') && this._ajaxForm.clear();
        },

        /**
         * @private
         * @param {object} e Объект события(Event Object)
         * @param {object} data BEM JSON
         */
        _onActionChange: function (e, data) {
            if (data.type === 'settings') {
                this._paranja || (this._paranja = this.findBlockInside('paranja'));

                this._paranja[data.checked ? 'open' : 'close']();
            }
        }

    });
})(BEM);
