(function (BEM) {
    'use strict';

    /**
     * Модули.
     */
    require('history');
    var _ = require('lodash');

    /**
     * Кэш текущего состояния.
     *
     * @type {Object}
     */
    var currentState = {};

    /**
     * Состояние модели по умолчанию.
     *
     * @type {Object}
     */
    var defaultValues = {
        kit: 0,
        lang: 'ru',
        tab: 1
    };

    /**
     * @type {Boolean}
     */
    var initial = true;

    /**
     * Ссылки на лицензионное соглашение.
     *
     * @type {Object}
     */
    var links = {
        ru: {
            bro: 'http://legal.yandex.ru/browser_agreement/',
            legal: 'http://soft.yandex.ru/distribution/',
            software: 'http://legal.yandex.ru/desktop_software_agreement/'
        },
        tr: {
            bro: 'http://legal.yandex.com.tr/browser_agreement/',
            legal: 'http://soft.yandex.com.tr/distribution/',
            software: 'http://legal.yandex.com.tr/elements_agreement/'
        }
    };

    /**
     * Флаг, препятствующий обновлению истории, когда происходит событие popstate.
     *
     * @type {Boolean}
     */
    var pushGuard = false;

    /**
     * Поля, на основе которых формируется кэш текущего состояния и осуществляется сравнение кэша.
     * Порядок полей в массиве очень важен.
     *
     * @type {Array}
     */
    var valuableFields = [
        'kit',
        'lang',
        'winState',
        'winType',
        'winTypes',
        'tab',
        'win',
        'page'
    ];

    /**
     * Минимальный интервал обновления истории.
     *
     * @type {Number}
     */
    var HISTORY_CHANGE_TIMEOUT = 1000;

    /**
     * Минимальный интервал рендера страницы.
     *
     * @type {Number}
     */
    var RENDER_INTERVAL = 50;

    BEM.DOM.decl('a-instruction', {
        onSetMod: {
            js: function () {
                /* Радиокнопки в интерактивном окне. */
                this.findBlocksOn('radio', 'a-control').forEach(function (radio) {
                    radio.on('change', this._switchState, this);
                }, this);

                var model = this._getModel();

                model.on('field-change', $.debounce(this._render, RENDER_INTERVAL, false, this));

                model.on('change', $.debounce(this._pushState, HISTORY_CHANGE_TIMEOUT, false, this));

                this._prepareState();

                this.bindToWin('popstate', this._onPopState, this);
                this._onPopState();
            }
        },

        onElemSetMod: {
            page: {
                shown: {
                    yes: function () {
                        this.delMod(this.findElem('page', 'shown', 'yes'), 'shown');
                    }
                }
            },

            types: { // Тип инсталятора (Радио кнопки)
                disabled: function (elem, modName, modVal) {
                    var control = this._getTypesControl(3);
                    control && control.setMod(modName, modVal);
                },

                shown: {
                    yes: function () {
                        this.delMod(this.findElem('types', 'shown', 'yes'), 'shown');
                    }
                }
            },

            win: { // Интерактивное окно
                shown: {
                    yes: function () {
                        var elem = this.findElem('win', 'shown', 'yes');
                        elem.__bemElemName = 'win';

                        this.delMod(elem, 'shown');
                    }
                }
            }
        },

        /**
         * Сравнивает поля второго объекта с первым и возвращает список полей,
         * которые не соответствуют друг другу.
         * Дополнительно отсеивает второстепенные поля.
         *
         * @param  {Object} current
         * @param  {Object} model
         * @return {Array}
         */
        _getDiffs: function (current, model) {
            return valuableFields.filter(function (key) {
                return key in model &&
                    !_.isEqual(current[key], model[key]);
            });
        },

        /**
         * Отдает ссылку на главную модельку.
         *
         * @return {BEM.MODEL}
         */
        _getModel: function () {
            return this._model ||
                (this._model = this.findBlockOn(this.domElem, 'i-gum').model);
        },

        /**
         * Возвращает ссылку на активный контрол выбора типа окна.
         *
         * @param  {Number}  num
         * @return {BEM.DOM}     a-control.
         */
        _getTypesControl: function (num) {
            num || (num = this._getModel().get('winTypes'));
            return this.findBlockInside(this.elem('types', 'kit', alphabetic(num)), 'a-control');
        },

        /**
         * Формирует урл с учетом параметров.
         *
         * @param  {String} attrs
         * @return {String}
         */
        _getUrl: function (attrs) {
            var a = document.createElement('a');
            a.href = location;

            return a.pathname + (attrs ? '?' + attrs : '');
        },

        /**
         * Обработчик перемещения пользователя по истории.
         */
        _onPopState: function () {
            var fields = this._getDiffs(currentState, history.state);

            if (fields.length === 0) {
                return false;
            }

            pushGuard = true;

            var model = this._getModel();
            var state = _.pick(history.state, _.difference(fields, ['winState']));
            var winState = history.state.winState;

            model.update(state);

            // Обновляем состояние чекбоксов
            this._updateWinListWithWinState(winState);
        },

        /**
         * Обновляет историю при инициализации.
         * Получает данные из адреса ссылки.
         */
        _prepareState: function () {
            if (history.state) {
                return false;
            }

            var pattern = /\?([^\?]+)$/;
            var attrs = pattern.exec(location.href);
            var state;

            if (attrs && attrs[1]) {
                state = attrs[1]
                    .split('&')
                    .reduce(function (hash, pair) {
                        pair = pair.split('=');
                        hash[pair[0]] = decodeURIComponent(pair[1]);

                        return hash;
                    }, {});

                state = _.pick(state, valuableFields);
            } else {
                state = defaultValues;
            }

            history.replaceState(state, document.title);
            this._updateCurrentState();
        },

        /**
         * Обновляет историю.
         */
        _pushState: function () {
            if (pushGuard) {
                pushGuard = !pushGuard; // Было загружено состояние из истории.

                return false;
            }

            var attrs = _.chain(currentState)
                .pick(function (val, key) {
                    var fields = [
                        'kit',
                        'lang',
                        'winState',
                        'winType'
                    ];

                    return fields.indexOf(key) > -1 && val && defaultValues[key] !== val;
                })
                .map(function (val, key) {
                    return [key, encodeURIComponent(val)].join('=');
                })
                .value()
                .join('&');

            history.pushState(currentState, document.title, this._getUrl(attrs));
        },

        /**
         * Визуализирует страницу.
         */
        _render: function () {
            var args = _.toArray(arguments);
            var model = this._getModel();
            var fields = this._getDiffs(currentState, model.toJSON());

            if (fields.length === 0) {
                return false;
            }

            this._updateCurrentState();

            fields.forEach(function (key) {
                var method = '_render' + key.replace(/^./, function (m) {
                    return m.toUpperCase();
                });

                var currentArgs = args.slice();
                currentArgs.unshift(model.get(key));

                this[method].apply(this, currentArgs);
            }, this);

        },

        /**
         * В зависимости от выбранного комплекта рисует или прячет табы.
         *
         * @param  {Number} kitIndex Комплект
         */
        _renderKit: function (kitIndex) {
            if (kitIndex === 3) {
                this.setMod(this.elem('tabs'), 'shown', 'yes');
            } else {
                this.delMod(this.elem('tabs'), 'shown');
            }
        },

        /**
         * В зависимости от выбранного языка поменяет отображаемое окно,
         * а вместе с ним и список чекбоксов в поле модели `winList`.
         */
        _renderLang: function () {
            var model = this._getModel();
            var winState = model.get('winState');
            var winType = model.get('winType');

            this._renderWin(model.get('win'));
            this._updateWinList(winType);
            this._updateWinListWithWinState(winState);
        },

        /**
         * Показывает выбранную страницу.
         * Возможные варианты: front | rear.
         *
         * @param {String} val
         */
        _renderPage: function (val) {
            this.setMod(this.elem('page', 'type', val), 'shown', 'yes');

            val === 'rear' && this.afterCurrentEvent(function () {
                this._updateITHeight();
            }, this);
        },

        _renderTab: function (val) {
            if (val) {
                this._updateHeader(val);
                this.findBlockOn('tabs', 'a-control').val(String(val));
            }

            this._getModel().get('winTypes') === 3 &&
                this.setMod(this.findElem('types', 'shown', 'yes'), 'disabled', val === 2 ? 'yes' : '');
        },

        /**
         * Переключает окно инсталлятора в соответствии с переданным параметром.
         * Также обновляет текст окна инсталлятора.
         * Если значение не передано, то прячет все окна.
         *
         * @param {String} val
         */
        _renderWin: function (val) {
            if (val) {
                var model = this._getModel();
                var elem = this.elem(val, 'lang', model.get('lang'));
                /* Небольшой хак, чтобы иметь возможность работать со вторым элементом на том же узле. */
                elem.__bemElemName = 'win';

                this.setMod(elem, 'shown', 'yes');
                this._updateInstallerText();
            } else {
                this.delMod(this.elem('win'), 'shown');
            }
        },

        _renderWinState: function () {
            this._updateCommandLine();
        },

        _renderWinType: function (val) {
            var winTypes = this._getModel().get('winTypes');
            winTypes && this._getTypesControl().val(val);

            val && this._updateWinList(val);

            if (initial) {
                history.state.winState && this._updateWinListWithWinState(history.state.winState);

                initial = false;
            }

            this._updateCommandLine();
        },

        _renderWinTypes: function (val) {
            var header = val === 3 ? this._getModel().get('tab') : 0;
            this._updateHeader(header);

            if (val) {
                this.setMod(this.elem('types', 'kit', alphabetic(val)), 'shown', 'yes');
            } else {
                this.delMod(this.elem('types'), 'shown');
            }
        },

        /**
         * Биндер для радиокнопок в интерактивном окне.
         *
         * @param  {Event} e
         */
        _switchState: function (e) {
            var win = e.block.domElem.closest('.a-instruction__win');
            var checkboxes = this.findBlocksInside(win, {block: 'a-control', modName: 'lego', modVal: 'checkbox'});

            if (e.block.val() === 'user') {
                checkboxes.forEach(function (control) {
                    control.delMod('disabled');
                });
            } else {
                this._getModel().get('winList').forEach(function (model) {
                    model.get('checked') || model.set('checked', true);
                });

                checkboxes.forEach(function (control) {
                    control.setMod('disabled', 'yes');
                });
            }
        },

        /**
         * Меняет содержимое окна "Код командной строки".
         * TODO: Вынести в модель при следующем хотфиксе.
         */
        _updateCommandLine: function () {
            var field = this._commandLine ||
                (this._commandLine = this.findBlockOn('command-line', 'a-control'));

            var commands = [];
            this._getModel().get('winList').forEach(function (model) {
                var val = model.get('value');
                model.get('checked') && val && commands.push(model.get('value'));
            });

            if (commands.length) {
                var ilightIndex = commands.indexOf('ILIGHT=1');

                if (ilightIndex > -1) {
                    commands.splice(ilightIndex, 1);
                } else {
                    commands.push('ILIGHT=1');
                }

                /* jshint maxlen: false */
                if (commands.length === 0) {
                    field.val('downloader.exe --partner <distributive name> --distr /passive /msicl " "');
                } else {
                    field.val('downloader.exe --partner <distributive name> --distr /passive /msicl "' + commands.join(' ') + '"');
                }
                /* jshint maxlen: 120 */
            } else {
                field.val('');
            }
        },

        /**
         * Обновление текущего состояния. Запускается в конце рендера страницы.
         */
        _updateCurrentState: function () {
            currentState = _.pick(this._getModel().toJSON(), valuableFields);
        },

        /**
         * Меняет содержимое окна "Текст инсталятора".
         */
        _updateInstallerText: function () {
            var field = this._installerText ||
                (this._installerText = this.findBlockInside('installer-text', 'a-control'));

            var lang = this._getModel().get('lang');
            var win = this._getModel().get('win');

            /* jshint maxlen: false */
            var texts = {
                ru: {
                    // Комплект 2 / Промежуточный экран
                    'abc-ya-bro': 'Возможности установки\nВыберите тип установки программы\n\nПолная установка (рекомендуется)\nУстановить ABC и рекомендуемые ABC программы и настройки Яндекса\n\nНастройка параметров\n\nЗагрузить и установить Яндекс.Браузер\n\nЗагрузить и установить Элементы Яндекса\n\nУстановить настройки поиска и сделать Яндекс домашней страницей\n\nЗагрузить и установить Менеджер Браузеров\n\nПодробно (%legal%)\n\nУстанавливая рекомендуемое ABC программное обеспечение, вы соглашаетесь с лицензионным соглашением настольного ПО Яндекса (%software%) и Яндекс.Браузера (%bro%).',
                    // Комплект 1,3 / Промежуточный экран
                    'abc-ya': 'Возможности установки\nВыберите тип установки программы\n\nПолная установка (рекомендуется)\nУстановить ABC и рекомендуемые ABC программы и настройки Яндекса\n\nНастройка параметров\n\nЗагрузить и установить Элементы Яндекса\n\nУстановить настройки поиска и сделать Яндекс домашней страницей Яндекса\n\nЗагрузить и установить Менеджер Браузеров\n\nПодробно (%legal%)\n\nУстанавливая рекомендуемое ABC программное обеспечение, вы соглашаетесь с лицензионным соглашением настольного ПО Яндекса (%software%).',
                    // Комплект 2 / Логотип слева
                    'logo-left-bro': 'Программы Яндекса\nУстанавливая программы Яндекса, вы помогаете ABC Software\nABC рекомендует установить программы и настройки Яндекса:\n\nЗагрузить и установить Яндекс.Браузер\n\nЗагрузить и установить Элементы Яндекса\n\nУстановить настройки поиска и сделать Яндекс домашней страницей\n\nЗагрузить и установить Менеджер Браузеров\n\nПодробно (%legal%)\n\nУстанавливая рекомендуемое ABC программное обеспечение, вы соглашаетесь с лицензионным соглашением настольного ПО Яндекса (%software%) и Яндекс.Браузера (%bro%).',
                    // Комплект 1,3 / Логотип слева
                    'logo-left': 'Программы Яндекса\nУстанавливая программы Яндекса, вы помогаете ABC Software\nABC рекомендует установить программы и настройки Яндекса:\n\nЗагрузить и установить Элементы Яндекса\n\nУстановить настройки поиска и сделать Яндекс домашней страницей\n\nЗагрузить и установить Менеджер Браузеров\n\nПодробно (%legal%)\n\nУстанавливая рекомендуемое ABC программное обеспечение, вы соглашаетесь с лицензионным соглашением настольного ПО Яндекса (%software%).',
                    // Комплект 2 / Логотип справа
                    'logo-right-bro': 'Программы Яндекса\nУстанавливая программы Яндекса, вы помогаете ABC Software\nABC рекомендует установить программы и настройки Яндекса:\n\nЗагрузить и установить Яндекс.Браузер\n\nЗагрузить и установить Элементы Яндекса\n\nУстановить настройки поиска и сделать Яндекс домашней страницей\n\nЗагрузить и установить Менеджер Браузеров\n\nПодробно (%legal%)\n\nУстанавливая рекомендуемое ABC программное обеспечение, вы соглашаетесь с лицензионным соглашением настольного ПО Яндекса (%software%) и Яндекс.Браузера (%bro%).',
                    // Комплект 1,3 / Логотип справа
                    'logo-right': 'Программы Яндекса\nУстанавливая программы Яндекса, вы помогаете ABC Software\nABC рекомендует установить программы и настройки Яндекса:\n\nЗагрузить и установить Элементы Яндекса\n\nУстановить настройки поиска и сделать Яндекс домашней страницей\n\nЗагрузить и установить Менеджер Браузеров\n\nПодробно (%legal%)\n\nУстанавливая рекомендуемое ABC программное обеспечение, вы соглашаетесь с лицензионным соглашением настольного ПО Яндекса (%software%).',
                    // Комплект 2 / Последний экран
                    'soft-abc-bro': 'Установка ABC завершена\nНажмите кнопку «Готово» для выхода из программы установки.\n\nЗапустить ABC\n\nABC рекомендует установить программы и настройки Яндекса:\n\nЗагрузить и установить Яндекс.Браузер\n\nЗагрузить и установить Элементы Яндекса\n\nУстановить настройки поиска и сделать Яндекс домашней страницей\n\nЗагрузить и установить Менеджер Браузеров\n\nПодробно (%legal%)\n\nУстанавливая рекомендуемое ABC программное обеспечение, вы соглашаетесь с лицензионным соглашением настольного ПО Яндекса (%software%) и Яндекс.Браузера (%bro%).',
                    // Комплект 1 / Последний экран
                    'soft-abc': 'Установка ABC завершена\nНажмите кнопку «Готово» для выхода из программы установки.\n\nЗапустить ABC\n\nABC рекомендует установить программы и настройки Яндекса:\n\nЗагрузить и установить Элементы Яндекса\n\nУстановить настройки поиска и сделать Яндекс домашней страницей\n\nЗагрузить и установить Менеджер Браузеров\n\nПодробно (%legal%)\n\nУстанавливая рекомендуемое ABC программное обеспечение, вы соглашаетесь с лицензионным соглашением настольного ПО Яндекса (%software%).',
                    // Комплект 4
                    'ya-bro': 'Яндекс.Браузер — это простая и удобная программа для работы в интернете. Быстрая загрузка, только самые нужные кнопки и много места для просмотра страниц.\n\nПростой и быстрый\nЛюбимые сайты рядом\nУскоренная загрузка страниц (Turbo)\nВстроенная защита от вирусов\nПеревод сайтов\nЛегкий перенос настроек\n\nЗагрузить и установить Яндекс.Браузер\n\nПринять участие в улучшении сервисов Яндекса: автоматически отправлять анонимную статистику использования браузера.\n\nУстанавливая Яндекс.Браузер, вы соглашаетесь с условиями лицензионного соглашения (%bro%).'
                },
                tr: {
                    'abc-ya-bro': 'Yandex Uygulamaları\nYandex\'in uygulamalarını kurarak ABC Software’i de desteklemiş oluyorsunuz\n\nTam kurulum (önerilen)\nABC ve ABC tarafından önerilen Yandex programları ve Yandex ayarlarının kurulumunu yapın\n\nParametre ayarları\n\nYandex Elements indirin ve kurun\n\nYandex\'i tarayıcılarda ana sayfa olarak ayarlayın\n\nArama ve diğer Yandex servislerine hızlı erişim ayarlarını yapın\n\nAyrıntılı bilgi (%legal%)\n\nÖnerilen ABC yazılımını kurarak Yandex Elements (%software%) ve Yandex.Browser (%bro%) lisans sözleşmesini kabul etmiş sayılırsınız.',
                    'abc-ya': 'Yandex Uygulamaları\nYandex\'in uygulamalarını kurarak ABC Software’i de desteklemiş oluyorsunuz\n\nTam kurulum (önerilen)\nABC ve ABC tarafından önerilen Yandex programları ve Yandex ayarlarının kurulumunu yapın\n\nParametre ayarları\n\nYandex Elements indirin ve kurun\n\nYandex\'i tarayıcılarda ana sayfa olarak ayarlayın\n\nArama ve diğer Yandex servislerine hızlı erişim ayarlarını yapın\n\nAyrıntılı bilgi (%legal%)\n\nÖnerilen ABC yazılımını kurarak Yandex Elements (%software%) lisans sözleşmesini kabul etmiş sayılıyorsunuz.',
                    'logo-left-bro': 'Yandex Uygulamaları\nYandex\'in uygulamalarını kurarak ABC Software’i de desteklemiş oluyorsunuz\nABC, Yandex uygulamalarının ve ayarlarının kurulmasını öneriyor:\n\nYandex Elements indirin ve kurun\n\nYandex\'i tarayıcılarda ana sayfa olarak ayarlayın\n\nArama ve diğer Yandex servislerine hızlı erişim ayarlarını yapın\n\nAyrıntılı bilgi (%legal%)\n\nÖnerilen ABC yazılımını kurarak Yandex Elements (%software%) ve Yandex Browser (%bro%) lisans sözleşmesini kabul etmiş sayılırsınız.',
                    'logo-left': 'Yandex Uygulamaları\nYandex\'in uygulamalarını kurarak ABC Software’i de desteklemiş oluyorsunuz\nABC, Yandex uygulamalarının ve ayarlarının kurulmasını öneriyor:\n\nYandex Elements indirin ve kurun\n\nYandex\'i tarayıcılarda ana sayfa olarak ayarlayın\n\nArama ve diğer Yandex servislerine hızlı erişim ayarlarını yapın\n\nAyrıntılı bilgi (%legal%)\n\nÖnerilen ABC yazılımını kurarak Yandex Elements (%software%) lisans sözleşmesini kabul etmiş sayılıyorsunuz.',
                    'logo-right-bro': 'Yandex Uygulamaları\nYandex\'in uygulamalarını kurarak ABC Software’i de desteklemiş oluyorsunuz\nABC, Yandex uygulamalarının ve ayarlarının kurulmasını öneriyor:\n\nYandex Elements indirin ve kurun\n\nYandex\'i tarayıcılarda ana sayfa olarak ayarlayın\n\nArama ve diğer Yandex servislerine hızlı erişim ayarlarını yapın\n\nAyrıntılı bilgi (%legal%)\n\nÖnerilen ABC yazılımını kurarak Yandex Elements (%software%) ve Yandex Browser (%bro%) lisans sözleşmesini kabul etmiş sayılırsınız.',
                    'logo-right': 'Yandex Uygulamaları\nYandex\'in uygulamalarını kurarak ABC Software’i de desteklemiş oluyorsunuz\nABC, Yandex uygulamalarının ve ayarlarının kurulmasını öneriyor:\n\nYandex Elements indirin ve kurun\n\nYandex\'i tarayıcılarda ana sayfa olarak ayarlayın\n\nArama ve diğer Yandex servislerine hızlı erişim ayarlarını yapın\n\nAyrıntılı bilgi (%legal%)\n\nÖnerilen ABC yazılımını kurarak Yandex Elements (%software%) lisans sözleşmesini kabul etmiş sayılıyorsunuz.',
                    'soft-abc-bro': 'ABC kurulumu tamamlandı\nKurulum programından çıkmak için “Hazır” butonuna tıklayın.\n\nABC baslat\n\nABC, Yandex uygulamalarının ve ayarlarının kurulmasını öneriyor:\n\nYandex Elements indirin ve kurun\n\nYandex\'i tarayıcılarda ana sayfa olarak ayarlayın\n\nArama ve diğer Yandex servislerine hızlı erişim ayarlarını yapın\n\nAyrıntılı bilgi (%legal%)\n\nÖnerilen ABC yazılımını kurarak Yandex Elements (%software%) ve Yandex.Browser (%bro%) lisans sözleşmesini kabul etmiş sayılırsınız.',
                    'soft-abc': 'ABC kurulumu tamamlandı\nKurulum programından çıkmak için “Hazır” butonuna tıklayın.\n\nABC baslat\n\nABC, Yandex uygulamalarının ve ayarlarının kurulmasını öneriyor:\n\nYandex Elements indirin ve kurun\n\nYandex\'i tarayıcılarda ana sayfa olarak ayarlayın\n\nArama ve diğer Yandex servislerine hızlı erişim ayarlarını yapın\n\nAyrıntılı bilgi (%legal%)\n\nÖnerilen ABC yazılımını kurarak Yandex Elements (%software%) lisans sözleşmesini kabul etmiş sayılıyorsunuz.',
                    'ya-bro': 'Yandex.Browser — internette gezinmek için basit ve kullanışlı bir yazılım. Yüksek yükleme hızı, sadece en çok işinize yarayacak butonlar ve sayfa görüntülemek için geniş alan.\n\nHızlı ve kullanışlı\nSevdiğiniz siteler elinizin altında\nHızlandırılmış sayfa yüklenmesi (Turbo)\nDahili antivirüs\nSite çevirmeni\nEski ayarların kolayca taşınması\n\nYandex.Browser’ı indirin ve kurun\n\nTarayıcı kullanımıyla ilgili gizli istatistiği otomatik olarak göndererek Yandex servislerinin geliştirilmesinde rol alın.\n\nYandex.Browser\'ı kurarak lisans sözleşmesinin koşullarını kabul etmiş sayılıyorsunuz (%bro%).'
                }
            };
            /* jshint maxlen: 120 */

            var text = texts[lang][win];

            ['bro', 'legal', 'software'].forEach(function (key) {
                text = text.replace('%' + key + '%', links[lang][key]);
            });

            field.val(text);
        },

        /**
         * Обновляет заголовок в соответствии с выбранным табом.
         *
         * @param  {Number} val
         */
        _updateHeader: function (val) {
            var headers = [
                translate('The text installer'),//'Текст инсталятора'
                translate('The text installer first screen'),//'Текст инсталятора первого экрана'
                translate('The text installer second screen')//'Текст инсталятора второго экрана'
            ];

            var headerBlock = this._headerBlock ||
                (this._headerBlock = this.findBlockInside('installer-text', {
                    block: 'a-text',
                    modName: 'form',
                    modVal: 'header-quaternary'
                }));

            headerBlock.domElem.text(headers[val]);
        },

        /**
         * Растягивает окно "Текст инсталятора" на все свободное пространство.
         */
        _updateITHeight: function () {
            var specialColumn = this.elem('column', 'last', 'yes');
            var textarea = this.elem('installer-text').find('textarea');

            var totalHeight = this.elem('column', 'special', 'one').eq(0).height();
            var contentHeight = 39; /* padding + smthing else :D */

            specialColumn
                .children()
                .filter('h3, .a-instruction__types_shown_yes, .a-instruction__langs')
                .each(function () {
                    contentHeight += $(this).outerHeight(true);
                });

            this.elem('installer-text')
                .children()
                .filter('h4, p')
                .each(function () {
                    contentHeight += $(this).outerHeight(true);
                });

            textarea.height(totalHeight - contentHeight);
        },

        /**
         * Обновляет список чекбоксов в коллекции.
         *
         * @param  {String} val Тип окна. Может содержать несколько типов, разделенных пробелом.
         */
        _updateWinList: function (val) {
            var checkboxModels = [];
            var lang = this._getModel().get('lang');

            val && val.split(' ').forEach(function (win) {
                this.findBlocksInside(this.elem(win, 'lang', lang), 'i-glue').forEach(function (glue) {
                    checkboxModels.push(glue.model);
                });
            }, this);

            this._getModel().set('winList', checkboxModels);
        },

        /**
         * Выставляет состояние текущих чекбоксов, сохраненных в модели в соответствии с переданным в функцию.
         *
         * @param  {String} winState
         */
        _updateWinListWithWinState: function (winState) {
            var model = this._getModel();

            winState || (winState = '111111111');

            var radiobox = this.findBlockInside(this.findElem('win', 'shown', 'yes'), {
                block: 'a-control',
                modName: 'lego',
                modVal: 'radiobox'
            });

            radiobox && radiobox.val(winState === '111111111' ? 'all' : 'user');

            model.get('winList').forEach(function (model, i) {
                var flag = Boolean(Number(winState.charAt(i)));

                model.get('checked') !== flag && model.set('checked', flag);
            });
        }
    }, {
        live: function () {
            /* Прелоадер картинок */
            $(this.buildSelector('win'))
                .map(function () {
                    return $(this).css('backgroundImage').match(/url\(["']?([^'")]+)["']?\)/)[1];
                })
                .toArray()
                .filter(unique())
                .forEach(function (url) {
                    new Image().src = url;
                });

            /* Биндер на выбор комплекта */
            this.liveBindTo('kit', 'click tap', function (e) {
                var type = this.getMod($(e.target).closest('.a-instruction__kit'), 'type');

                e.preventDefault();
                e.stopPropagation();

                this._getModel().set('kit', numeric(type));
            });

            /* Биндер на переключение страницы. Кнопка "назад". */
            this.liveInitOnBlockInsideEvent('click', 'button', function () {
                this._getModel().set('kit', 0);
            });

            return false;
        }
    });

    /**
     * @param  {Number} digit
     * @return {String}
     */
    function alphabetic(digit) {
        var a = [
            'first',
            'second',
            'third',
            'fourth'
        ];

        return a[digit - 1];
    }

    /**
     * @param  {String} str
     * @return {Number}
     */
    function numeric(str) {
        var digit = {
            first: 1,
            second: 2,
            third: 3,
            fourth: 4
        };

        return digit[str] || 0;
    }

    /**
     * Вернет переведенную фразу на текущий язык по заданному ключу.
     *
     * @param  {String} key
     * @param  {Object} params
     * @return {String}
     */
    function translate(key, params) {
        params = params || {};

        return BEM.I18N('a-instruction', key, params);
    }

    /**
     * Фильтр уникальных значений.
     *
     * @return {Function}
     */
    function unique() {
        var hash = {};

        return function (item) {
            if (hash[item]) {
                return false;
            }

            return (hash[item] = true);
        };
    }
})(BEM);
