'use strict';

/**
 * Максимум отображаемых колонок.
 *
 * @type {Number}
 */
var MAX_COLUMNS = 30;

/**
 * Максимум отображаемых линий (кривых).
 *
 * @type {Number}
 */
var MAX_LINES = 30;

var colors = require('utils/colors');
var cdate = require('utils/dates').CDate;
var lodash = require('lodash');

/**
 * Набор стилей для графиков.
 *
 * @type {Array}
 */
var dashStyles = [
    'Solid',
    'ShortDash',
    'ShortDot',
    'ShortDashDot',
    'ShortDashDotDot',
    'Dot',
    'Dash',
    'LongDash',
    'DashDot',
    'LongDashDot',
    'LongDashDotDot'
];

/**
 * Возвращает код цвета по индексу с учетом номера отчета.
 * Если идет сравнение, то цвета второго отчета будет поярче.
 *
 * @param  {Number} index
 * @param  {Number} report
 * @return {String}
 */
function getColor(index, report) {
    var color = colors.getColorByIndex(index);

    return report > 0 ?
        colors.highlight(color, report * 0.3 + 1) :
        color;
}

function getMaxColumns() {
    return MAX_COLUMNS;
}

function getMaxLines() {
    return MAX_LINES;
}

/**
 * Регулярка, позволяющая разбить строку по трем разрядам с учетом минуса.
 *
 * @type {RegExp}
 */
var reversedThreeDigit = /\d{3}-?|\d{1,3}-?/g;

/**
 * Отбивает строку переданным символом.
 *
 * @param  {String} str
 * @param  {String} symbol
 * @return {String}
 */
function insertSymbolAfterThreeDigits(str, symbol) {
    return str.match(reversedThreeDigit).join(symbol);
}

/**
 * @param  {Object} params
 * @param  {String} params.currency
 * @param  {String} params.title
 * @param  {String} params.type
 * @return {Object}
 */
function formatAxis(params) {
    var obj = lodash.chain(params)
        .pick('type')
        .assign({
            extraValues: {},
            label: params.title
        })
        .value();

    params.currency && (obj.extraValues.entity_field = {
        currency: params.currency
    });

    return obj;
}

/**
 * Форматирует числа для вывода.
 * Отбивает разряды пробелами и подменяет минут на &minus;
 *
 * @param  {Number}  num
 * @param  {Boolean} money
 * @return {String}
 */
function formatNumber(num, money) {
    money && (num = num.toFixed(2));

    var digit = String(num).split('.');

    digit[0] = reverseString(insertSymbolAfterThreeDigits(reverseString(digit[0]), ' '));

    return digit.join(',').replace(/^-/, '\u2212'); // &minus;
}

/**
 * Форматирует периоды для человечесткого восприятия :)
 *
 * @param  {Array} period
 * @return {Array}
 */
function formatPeriod(period) {
    if (Array.isArray(period)) {
        var formatted = period.map(function (date) {
            return cdate(date).format('DD MMM YY');
        });

        return [
            formatted[0] + ' - ' + formatted[1],
            formatted[2] + ' - ' + formatted[3]
        ];
    }

    return [];
}

/**
 * @param  {Object} params
 * @param  {String} params.currency
 * @param  {String} params.title
 * @param  {String} params.type
 * @return {Object}
 */
function formatSeries(params) {
    var obj = lodash.chain(params)
        .pick(['data', 'type', 'yAxis'])
        .defaults({type: null})
        .assign({
            color: getColor(params.index, params.report),
            name: params.title
        })
        .value();

    obj.data = lodash.sortBy(obj.data, ['x']);
    params.diff && (obj.name += ' ' + params.period[params.report]);
    params.report > 0 && (obj.dashStyle = dashStyles[params.report]);

    return obj;
}

/**
 * Добавляет переносы, если строка длинее указанного интервала.
 *
 * @param  {String} str
 * @param  {Number} restriction     18|30
 * @return {String}
 */
function formatTitle(str, restriction) {
    str = replaceHTMLCharts(str);

    if (typeof restriction !== 'undefined') {
        var pieces = [];
        var escape = false;
        var lastIndex = 0;
        var len = str.length;

        for (var i = 0; i < len; i++) {
            switch (str.charAt(i)) {
            case '<':
                escape = true;
                continue;
            case '>':
                escape = false;
                continue;
            case ' ':
                if (escape) {
                    continue;
                }

                lastIndex !== i && pieces.push({
                    content: str.slice(lastIndex, i),
                    length: stripHTML(str.slice(lastIndex, i)).length
                });

                lastIndex = i + 1;
                continue;
            }
        }

        pieces.push({
            content: str.slice(lastIndex, len),
            length: stripHTML(str.slice(lastIndex, len)).length
        });

        var buff = '';
        var buffLength = 0;

        pieces.forEach(function (piece) {
            if (buffLength + piece.length > restriction) {
                buff.length && (buff += '<br/>');
                buffLength = piece.length;
            } else {
                buff.length && (buff += ' ');
                buffLength += piece.length;
            }

            buff += piece.content;
        });

        str = buff;
    }

    return str;
}

/**
 * Хак для html спецсимволов, которые доезжают из танкера.
 *
 * @param  {String} string
 * @return {String}
 */
function replaceHTMLCharts(string) {
    var pairs = {
        '&nbsp;': ' '
    };

    lodash.each(pairs, function (val, pattern) {
        string = string.replace(pattern, val);
    });

    return string;
}

/**
 * Удаляет html теги из текста. Не работает для вложенных тегов.
 *
 * @param  {String} str
 * @return {String}
 */
function stripHTML(str) {
    var tagPattern = /<([a-z]+)([^<]+)*(?:>(.*)<\/\1>|\s+\/>)/g;

    return str.replace(tagPattern, function (match, tag, attrs, content) {
        return content;
    });
}

/**
 * Разворачивает строку.
 *
 * @param  {String} str
 * @return {String}
 */
function reverseString(str) {
    return str.split('').reverse().join('');
}

exports.getColor = getColor;
exports.getMaxColumns = getMaxColumns;
exports.getMaxLines = getMaxLines;
exports.formatAxis = formatAxis;
exports.formatNumber = formatNumber;
exports.formatPeriod = formatPeriod;
exports.formatSeries = formatSeries;
exports.formatTitle = formatTitle;
exports.replaceHTMLCharts = replaceHTMLCharts;
