'use strict';

/**
 * Список цветов для графиков.
 *
 * @type {Array}
 */
var COLORS = [
    '#4572A7',
    '#AA4643',
    '#89A54E',
    '#80699B',
    '#3D96AE',
    '#DB843D',
    '#92A8CD',
    '#A47D7C',
    '#B5CA92'
];

/**
 * Количество возможных цветов.
 *
 * @type {Number}
 */
var colorsLength = COLORS.length;

/**
 * Возвращает код цвета по индексу.
 *
 * @param  {Number} index
 * @return {String}
 */
function colorByIndex(index) {
    return COLORS[index % colorsLength];
}

/**
 * @param  {Number} p
 * @param  {Number} q
 * @param  {Number} t
 * @return {Number}
 */
function hue2rgb(p, q, t) {
    t < 0 && (t += 1);
    t > 1 && (t -= 1);

    if (t < 1/6) {
        return p + (q - p) * 6 * t;
    }

    if (t < 1/2) {
        return q;
    }

    if (t < 2/3) {
        return p + (q - p) * (2/3 - t) * 6;
    }

    return p;
}

/**
 * @param  {Number} p
 * @param  {Number} q
 * @param  {Number} t
 * @return {Number}
 */
function hslToRgb(h, s, l) {
    var r, g, b;

    if (s === 0) {
        r = g = b = l; // achromatic
    } else {
        var q = l < 0.5 ? l * (1 + s) : l + s - l * s;
        var p = 2 * l - q;

        r = hue2rgb(p, q, h + 1/3);
        g = hue2rgb(p, q, h);
        b = hue2rgb(p, q, h - 1/3);
    }

    return [
        Math.round(r * 255),
        Math.round(g * 255),
        Math.round(b * 255)
    ];
}

/**
 * @param  {Number} p
 * @param  {Number} q
 * @param  {Number} t
 * @return {Number}
 */
function rgbToHsl(r, g, b) {
    r /= 255;
    g /= 255;
    b /= 255;

    var max = Math.max(r, g, b);
    var min = Math.min(r, g, b);
    var h;
    var s;
    var l = (max + min) / 2;

    if (max == min) {
        h = s = 0; // achromatic
    } else {
        var d = max - min;

        s = l > 0.5 ? d / (2 - max - min) : d / (max + min);

        switch (max) {
        case r:
            h = (g - b) / d + (g < b ? 6 : 0);
            break;
        case g:
            h = (b - r) / d + 2;
            break;
        case b:
            h = (r - g) / d + 4;
            break;
        }

        h /= 6;
    }

    return [h, s, l];
}

/**
 * Раскладывает цвет по каналам и возвращает их численное представление.
 *
 * @param  {String} color
 * @return {Array}
 */
function dec(color) {
    color = color.replace(/^#/, '');

    return [0, 2, 4].map(function (i) {
        return parseInt(color.slice(i, i + 2), 16);
    });
}

/**
 * Перегоняет массив с численным представлением rgb в hex.
 * [0, 0, 255] -> '#0000ff'.
 *
 * @param  {Array}  arr
 * @return {String}
 */
function hex(arr) {
    return '#' + arr.map(function (a) {
        return ('0' + parseInt(a, 10).toString(16)).slice(-2);
    }).join('');
}

/**
 * Делает светлее исходный цвет.
 *
 * @param  {String} color
 * @param  {Number} k
 * @return {String}
 */
function highlight(color, k) {
    var hsl = rgbToHsl.apply(null, dec(color));
    hsl[2] *= k || 1.1;

    return hex(hslToRgb.apply(null, hsl));
}

exports.getColorByIndex = colorByIndex;
exports.highlight = highlight;
