'use strict';

/**
 * Возвращает bemjson для элемента option блока select
 *
 * @param {string} selectedValue Выбранное значение
 * @param {object} item {label, value}
 * @return {bemjson}
 **/
function _getSelectItemBemjson(selectedValue, item) {
    var option = {
        elem: 'option',
        attrs: {value: item.value},
        content: item.label
    };

    if (selectedValue === item.value) {
        option.attrs.selected = 'selected';
    }

    if (item.disabled === 'yes') {
        option.attrs.disabled = 'disabled';
    }

    return option;
}

/**
 * Возвращает текст выбранного пункта меню.
 *
 * @param  {Array}  items        Список пунктов меню.
 * @param  {String} selectedItem Значение поля value выбранного пункта.
 * @return {String}
 */
function getSelectedLabel(items, selectedItem) {
    var selected = items.filter(function (item) {
        return item.value === selectedItem;
    });

    if (selected.length) {
        return selected[0].label;
    } else {
        return items[0].label;
    }
}

/**
 * Преобразует [{value: {string}, label: {string}}] в
 * bemjson списка элементов option блока select
 *
 * @param {array} items
 * @param {string} [selectedItemValue]
 **/
function getSelectItemsBemjson(items, selectedItemValue) {
    return items.map(_getSelectItemBemjson.bind(null, selectedItemValue));
}

/**
 * Возвращает bemjson __radio для блока radio-button
 *
 * @param {object} item {label: {string}, value: {string}, tabindex: {number}, disabled: {string}}
 * @return {object}
 **/
function _getRadioButtonItemBemjson(item) {
    var radio =  {
        elem: 'radio',
        controlAttrs: {
            value: item.value
        },
        content: item.label
    };

    if (typeof item.disabled !== 'undefined') {
        radio.elemMods = {
            disabled: item.disabled
        };
    }

    if (typeof item.id !== 'undefined') {
        radio.controlAttrs.id = item.id;
    }

    if (typeof item.tabindex !== 'undefined') {
        radio.controlAttrs.tabindex = item.tabindex;
    }

    return radio;
}

/**
 * Возвращает bemjson для элементов radio блока radio-button
 *
 * @param {array} items параметры кнопок [{label: {string}, value: {string}, tabindex: {number}, disabled: {string}}]
 * @return {array}
 **/
function getRadioButtonItemsBemjson(items) {
    return items.map(_getRadioButtonItemBemjson);
}

/**
 * Возвращает элемент radio блока radiobox
 * Формат параметров {
 *  value: {string},
 *  label: {string},
 *  disabled: {string},
 *  tabindex: {number}
 * }
 *
 * @param {object} item Параметры элемента radio блока radiobox
 * @return {object}
 **/
function _getRadioboxItemBemjson(item) {
    var radio = {
        elem: 'radio',
        controlAttrs: {value: item.value},
        content: item.label
    };

    if (typeof item.disabled !== 'undefined') {
        radio.elemMods = {disabled: item.disabled};
    }

    if (typeof item.id !== 'undefined') {
        radio.controlAttrs.id = item.id;
    }

    if (typeof item.tabindex !== 'undefined') {
        radio.controlAttrs.tabindex = item.tabindex;
    }

    return radio;
}

/**
 * Возвращает список __radio для блока radiobox
 * Формат параметров [{
 *  value: {string},
 *  label: {string},
 *  disabled: {string},
 *  tabindex: {number}
 * }]
 *
 * @param {array} items Параметры элемента
 **/
function getRadioboxItemsBemjson(items) {
    return items.map(_getRadioboxItemBemjson);
}

exports.getSelectItemsBemjson = getSelectItemsBemjson;
exports.getRadioButtonItemsBemjson = getRadioButtonItemsBemjson;
exports.getRadioboxItemsBemjson = getRadioboxItemsBemjson;
exports.getSelectedLabel = getSelectedLabel;
