'use strict';

/**
 * Добавила этот файл для задания языка для moment
 * и для добавления пресетов формат дат, т.к. в
 * стандартных для moment присутствует "г." для года
 *
 * Форматирование http://momentjs.com/docs/#/parsing/string-format/
 **/
var moment = require('moment');

function formatDate(date, format) {

    if (typeof format === 'undefined') {
        format = 'C';
    }

    switch (format) {
    case 'c': // от custom =)
        format = 'DD.M.YYYY';
        break;
    case 'C':
        format = 'DD.MM.YYYY';
        break;
    case 'cc':
        format = 'DD MMM YYYY';
        break;
    case 'CC':
        format = 'DD MMMM YYYY';
        break;
    case 'ccc':
        format = 'DD MMM YYYY HH:mm';
        break;
    case 'CCC':
        format = 'DD MMMM YYYY HH:mm';
        break;
    case 'cccc':
        format = 'DD MMM YYYY HH:mm (ddd)';
        break;
    case 'CCCC':
        format = 'DD MMMM YYYY HH:mm (dddd)';
        break;
    }

    return moment(date).format(format);
}

exports.formatDate = formatDate;

var lodash = require('lodash');

/**
 * Парсит строковое представление даты и возвращает экземпляр moment.js.
 *
 * @param  {String} date
 * @param  {String} interval
 * @return {Moment}
 */
function parseDate(date, interval) {
    var parts;

    switch (interval) {
    case 'day':
        parts = date.split('-');

        return moment(Date.UTC(Number(parts[0]), Number(parts[1]) - 1, Number(parts[2])));

    case 'week':
        parts = date.split('-');

        return moment(Date.UTC(Number(parts[0]), 0))
            .week(Number(parts[1]));

    case 'month':
        parts = date.split('-');

        return moment(Date.UTC(Number(parts[0]), Number(parts[1]) - 1));

    case 'year':
        return moment(Date.UTC(Number(date), 0));

    default:
        return moment(date);
    }
}

/**
 * Количество миллисекунд в одном дне.
 *
 * @type {Number}
 */
var MsInOneDay = 864e5;

/**
 * Считает номер дня в текущем году.
 *
 * @param  {String} date
 * @return {Number}
 */
function dayOfYear(date) {
    var year = date.slice(0, 4);
    var diff = new Date(date) - new Date(year + '-01-01');

    return Math.round(diff / MsInOneDay) + 1;
}

/**
 * Свой конструктор дат.
 * Учитывает интервалы, что повзоляет ин(де)крементировать даты на величину этих интервалов.
 *
 * @param {Date|String} date
 * @param {String}      [interval] day|week|month|year. По умолчанию "day".
 */
function CDate(date, interval) {
    if (!(this instanceof CDate)) {
        return new CDate(date, interval);
    }

    if (typeof interval === 'undefined') {
        interval = 'day';
    }

    this._interval = interval;

    if (typeof date === 'string') {
        this._date = parseDate(date, interval);
    }

    if (date instanceof Date) {
        this._date = moment(date);
    }

    if (typeof date === 'undefined') {
        this._date = moment();
    }
}

lodash.assign(CDate.prototype, {
    /**
     * Уменьшение даты на величину установленного интервала (день, неделя и т.д.).
     *
     * @return {CDate}
     */
    decrement: function () {
        var interval = this._interval;
        var current = this._date[interval]();

        this._date[interval](current - 1);

        return this;
    },

    /**
     * Увеличение даты на величину установленного интервала (день, неделя и т.д.).
     *
     * @return {CDate}
     */
    increment: function () {
        var interval = this._interval;
        var current = this._date[interval]();

        this._date[interval](current + 1);

        return this;
    },

    /**
     * Меняет интервал.
     *
     * @param  {String} intr    "day|week|month|year"
     */
    interval: function (intr) {
        this._interval = intr;

        return this;
    },

    /**
     * Согласится, если текущая дата является субботой или воскресением.
     *
     * @return {Boolean}
     */
    isWeekend: function () {
        var weekday = this._date.day();

        return weekday === 0 || weekday === 6;
    },

    /**
     * Милисекунды от 1970 г.
     *
     * @return {Number}
     */
    time: function () {
        return this._date.valueOf();
    },

    /**
     * Приводит дату к строчному виду. Учитывает установленный интервал при форматировании.
     *
     * @return {String}
     */
    toString: function () {
        var format;

        switch (this._interval) {
        case 'day':
            format = 'YYYY-MM-DD';
            break;

        case 'week':
            format = 'gggg-WW';
            break;

        case 'month':
            format = 'YYYY-MM';
            break;

        case 'year':
            format = 'YYYY';
            break;

        default:
            format = 'YYYY-MM-DD HH:mm:ss';
        }

        return this._date.format(format);
    },

    /**
     * Выдает некоторое числовое представление даты с учетом интервала.
     * Может посчитать в днях, неделях, месяцах, годах.
     *
     * @return {Number}
     */
    valueOf: function () {
        var interval = this._interval;
        var years = moment.duration(interval === 'week' ? this._date.weekYear() : this._date.year(), 'years');

        switch (interval) {
        case 'day':
            return dayOfYear(this._date.format('YYYY-MM-DD')) + years.asDays();

        case 'week':
            return this._date.week() + Math.floor(years.asWeeks());

        case 'month':
            return this._date.month() + years.asMonths();

        case 'year':
            return years;

        default:
            return Math.floor(this._date.valueOf() / 1000);
        }
    }
});

/**
 * Проксируемые методы moment.js.
 *
 * @type {Array}
 */
var momentJsMethods = [
    'date',
    'day',
    'format',
    'month',
    'months',
    'year',
    'years'
];

momentJsMethods.forEach(function (m) {
    CDate.prototype[m] = (function (m) {
        return function () {
            return this._date[m].apply(this._date, arguments);
        };
    })(m);
});

/**
 * Выводит дату в формате YYYY-MM-DD.
 *
 * @param  {Date}   date
 * @param  {String} day
 * @param  {String} month
 * @param  {String} year
 * @return {String}
 */
function formatDate(date, day, month, year) {
    return [
        year || date.getFullYear(),
        formatDigits(month || date.getMonth() + 1),
        formatDigits(day || date.getDate())
    ].join('-');
}

/**
 * Добавляет ноль одноразрядным числам.
 *
 * @param  {Number|String} string
 * @return {String}
 */
function formatDigits(string) {
    return ('0' + String(string)).substr(-2);
}

/**
 * Умеет пересчитывать дату для некоторых периодов.
 * Возвращает дату в формате "YYYY-MM-DD".
 *
 * @param  {String} period      "30days|90days|180days|365days|thismonth|lastmonth|thisyear".
 * @param  {String} [interval]  "day|weel|month|year" или какие-нибудь невалидные.
 * @return {String}
 */
function getPeriod(period, interval) {
    var today = new Date();
    var date;

    switch (period) {
    case '30days':
        today.setDate(today.getDate() - 29);
        date = formatDate(today);

        break;

    case '90days':
        today.setDate(today.getDate() - 89);
        date = formatDate(today);

        break;

    case '180days':
        today.setDate(today.getDate() - 179);
        date = formatDate(today);

        break;

    case '365days':
        today.setDate(today.getDate() - 364);
        date = formatDate(today);

        break;

    case 'thismonth':
        date = formatDate(today, '01');

        break;

    case 'lastmonth':
        today.setMonth(today.getMonth() - 1);
        date = formatDate(today, '01');

        break;

    case 'thisyear':
        date = formatDate(today, '01', '01');

        break;

    default:
        return;
    }

    switch (interval) {
    case 'week':
        return new CDate(date)
            .interval('week')
            .toString();

    case 'month':
        return new CDate(date)
            .interval('month')
            .toString();

    case 'year':
        return new CDate(date)
            .interval('year')
            .toString();

    default:
        return date;
    }
}

exports.CDate = CDate;
exports.getPeriod = getPeriod;
