'use strict';

/**
 * Регулярка, позволяющая разбить строку по трем разрядам с учетом минуса.
 *
 * @type {RegExp}
 */
var reversedThreeDigit = /\d{3}-?|\d{1,3}-?/g;

/**
 * Отбивает пробелами разряды.
 *
 * @private
 * @param  {string} string
 * @return {string}
 */
function insertSpaceAfterThreeDigits(string) {
    return string.match(reversedThreeDigit).join(' ');
}

/**
 * Переворачивает строку.
 *
 * @param  {string} string
 * @return {string}
 */
function reverseString(string) {
    return string.split('').reverse().join('');
}

/**
 * Форматирует деньги: Отбивает пробелами и оставляет два знака после запятой.
 *
 * @public
 * @param  {number} val
 * @return {string}
 */
exports.formatMoney = function (val, delimiter) {
    return exports.formatNumber(val, delimiter, 2);
};

/**
 * Отбивает разряды пробелами.
 *
 * @public
 * @param  {number} val       Исходная величина.
 * @param  {string} delimiter Разделитель. По умолчанию ",".
 * @param  {number} fixed     Количество разрядов после запятой.
 * @return {string}
 */
exports.formatNumber = function (val, delimiter, fixed) {
    if (typeof fixed === 'number') {
        val = val.toFixed(fixed);
    }

    var parts = String(val).split('.');

    parts[0] = reverseString(insertSpaceAfterThreeDigits(reverseString(parts[0])));

    return parts
        .join(delimiter || ',')
        .replace(/^-/, '\u2212'); // &minus;
};

/**
 * Округляет значение до целых и добавляет знак %.
 * В противном случае вернет '-'.
 *
 * @public
 * @param  {string|number} val
 * @return {string}
 */
exports.formatPercent = function (val) {
    val = parseFloat(val);

    if (isNaN(val)) {
        return '-';
    }

    return Math.round(val) + ' %';
};
