'use strict';

function isLeaf(node) {
    return node.type == 'item';
}

function isSubGroup(node) {
    return node.type === 'sub-group';
}

function urlIsEqual(url) {
    return function (node) {
        return node.url === url;
    };
}

function urlLengthComparator(a, b) {
    if (a.url == null && b.url == null) {
        return 0;
    }
    if (a.url == null) {
        return 1;
    }
    if (b.url == null) {
        return -1;
    }

    var res = b.url.length - a.url.length;

    if (res === 0) {
        if (b.url < a.url) {
            return 1;
        }

        if (b.url > a.url) {
            return -1;
        }

        return 0;
    }

    return res;
}

function getTreeLeaves(current, parent) {
    var result = [];
    var prevItem = null;
    current.items.forEach(function (item) {
        var node = item2node(item, parent, prevItem, null);
        if (prevItem) {
            prevItem.next = node;
        }
        if (isLeaf(item)) {
            result.push(node);
        } else if (isSubGroup(item)) {
            item.items.forEach(function (item) {
                item.subGroup = true;
            });

            result = result.concat(getTreeLeaves(item, parent));
        } else if (item.items) {
            result = result.concat(getTreeLeaves(item, node));
        }
        prevItem = node;
    });

    return result.sort(urlLengthComparator);
}

function item2node(item, parent, previous, next) {
    var node = getNode(
        item.type,
        item.name,
        item.url,
        item.cmd,
        parent,
        previous,
        next,
        item.active === 1
    );

    if (item.subGroup) {
        node.subGroup = true;
    }

    if (parent) {
        parent.children = parent.children || [];
        parent.children.push(node);
    }

    if (item.type === 'item' && item.items) {
        node.children = item.items.map(function (x) {
            return item2node(x, node, null, null);
        });
    }

    return node;
}

function getNode(type, name, url, cmd, parent, previous, next, active) {
    return {
        type: type,
        name: name,

        url: url || null,
        cmd: cmd || null,

        parent: parent || null,
        next: next || null,
        previous: previous || null,

        active: active
    };
}

function getSelectedLevels(navLeaves, predicate) {
    var result = [];
    var activeLeaf = getActiveLeaf(navLeaves, predicate);
    var node = activeLeaf ? activeLeaf : navLeaves[0];

    if (!activeLeaf) {
        while (node) {
            if (!node.parent) {
                return [getLevelByNode(node)];
            }

            node = node.parent;
        }
    } else {
        var level;
        while (node) {
            level = getLevelByNode(node);
            result.push(level);

            node = node.parent;
        }
    }

    return result;
}

function getLevelByNode(navNode) {
    var current = getTopSibling(navNode);
    var result = [];
    /*jshint boss:true*/
    do {
        if (current.name) {
            result.push(current);
        }
    } while (current = current.next);
    /*jshint boss:false*/
    return result;
}

function getTopSibling(navNode) {
    var result = navNode;
    while (result.previous) {
        result = result.previous;
    }

    return result;
}

function isActiveLeaf(leaf) {
    return leaf.active === true;
}

function getActiveLeaf(navLeaves, predicate) {
    predicate = predicate || isActiveLeaf;
    var selectedLeaves = navLeaves.filter(function (x) {return predicate(x);});
    return selectedLeaves[0] || null;
}

function subitemBemjson(navNode) {
    return {
        block: 'a-navigation',
        elem: 'subitem',
        content: navNode.name,
        url: navNode.url
    };
}

function itemBemjson(navNode) {
    var result = {
        block: 'a-navigation',
        elem: 'item',
        mix: navNode.subGroup
            ? {
                block: 'a-navigation',
                elem: 'sub-group-item'
            }
            : null,
        elemMods: {active: navNode.active ? 'yes' : ''},
        url: navNode.url,
        content: navNode.name
    };

    if (navNode.type == 'item' && navNode.children && navNode.children.length > 0) {
        result.subitems = navNode.children.map(subitemBemjson);
    }

    return _addNonBreakingHyphen(result);
}

/**
 * @param {object} bemjson
 * @return {object}
 */
function _addNonBreakingHyphen(bemjson) {
    if (/[a-zа-яё\d]-[a-zа-яё\d]/i.test(bemjson.content)) {
        bemjson.content = bemjson.content.replace(/([a-zа-яё\d]+(?:-[a-zа-яё\d]+)+)/gi, '<nobr>$1</nobr>');
    }

    return bemjson;
}

function groupBemjson(navNode) {
    return {
        block: 'a-navigation',
        elem: 'group',
        content: [
            {block: 'a-navigation', elem: 'group-title', content: navNode.name}
        ].concat(navNode.children.map(itemBemjson))
    };
}

function groupOrItemBemjson(navNode) {
    if (navNode.type == 'group') {
        return groupBemjson.apply(null, arguments);
    }

    return itemBemjson.call(null, navNode);
}

function getNavigationBemjson(navTree, url) {
    var predicate = url ? urlIsEqual(url) : null;
    var leaves = getTreeLeaves({items: navTree}, null);
    var levels = getSelectedLevels(leaves, predicate);
    var result = {};

    // Уровни навигации появляются в таком порядке:
    var havePrimary = levels.length > 0; // Левое темное меню
    var haveSecondary = levels.length > 2; // Левое светлое меню
    var secondaryGroups = levels.length > 3; // Есть ли группы в левом светлом меню

    if (havePrimary) {
        result.primary = {
            block: 'a-navigation',
            mods: {type: 'primary'},
            content: levels[levels.length - 1].map(itemBemjson)
        };
    }

    if (haveSecondary) {
        result.secondary = {
            block: 'a-navigation',
            mods: {type: 'secondary'},
            content: levels[secondaryGroups ? 2 : 1].map(groupOrItemBemjson)
        };
    }

    // Показывать ли верхнее меню
    var haveTertiary = levels.length > 1 && (!result.secondary || secondaryGroups);

    if (haveTertiary) {
        var tertiary = replaceActiveLeafName(levels[0]);
        result.tertiary = {
            block: 'a-navigation',
            mods: {type: 'tertiary'},
            content: tertiary.map(itemBemjson)
        };
    }

    return result;
}

function replaceActiveLeafName(leafs) {
    leafs.forEach(function (node) {
        if (!node.children) {
            return;
        }

        var activeChild = node.children.filter(function (child) {return child.active;})[0];
        activeChild && (node.name = activeChild.name);
    });

    return leafs;
}

function getSecondaryBemjson(navTree, url) {
    return getNavigationBemjson(navTree, url).secondary;
}

exports.getTreeLeaves = getTreeLeaves;
exports.isLeaf = isLeaf;
exports.getNode = getNode;
exports.getSelectedLevels = getSelectedLevels;
exports.getActiveLeaf = getActiveLeaf;
exports.getTopSibling = getTopSibling;
exports.getLevelByNode = getLevelByNode;
exports.getNavigationBemjson = getNavigationBemjson;
exports.replaceActiveLeafName = replaceActiveLeafName;
exports.getSecondaryBemjson = getSecondaryBemjson;
