'use strict';

/**
 * Модуль для работы с датами.
 *
 * @type {Object}
 */
var cdate = require('utils/dates').CDate;
var getPeriod = require('utils/dates').getPeriod;
var lodash = require('lodash');

var counter = 0;
var ID_PREFIX = 'uniq';
var ID_PROP = '_PId';

/**
 * Формирует уникальный список возможных группировок.
 *
 * @param  {Array} arr
 * @param  {Array} fields
 * @return {Array}
 */
function axisList(arr, fields) {
    var uniq = {};

    arr.forEach(function (dataItem) {
        var key = serialize(dataItem, fields);

        uniq.hasOwnProperty(key) || (uniq[key] = lodash.pick(dataItem, fields));
    });

    return lodash.chain(uniq)
        .map(lodash.identity)
        .sortBy(['currency_id'])
        .value();
}

/**
 * Возвращает полный список дат, соответствующий исходному периоду.
 *
 * @param  {String} from                По умолчанию в днях. Или в sourceInterval.
 * @param  {String} to                  По умолчанию в днях. Или в sourceInterval.
 * @param  {String} finalInterval       "day|week|month|year".
 * @param  {String} sourceInterval      "day|week|month|year".
 * @return {Array}
 */
function datesList(from, to, finalInterval, sourceInterval) {
    var dates = [];
    var current = cdate(from, sourceInterval || 'day');
    to = cdate(to, sourceInterval || 'day');

    current.interval(finalInterval);
    to.interval(finalInterval);

    while (current <= to) {
        dates.push(String(current));
        current.increment();
    }

    return dates;
}

/**
 * Формирует список полей, которые можно будет использовать для сортивки.
 *
 * @param {Array}  [...] Массив (один или несколько) с данными.
 * @return {Array}
 */
function fieldsList() {
    return lodash.chain(arguments)
        .reduce(function (acc, arg) {
            return Array.isArray(arg) ?
                acc.concat(arg) :
                acc;
        }, [])
        .pluck('id')
        .value();
}

/**
 * Фиксит отчеты.
 *
 * @param  {Array}        d
 * @param  {Object}       params
 * @param  {Array}        params.currencies
 * @param  {Array}        params.fields
 * @param  {Array}        params.groupingFields
 * @param  {Boolean}      params.hasCurrencies
 * @param  {Boolean}      params.hasDate
 * @param  {Array}        params.valueableFields
 * @param  {String}       params.interval           "day|week|month|year".
 * @param  {Array|String} params.period
 * @return {Array}
 */
function fixData(d, params) {
    // Количество периодов.
    var reportsNum = d.length;
    var reports = lodash.range(reportsNum);

    if (['day', 'week', 'month', 'year'].indexOf(params.interval) === -1) {
        return d;
    }

    if (reportsNum === 1) {
        return fixDates(d, params);
    }

    // Пропущенные периоды.
    var missing = reports.reduce(function (h, r) {
        h[r] = [];

        return h;
    }, {});

    // Болванка для пустых отчетов.
    var sample = params.valueableFields.reduce(function (h, f) {
        h[f] = '-';

        return h;
    }, {});

    var datesMap = {};
    if (params.hasDate) {
        var datesLists = reports.map(function (r) {
            datesMap[r] = {};

            return datesList(params.period[r * 2], params.period[r * 2 + 1], params.interval);
        });

        datesLists[0].forEach(function (date, i) {
            datesMap[0][date] = datesLists[1][i];
            datesMap[1][datesLists[1][i]] = date;
        });
    }

    // Хэш с ключами для разных отчетов.
    var data = {};
    reports.forEach(function (r) {
        data[r] = d[r].data.reduce(function (h, dataItem, i) {
            h[serialize(dataItem, params.groupingFields)] = i;

            return h;
        }, {});
    });

    // Ищем недостающие.
    reports.forEach(function (r) {
        d[r].data.forEach(function (dataItem) {
            var criterion = params.hasDate ?
                lodash.assign({}, dataItem, {
                    date: datesMap[r][dataItem.date]
                }) :
                dataItem;
            var key = serialize(criterion, params.groupingFields);

            if (!data[Number(!r)].hasOwnProperty(key)) {
                missing[Number(!r)].push(lodash.assign({}, sample, lodash.pick(criterion, params.groupingFields)));
            }
        });
    });

    lodash.each(missing, function (arr, r) {
        arr.length && (d[r].data = d[r].data.concat(arr));
    });

    return d;
}

/**
 * Фиксит пропущенные даты.
 *
 * @param  {Array}        d
 * @param  {Object}       params
 * @param  {Array}        params.currencies
 * @param  {Array}        params.fields
 * @param  {Array}        params.groupingFields
 * @param  {Boolean}      params.hasCurrencies
 * @param  {Boolean}      params.hasDate
 * @param  {Array}        params.valueableFields
 * @param  {String}       params.interval           "day|week|month|year".
 * @param  {Array|String} params.period
 * @return {Array}
 */
function fixDates(d, params) {
    if (!params.hasDate) {
        return d;
    }

    var data = d[0].data;
    var fieldsWoDate = lodash.difference(params.groupingFields, ['date']);

    // Недостающие отчеты.
    var missing = [];
    // Болванка.
    var sample = {};
    // Возможные группировки.
    var unique = {};

    // Дописываем пустые значения в болванку по значащим полям.
    params.valueableFields.forEach(function (field) {
        sample[field] = '-';
    });

    var key;
    // Сериализованные исходные данные по полям группировки (с учетом даты).
    var dataKeys = data.reduce(function (keys, dataItem, i) {
        key = serialize(dataItem, fieldsWoDate); // Без даты.

        // Обновляем группировки.
        unique.hasOwnProperty(key) || (unique[key] = lodash.pick(dataItem, fieldsWoDate));
        // Обновляем общие ключи.
        keys[key + dataItem.date] = i;

        return keys;
    }, {});

    // Список дат для проверки.
    var dates = Array.isArray(params.period) ?
        datesList(params.period[0], params.period[1], params.interval) :
        datesList(
            getPeriod(params.period, params.interval) || data[0].date,
            data[data.length - 1].date,
            params.interval,
            params.interval
        );

    dates.forEach(function (date) {
        lodash.each(unique, function (groupingFields, id) {
            key = id + date;

            dataKeys.hasOwnProperty(key) || missing.push(lodash.assign({}, groupingFields, sample, {date: date}));
        });
    });

    missing.length && (d[0].data = data.concat(missing));

    return d;
}

/**
 * Формирует список полей для сортировки из входных данных.
 *
 * @param  {Object}  fieldsToRender
 * @param  {Array}   fieldsToRender.dimension_fields
 * @param  {Array}   fieldsToRender.entity_fields
 * @param  {Array}   fieldsToRender.fields
 * @param  {Boolean} withCurrency
 */
function fieldsToSort(fieldsToRender, withCurrency) {
    return fieldsList(fieldsToRender.dimension_fields)
        .concat(withCurrency ? 'currency_id' : [])
        .concat(fieldsList(fieldsToRender.entity_fields, fieldsToRender.fields));
}

/**
 * Выдает валютную символику. Используется в highcharts.
 *
 * @param  {String} code
 * @return {String}
 */
function formatCurrency(code) {
    switch (code) {
    case 'USD':
        return '$';

    case 'RUB':
        // через блок i-font сделать не получилось, т.к.
        // он делает это через class и css, а highchart,
        // по всей видимости, не может прокинуть class на
        // график, зато он понимает span с атрибутом style
        return '<span style="font-family: rub-arial-regular;">Р</span>';

    case 'EUR':
        return '€';
    }

    return code;
}

/**
 * Парсит число и если нужно обрезает лишние числа.
 *
 * @param  {(Number|String)} val
 * @param  {String}          [type] Тип поля
 * @return {Number}
 */
function formatValue(val, type) {
    val = numeric(val);

    if (type === 'float') {
        val = parseFloat(val.toFixed(4));
    }

    return val;
}

/**
 * Возвращает уникальный id.
 *
 * @return {String}
 */
function getUniqId() {
    return ID_PREFIX + (++counter);
}

/**
 * Возвращает уникальный id объекта.
 * Если у объекта нет id, то помечает его предварительно.
 * Если объект не указан, просто вернет id.
 *
 * @param  {Object} obj
 * @return {String}
 */
function identify(obj) {
    if (typeof obj === 'undefined') {
        return getUniqId();
    }

    return ID_PROP in obj ?
        obj[ID_PROP] :
        obj[ID_PROP] = getUniqId();
}

/**
 * Приводит значение к булевому и инвертирует его.
 *
 * @param  {*}       a
 * @return {Boolean}
 */
function inverse(a) {
    return !Boolean(a);
}

/**
 * Приводит строку к числу.
 * Если isNaN === true, то возвращает 0.
 * Если отчет был востанновлен, то вместо числа может фигурировать "-".
 *
 * @param  {String} a
 * @return {Number}
 */
function numeric(a) {
    a = Number(a);

    return isNaN(a) ? 0 : a;
}

/**
 * Создает ключи.
 *
 * @param  {Object} dataItem
 * @param  {Array}  list
 * @return {String}
 */
function serialize(dataItem, list) {
    var str = '';

    list.forEach(function (attr) {
        str += String(dataItem[attr]);
    });

    return str;
}

/**
 * Сортирует массив по заданному критерию.
 *
 * @param  {Array}    arr
 * @param  {Function} criterion
 * @return {Array}
 */
function sortData(arr, criterion) {
    if (!Array.isArray(arr)) {
        throw new TypeError('Should use Array.');
    }

    if (typeof criterion !== 'function') {
        throw new TypeError('Should use Function.');
    }

    return arr.sort(criterion);
}

/**
 * Сортирует массив с данными по указанным ключам.
 * Есть необходимость также указывать словарь полей, чтобы грамотно сравнивать числа нестрочно.
 *
 * @param  {Array}   arr
 * @param  {Array}   fields     Список полей, по которым будут отсортированные данные.
 * @param  {Object}  fieldDesc  Словарь полей, используемых в статистике.
 * @param  {Boolean} [reversed] Сортировка в обратном порядке.
 * @return {Array}
 */
function sortByFields(arr, fields, fieldDesc, reversed) {
    if (!Array.isArray(fields)) {
        throw new TypeError('Should use Array.');
    }

    var fieldsLen = fields.length;
    var field;
    var i;
    var av;
    var bv;

    return sortData(arr, function (a, b) {
        for (i = 0; i < fieldsLen; i++) {
            field = fields[i];

            if (['money', 'number', 'straight_number'].indexOf(fieldDesc[field].type) > -1) {
                av = numeric(a[field]);
                bv = numeric(b[field]);
            } else {
                av = a[field];
                bv = b[field];
            }

            if (av < bv) {
                return reversed ? 1 : -1;
            }

            if (av > bv) {
                return reversed ? -1 : 1;
            }
        }

        return 0;
    });
}

/**
 * Вернет список полей с баблом.
 *
 * @param  {Object} fieldsToRender
 * @return {Array}
 */
function valueableFields(fieldsToRender) {
    return fieldsList(fieldsToRender.fields);
}

/**
 * Создает словарик из исходного массива по заданному ключу.
 *
 * @param  {Array}  arr
 * @param  {String} key
 * @return {Object}
 */
function vocabulary(arr, key) {
    return arr.reduce(function (hash, item) {
        hash[item[key]] = lodash.omit(item, key);

        return hash;
    }, {});
}

exports.axisList = axisList;
exports.fieldsList = fieldsList;
exports.fieldsToSort = fieldsToSort;
exports.fixData = fixData;
exports.formatCurrency = formatCurrency;
exports.formatValue = formatValue;
exports.identify = identify;
exports.inverse = inverse;
exports.numeric = numeric;
exports.sortByFields = sortByFields;
exports.valueableFields = valueableFields;
exports.vocabulary = vocabulary;
