describe('utils/dates', function () {
    'use strict';

    var dates = require('utils/dates');

    function addZero(number, length) {
        number = String(number);
        var d = length - number.length;

        return String(Math.pow(10, d)).slice(1) + number;
    }

    describe('CDate', function () {
        describe('increment()', function () {
            it('корректно инкрементирует дни', function () {
                var today = new Date();
                var cdate = dates.CDate(new Date());

                today.setDate(today.getDate() + 1);

                cdate.increment().date().must.be(today.getDate());
            });

            it('корректно инкрементирует недели', function () {
                var today = new Date();
                var cdate = dates.CDate(new Date(), 'week');

                today.setDate(today.getDate() + 7);

                cdate.increment().date().must.be(today.getDate());
            });

            it('корректно инкрементирует месяца', function () {
                var cdate = dates.CDate('2014-02-15', 'month');

                cdate.increment().format('MM').must.be('03');
            });

            it('корректно инкрементирует года', function () {
                var today = new Date();
                var cdate = dates.CDate(new Date(), 'year');

                today.setFullYear(today.getFullYear() + 1);

                cdate.increment().year().must.be(today.getFullYear());
            });
        });

        describe('decrement()', function () {
            it('корректно декрементирует дни', function () {
                var today = new Date();
                var cdate = dates.CDate(new Date());

                today.setDate(today.getDate() - 1);

                cdate.decrement().date().must.be(today.getDate());
            });

            it('корректно декрементирует недели', function () {
                var today = new Date();
                var cdate = dates.CDate(new Date(), 'week');

                today.setDate(today.getDate() - 7);

                cdate.decrement().date().must.be(today.getDate());
            });

            it('корректно декрементирует месяца', function () {
                var cdate = dates.CDate('2014-02-15', 'month');

                cdate.decrement().format('MM').must.be('01');
            });

            it('корректно декрементирует года', function () {
                var today = new Date();
                var cdate = dates.CDate(new Date(), 'year');

                today.setFullYear(today.getFullYear() - 1);

                cdate.decrement().year().must.be(today.getFullYear());
            });
        });

        describe('isWeekend()', function () {
            it('согласится, что 2014-06-28 - это суббота', function () {
                dates.CDate('2014-06-28').isWeekend().must.be(true);
            });

            it('согласится, что 2014-06-29 - это воскресение', function () {
                dates.CDate('2014-06-29').isWeekend().must.be(true);
            });
        });

        describe('time()', function () {
            it('вернет дату в милисекундах', function () {
                var today = new Date();
                var cdate = dates.CDate(today);

                cdate.time().must.be(today.getTime());
            });
        });

        describe('toString()', function () {
            it('вернет дату в формате yyyy-mm-dd для дневного интервала', function () {
                var today = new Date();
                var cdate = dates.CDate(today);

                cdate.toString().must.be([
                    today.getFullYear(),
                    addZero(today.getMonth() + 1, 2),
                    addZero(today.getDate(), 2)
                ].join('-'));
            });

            it('вернет дату в формате yyyy-ww для недельного интервала', function () {
                dates.CDate('2014-50', 'week').toString().must.be('2014-50');
            });

            it('вернет дату в формате yyyy-mm для месячного интервала', function () {
                var today = new Date();
                var cdate = dates.CDate(today, 'month');

                cdate.toString().must.be([
                    today.getFullYear(),
                    addZero(today.getMonth() + 1, 2)
                ].join('-'));
            });

            it('вернет дату в формате yyyy для годового интервала', function () {
                var today = new Date();
                var cdate = dates.CDate(today, 'year');

                cdate.toString().must.be(String(today.getFullYear()));
            });

            it('должен вернуть правильный год в недельном интервале', function () {
                var date = '2014-01-01';
                var cdate = dates.CDate(date, 'day');

                cdate.interval('week').toString().must.be('2014-01');
            });
        });

        describe('valueOf()', function () {
            it('корректно считает разницу дат для дневного интервала', function () {
                (dates.CDate('2014-05-03') - dates.CDate('2014-05-02')).must.be(1);
            });

            it('корректно считает разницу дат для недельного интервала', function () {
                (dates.CDate('2014-51', 'week') - dates.CDate('2014-49', 'week')).must.be(2);
            });

            it('корректно считает разницу дат для месячного интервала', function () {
                (dates.CDate('2014-12', 'month') - dates.CDate('2014-09', 'month')).must.be(3);
            });

            it('корректно считает разницу дат для годового интервала', function () {
                (dates.CDate('2014', 'year') - dates.CDate('2010', 'year')).must.be(4);
            });
        });
    });

    describe('getPeriod()', function () {
        it('для 30days вернет ', function () {
            var today = new Date();
            today.setDate(today.getDate() - 29);

            dates.getPeriod('30days')
                .must.be([
                    today.getFullYear(),
                    addZero(today.getMonth() + 1, 2),
                    addZero(today.getDate(), 2)
                ].join('-'));
        });

        it('для 90days вернет ', function () {
            var today = new Date();
            today.setDate(today.getDate() - 89);

            dates.getPeriod('90days')
                .must.be([
                    today.getFullYear(),
                    addZero(today.getMonth() + 1, 2),
                    addZero(today.getDate(), 2)
                ].join('-'));
        });

        it('для 180days вернет ', function () {
            var today = new Date();
            today.setDate(today.getDate() - 179);

            dates.getPeriod('180days')
                .must.be([
                    today.getFullYear(),
                    addZero(today.getMonth() + 1, 2),
                    addZero(today.getDate(), 2)
                ].join('-'));
        });

        it('для 365days вернет ', function () {
            var today = new Date();
            today.setDate(today.getDate() - 364);

            dates.getPeriod('365days')
                .must.be([
                    today.getFullYear(),
                    addZero(today.getMonth() + 1, 2),
                    addZero(today.getDate(), 2)
                ].join('-'));
        });

        it('для thismonth вернет ', function () {
            var today = new Date();

            dates.getPeriod('thismonth').must.be(today.getFullYear() + '-' + addZero(today.getMonth() + 1, 2) + '-01');
        });

        it('для lastmonth вернет ', function () {
            var today = new Date();
            today.setMonth(today.getMonth() - 1);

            dates.getPeriod('lastmonth').must.be(today.getFullYear() + '-' + addZero(today.getMonth() + 1, 2) + '-01');
        });

        it('для thisyear вернет ', function () {
            var year = new Date().getFullYear();

            dates.getPeriod('thisyear').must.be(year + '-01-01');
        });

        it('для непредусмотренных случаев вернет undefined', function () {
            (typeof dates.getPeriod('asd') === 'undefined').must.be(true);
        });

        it('если указать интервал week вторым агрументом, то вернет дату в неделях', function () {
            dates.getPeriod('thismonth', 'week').must.match(/^\d{4}-\d{2}$/);
        });

        it('если указать интервал year вторым агрументом, то вернет дату в годах', function () {
            dates.getPeriod('thismonth', 'year').must.match(/^\d{4}$/);
        });
    });
});
