'use strict';
/*jshint maxlen:false*/
describe('utils/navigation', function () {
    var navigation = require('utils/navigation');
    var navigationSample = [
        {
            type: 'group',
            name: 'item1_1',
            url: '/item1_1',
            active: 1,
            items: [
                {
                    type: 'item',
                    name: 'item1_1/item2_1',
                    url: '/item1_1/item2_1'
                },
                {
                    type: 'item',
                    name: 'item1_1/item2_2',
                    url: '/item1_1/item2_2',
                    active: 1,
                    items: [{
                        type: 'sub-item',
                        name: 'item1_1/item2_1?deleted=1',
                        url: '/item1_1/item2_1?deleted=1',
                        active: 1
                    }]
                },
                {
                    type: 'item',
                    name: 'item1_1/item2_3?id=123',
                    url: '/item1_1/item2_3?id=123'
                },
                {
                    type: 'item',
                    name: 'item1_1/item2_4',
                    url: 'item1_1/item2_4'
                }
            ]
        },
        {
            type: 'group',
            name: 'item1_2',
            url: 'item1_2',
            items: [{
                name: 'item1_2/item2_2',
                url: '/item1_2/item2_2',
                type: 'item'
            }]
        }
    ];

    describe('getSelectedLevels()', function () {
        it('должна возвращать раскрытые уровни навигации', function () {
            var input = navigation.getTreeLeaves({items: navigationSample}, null);
            var actual = navigation.getSelectedLevels(input);
            var names = actual.map(function(level){
                return level.map(function (item) {
                    return item.name;
                });
            });

            names.must.be.eql([
                ['item1_1/item2_1', 'item1_1/item2_2', 'item1_1/item2_3?id=123', 'item1_1/item2_4'],
                ['item1_1', 'item1_2']
            ]);
        });
    });

    describe('getNode(type, name, url, cmd, items, parent, previous, next, active)', function () {
        describe('возвращенный результат', function () {
            it('должен иметь поля type, name, previous, next, parent', function () {
                var parent = navigation.getNode('group', 'Foo');
                var previous = navigation.getNode('group', 'Bar');
                var next = navigation.getNode('group', 'Qux');
                var actual = navigation.getNode('group', 'Baz', 'http://example.com/', 'example', parent, previous, next, false);

                actual.type.must.be.eql('group');
                actual.name.must.be.eql('Baz');
                actual.url.must.be.eql('http://example.com/');
                actual.cmd.must.be.eql('example');
                actual.previous.must.be.eql(previous);
                actual.next.must.be.eql(next);
                actual.parent.must.be.eql(parent);
                actual.active.must.be.eql(false);
            });
        });
    });

    describe('isLeaf()', function () {
        it('должна возвращать true, если переданный узел является листом', function () {
            var input = {type: 'item'};
            var actual = navigation.isLeaf(input);
            actual.must.be.eql(true);
        });

        it('должна возвращать false, если переданный узел не является листом', function () {
            var input = {type: 'group'};
            var actual = navigation.isLeaf(input);
            actual.must.be.eql(false);
        });
    });

    describe('getTreeLeaves()', function () {
        it('должна возвращать список листьев дерева с проставленными значениями previous, next и parent', function () {
            var actual = navigation.getTreeLeaves({items: navigationSample}, null);
            var names = actual.map(function (leaf) {
                return leaf.name;
            });

            names.must.be.eql([
                'item1_1/item2_3?id=123',
                'item1_1/item2_1',
                'item1_1/item2_2',
                'item1_2/item2_2',
                'item1_1/item2_4'
            ]);
        });
    });

    describe('getActiveLeaf()', function () {
        it('должна возвращать активнй лист дерева навигации', function () {
            var input = navigation.getTreeLeaves({items: navigationSample}, null);
            var actual = navigation.getActiveLeaf(input);
            actual.name.must.be.eql('item1_1/item2_2');
        });

        it('должна возвращать корректный лист дерева, если url соответствует sub-item`у листа', function () {
            var input = navigation.getTreeLeaves({items: navigationSample}, null);
            var actual = navigation.getActiveLeaf(input);

            actual.name.must.be.eql('item1_1/item2_2');
        });
    });

    describe('getTopSibling()', function () {
        it('должна возвращать первый узел на том же уровне, что и переданный', function () {
            var navLeaves = navigation.getTreeLeaves({items: navigationSample}, null);
            var leaf = navLeaves.filter(function (leaf) {return leaf.name === 'item1_1/item2_4';})[0];
            var actual = navigation.getTopSibling(leaf);

            actual.name.must.be.eql('item1_1/item2_1');
        });
    });

    describe('getLevelByNode()', function () {
        it('должна возвращать все узлы на том же уровне, что и переданный', function () {
            var leaves = navigation.getTreeLeaves({items: navigationSample}, null);
            var leaf = leaves.filter(function (leaf) {return leaf.name === 'item1_1/item2_4';})[0];
            var actual = navigation.getLevelByNode(leaf);
            var names = actual.map(function (leaf) {return leaf.name;});

            names.must.be.eql([
                'item1_1/item2_1',
                'item1_1/item2_2',
                'item1_1/item2_3?id=123',
                'item1_1/item2_4'
            ]);
        });
    });

    describe('getNavigationBemjson()', function () {
        it('должна возвращать одно меню типа primary без активных итемов, если не активных узлов', function () {
            var tree = [{
                type: 'item',
                name: 'first-level-item'
            }];

            navigation.getNavigationBemjson(tree).must.be.eql({
                primary: {
                    block: 'a-navigation',
                    mods: {type: 'primary'},
                    content: [{
                        block: 'a-navigation',
                        elem: 'item',
                        content: 'first-level-item',
                        url: null,
                        elemMods: {active: ''}
                    }]
                }
            });
        });

        it('должна возвращать одно меню типа primary для раздела на первом уровне вложенности', function () {
            var tree = [{
                type: 'item',
                name: 'first-level-item',
                active: 1
            }];

            navigation.getNavigationBemjson(tree).must.be.eql({
                primary: {
                    block: 'a-navigation',
                    mods: {type: 'primary'},
                    content: [{
                        block: 'a-navigation',
                        elem: 'item',
                        content: 'first-level-item',
                        url: null,
                        elemMods: {active: 'yes'}
                    }]
                }
            });
        });

        it('должна возвращать меню типа primary и меню tertiary для раздела на втором уровне вложенности', function () {
            var tree = [{
                type: 'group',
                name: 'first-level-group',
                active: 1,
                items: [{
                    type: 'item',
                    name: 'second-level-item',
                    active: 1
                }]
            }];

            navigation.getNavigationBemjson(tree).must.be.eql({
                primary: {
                    block: 'a-navigation',
                    mods: {type: 'primary'},
                    content: [{
                        block: 'a-navigation',
                        elem: 'item',
                        content: 'first-level-group',
                        url: null,
                        elemMods: {active: 'yes'}
                    }]
                },
                tertiary: {
                    block: 'a-navigation',
                    mods: {type: 'tertiary'},
                    content: [{
                        block: 'a-navigation',
                        elem: 'item',
                        content: 'second-level-item',
                        url: null,
                        elemMods: {active: 'yes'}
                    }]
                }
            });
        });

        it('должна возвращать меню primary и secondary для раздела на третьем уровне', function () {
            var tree = [{
                type: 'group',
                name: 'first-level-group',
                active: 1,
                items: [{
                    type: 'group',
                    name: 'second-level-group',
                    active: 1,
                    items: [{
                        type: 'item',
                        name: 'third-level-item',
                        active: 1
                    }]
                }]
            }];

            navigation.getNavigationBemjson(tree).must.be.eql({
                primary: {
                    block: 'a-navigation',
                    mods: {type: 'primary'},
                    content: [{
                        block: 'a-navigation',
                        elem: 'item',
                        content: 'first-level-group',
                        url: null,
                        elemMods: {active: 'yes'}
                    }]
                },
                secondary: {
                    block: 'a-navigation',
                    mods: {type: 'secondary'},
                    content: [{
                        block: 'a-navigation',
                        elem: 'group',
                        content: [
                            {block: 'a-navigation', elem: 'group-title',content: 'second-level-group'},
                            {block: 'a-navigation', elem: 'item', content: 'third-level-item', url: null, elemMods: {active: 'yes'}}
                        ]
                    }]
                }
            });
        });

        it('должна возвращать меню primary, secondary и tertiary для раздела на четвертом уровне', function () {
            var tree = [{
                type: 'group',
                name: 'first-level-group',
                active: 1,
                items: [{
                    type: 'group',
                    name: 'second-level-group',
                    active: 1,
                    items: [{
                        type: 'group',
                        name: 'third-level-group',
                        active: 1,
                        items: [
                            {
                                type: 'item',
                                name: 'fourth-level-item'
                            },
                            {
                                type: 'item',
                                name: 'one-more-fourth-level-item',
                                active: 1
                            }
                        ]
                    }]
                }]
            }];

            navigation.getNavigationBemjson(tree).must.be.eql({
                primary: {
                    block: 'a-navigation',
                    mods: {type: 'primary'},
                    content: [{
                        block: 'a-navigation',
                        elem: 'item',
                        content: 'first-level-group',
                        url: null,
                        elemMods: {active: 'yes'}
                    }]
                },
                secondary: {
                    block: 'a-navigation',
                    mods: {type: 'secondary'},
                    content: [{
                        block: 'a-navigation',
                        elem: 'group',
                        content: [
                            {block: 'a-navigation',elem: 'group-title', content: 'second-level-group'},
                            {block: 'a-navigation', elem: 'item', content: 'third-level-group', url: null, elemMods: {active: 'yes'}}
                        ]
                    }]
                },
                tertiary: {
                    block: 'a-navigation',
                    mods: {type: 'tertiary'},
                    content: [
                        {
                            block: 'a-navigation',
                            elem: 'item',
                            content: 'fourth-level-item',
                            url: null,
                            elemMods: {active: ''}
                        },
                        {
                            block: 'a-navigation',
                            elem: 'item',
                            content: 'one-more-fourth-level-item',
                            url: null,
                            elemMods: {active: 'yes'}
                        }
                    ]
                }
            });
        });

        it('должна возвращать dropdown, если есть список subitem`ов', function () {
            var tree = [{
                type: 'group',
                name: 'first-level-group',
                active: 1,
                items: [{
                    type: 'item',
                    name: 'second-level-item',
                    active: 1,
                    items: [{
                        type: 'subitem',
                        name: 'third-level-subitem',
                        active: 1
                    }]
                }]
            }];
            navigation.getNavigationBemjson(tree).must.be.eql({
                primary: {
                    block: 'a-navigation',
                    mods: {type: 'primary'},
                    content: [{
                        block: 'a-navigation',
                        elem: 'item',
                        content: 'first-level-group',
                        url: null,
                        elemMods: {active: 'yes'}
                    }]
                },
                tertiary: {
                    block: 'a-navigation',
                    mods: {type: 'tertiary'},
                    content: [{
                        block: 'a-navigation',
                        elem: 'item',
                        content: 'third-level-subitem',
                        url: null,
                        elemMods: {active: 'yes'},
                        subitems: [{
                            block: 'a-navigation',
                            elem: 'subitem',
                            content: 'third-level-subitem',
                            url: null
                        }]
                    }]
                }
            });
        });
    });

    describe('replaceActiveLeafName()', function () {
        it('должна заменять имя листа на имя активного сабитема', function () {
            var leafs = [
                {name: 'Create report'},
                {
                    name: 'Reports',
                    children: [
                        {name: 'This month'},
                        {name: 'This quarter', active: true},
                        {name: 'This year'}
                    ]
                }
            ];
            var result = navigation.replaceActiveLeafName(leafs);
            result.must.be.eql([
                {name: 'Create report'},
                {
                    name: 'This quarter',
                    children: [
                        {name: 'This month'},
                        {name: 'This quarter', active: true},
                        {name: 'This year'}
                    ]
                }
            ]);
        });
    });

    describe('getSecondaryBemjson()', function () {
        var tree = [{
            type: 'group',
            name: 'first-level-group',
            items: [{
                type: 'group',
                name: 'second-level-group',
                items: [{
                    type: 'group',
                    name: 'third-level-group',
                    items: [{
                        type: 'item',
                        name: 'fourth-level-item',
                        url: '/url'
                    }]
                }]
            }]
        }];

        navigation.getSecondaryBemjson(tree, '/url').must.be.eql({
            block: 'a-navigation',
            mods: {type: 'secondary'},
            content: [{
                block: 'a-navigation',
                elem: 'group',
                content: [
                    {block: 'a-navigation',elem: 'group-title', content: 'second-level-group'},
                    {block: 'a-navigation', elem: 'item', content: 'third-level-group', url: null, elemMods: {active: ''}}
                ]
            }]
        });
    });
});
