describe('utils/statistic', function () {
    'use strict';

    var datas = require('utils/statistic');
    var currencies;
    var data;
    var fields;
    var fieldsToRender;

    before(function () {
        currencies = [
            {id: '1', code: 'rub'},
            {id: '2', code: 'usd'},
            {id: '3', code: 'eur'}
        ];

        data = [
            {a: 3, b: 7, c: 'ah', date: '2014-05-01'},
            {a: 5, b: 3, c: 'uh', date: '2014-05-07'},
            {a: 9, b: 5, c: 'oh', date: '2014-05-03'},
            {a: 1, b: 7, c: 'hm', date: '2014-05-04'}
        ];

        fields = {
            a: {type: 'number'},
            b: {type: 'number'},
            c: {type: 'something'},
            date: {type: 'date'}
        };

        fieldsToRender = {
            dimension_fields: [
                {id: 'date', type: 'date'}
            ],
            fields: [
                {id: 'a', type: 'number'},
                {id: 'b', type: 'number'},
                {id: 'c', type: 'something'}
            ]
        };
    });

    describe('axisList()', function () {
        it('сформирует список с уникальными критериями на основе входных данных', function () {
            datas.axisList(data, ['a', 'b']).must.be.eql([
                {
                    a: 3,
                    b: 7
                },
                {
                    a: 5,
                    b: 3
                },
                {
                    a: 9,
                    b: 5
                },
                {
                    a: 1,
                    b: 7
                }
            ]);
        });
    });

    describe('fieldsList()', function () {
        it('вернет массив строк с названиями полей (тащит по id)', function () {
            datas.fieldsList(fieldsToRender.fields).must.be.eql([
                'a',
                'b',
                'c'
            ]);
        });
    });

    describe('fieldsToSort()', function () {
        it('получит список полей для сортировки', function () {
            datas.fieldsToSort(fieldsToRender).must.be.permutationOf([
                'a',
                'b',
                'c',
                'date'
            ]);
        });
    });

    describe('formatCurrency()', function () {
        it('рубли', function () {
            datas.formatCurrency('RUB').must.be('<span style="font-family: rub-arial-regular;">Р</span>');
        });

        it('доллары', function () {
            datas.formatCurrency('USD').must.be('$');
        });

        it('евро', function () {
            datas.formatCurrency('EUR').must.be('€');
        });
    });

    describe('formatValue()', function () {
        it('распарсит строку в число', function () {
            datas.formatValue('55').must.be(55);
        });

        it('если строка невалидная, то вернет 0', function () {
            datas.formatValue('abs').must.be(0);
        });

        it('если указан тип float, то округлит число до 4 знаком после запятой', function () {
            datas.formatValue('0.1257654', 'float').must.be(0.1258);
        });
    });

    describe('identify()', function () {
        it('идентифицирует объект и возвращает его айдишник', function () {
            var obj = {};
            var id = datas.identify(obj);

            obj._PId.must.be(id);
        });

        it('без указания объекта возвращает уникальный айдишник', function () {
            /^uniq[0-9]+$/.test(datas.identify()).must.be(true);
        });
    });

    describe('inverse()', function () {
        it('приведет выражение к булевому значению и инвертирует его', function () {
            datas.inverse(false).must.be(true);
        });
    });

    describe('numeric()', function () {
        it('распарсит строку в число', function () {
            datas.numeric('55').must.be(55);
        });

        it('если строка невалидная, то вернет 0', function () {
            datas.numeric('abs').must.be(0);
        });
    });

    describe('sortByFields()', function () {
        it('сортирует данные по перечисленным полям', function () {
            var d = data.slice();

            datas.sortByFields(d, ['b', 'date'], fields)
                .must.be.eql([
                    {a: 5, b: 3, c: 'uh', date: '2014-05-07'},
                    {a: 9, b: 5, c: 'oh', date: '2014-05-03'},
                    {a: 3, b: 7, c: 'ah', date: '2014-05-01'},
                    {a: 1, b: 7, c: 'hm', date: '2014-05-04'}
                ]);
        });
    });

    describe('valueableFields()', function () {
        it('вытащит список полей из свойства fields', function () {
            datas.valueableFields(fieldsToRender).must.be.permutationOf([
                'a',
                'b',
                'c'
            ]);
        });
    });

    describe('vocabulary()', function () {
        it('сформирует из массива объектов объект по заданному ключу', function () {
            datas.vocabulary(currencies, 'id').must.be.eql({
                1: {code: 'rub'},
                2: {code: 'usd'},
                3: {code: 'eur'}
            });
        });
    });
});
