package API::Method;

use qbit;

use base qw(QBit::HTTPAPI::Method);

use Compress::Zlib;

use Exception::BlackBox;
use Exception::BlackBox::NeedAuth;
use Exception::Denied;
use Exception::TooManyRequests;
use Exception::Validation::BadArguments;
use Exception::Validation::BadArguments::InvalidJSON;

use PiConstants qw($AUTHORIZATION_USER_FIELDS $SYSTEM_CRON_USER_ID);

__PACKAGE__->model_accessors(
    api_blackbox => 'QBit::Application::Model::API::Yandex::BlackBox',
    users        => 'Application::Model::Users',
);

sub MODIFY_CODE_ATTRIBUTES {
    my ($package, $sub, @attrs) = @_;

    my @unknown_attrs = ();

    foreach my $attr (@attrs) {
        if ($attr eq 'ALLOWHTTP') {
            $package->_set_method_attr($sub, ALLOWHTTP => TRUE);
        } elsif ($attr eq 'NOAUTH') {
            $package->_set_method_attr($sub, NOAUTH => TRUE);
        } elsif ($attr =~ /^CHECK_RIGHTS\s*\((.+)\)\s*$/) {
            my @rights = eval($1);

            $package->_set_method_attr($sub, check_rights => \@rights);
        } else {
            push(@unknown_attrs, $attr);
        }
    }

    throw "$package: CHECK_RIGHTS attribute is not given for method."
      unless grep {/^CHECK_RIGHTS\s*\((.+)\)\s*$/ || $_ eq 'NOAUTH'} @attrs;

    return $package->SUPER::MODIFY_CODE_ATTRIBUTES($sub, @unknown_attrs);
}

sub get_model_opts {
    my ($self, %params) = @_;

    throw Exception::Validation::BadArguments gettext("Parameter '%s' must be a whole positive number.", 'limit')
      if defined($params{'limit'}) && !($params{'limit'} =~ /^[0-9]+$/ && $params{'limit'} > 0);
    throw Exception::Validation::BadArguments gettext("Parameter '%s' must be a whole not negative number.", 'offset')
      if defined($params{'offset'}) && !($params{'offset'} =~ /^[0-9]+$/ && $params{'offset'} >= 0);
    throw Exception::Validation::BadArguments gettext("Parameter '%s' must be an array.", 'sort')
      unless ref($params{'sort'}) eq 'ARRAY';

    $params{'sortdesc'} = !!$params{'sortdesc'} if defined($params{'sortdesc'});

    my %model_opts;
    $model_opts{'fields'} = $params{'field'} if @{$params{'field'}};
    if (exists($params{'search_json'})) {
        try {
            $model_opts{'filter'} = from_json($params{'search_json'});
        }
        catch Exception::Validation::BadArguments::InvalidJSON with {
            throw Exception::Validation::BadArguments::InvalidJSON gettext("Invalid json in param '%s'\n%s",
                'search_json', $_[0]->message());
        };
    }
    $model_opts{'limit'} = $params{'limit'} if exists($params{'limit'});
    $model_opts{'offset'} = $params{'offset'} if exists($params{'limit'}) && exists($params{'offset'});
    $model_opts{'order_by'} = [[@{$params{'sort'}}, $params{'sortdesc'}]] if @{$params{'sort'}};

    return %model_opts;
}

sub on_error {
    my ($self, $method, $e) = @_;

    if ($e->isa('Exception::DB')) {
        $e->{'text'} = gettext('DB error');
    }
}

sub post_run {
    my ($self, $method, $params, $result) = @_;

    if (ref($result) ne 'SCALAR') {
        $result = \'streaming result';
    }

    $self->app->response->headers->{'X-Yandex-Login'} = $self->get_option(cur_user => {})->{login};
}

sub pre_run {
    my ($self, $method, $params) = @_;

    $self->SUPER::pre_run;

    if ($self->{'attrs'}{NOAUTH}) {
        $self->app->set_cur_user({id => $SYSTEM_CRON_USER_ID, login => 'system-cron'});
        return;
    }

    my $lang = $self->app->request->param('lang') || $self->app->request->http_header('Accept-Language') || 'ru';
    $self->app->set_app_locale($lang);

    my $token = $self->app->request->param('oauth_token') // $self->app->request->http_header('Authorization');

    throw Exception::Denied gettext('Need OAuth token') unless $token;
    $token =~ s/^OAuth(?:2)? //;

    my $bb;
    try {
        $bb = $self->api_blackbox->oauth(token => $token, userip => $self->app->request->remote_addr);
    }
    catch Exception::BlackBox::NeedAuth with {
        throw Exception::Denied gettext('Invalid token');
    }
    catch Exception::BlackBox with {
        throw Exception::Denied gettext('Authorization error');
    };

    throw Exception::Denied gettext('Application does not have enought permissions')
      unless grep {$_ eq 'pi:all'} split(/\s+/, $bb->{'OAuth'}{'scope'}{'content'});

    my $cur_user;
    {
        my $tmp_rights =
          $self->app->add_tmp_rights(qw(users_view_all users_view_field__multistate users_view_field__client_id));

        throw Exception::Denied gettext('Unknown user')
          unless $cur_user =
              $self->users->get_by_uid($bb->{'uid'}{'content'}, fields => $AUTHORIZATION_USER_FIELDS)
    }

    $cur_user->{'__OAuthInfo__'} = {map {$_ => $bb->{'OAuth'}{$_}{'content'}} keys(%{$bb->{'OAuth'}})};

    $self->app->set_cur_user($cur_user);

    $self->app->set_option(fake_login => $self->check_rights('fake_login'))
      if $self->get_option('allow_fake_login');

    if ($self->get_option('fake_login') && $self->app->request->param('fakelogin')) {
        my $fake_user =
          $self->users->get_by_login($self->app->request->param('fakelogin'), fields => $AUTHORIZATION_USER_FIELDS)
          // throw gettext('Bad fake login');
        $fake_user->{'fake'}      = TRUE;
        $fake_user->{'real_user'} = $cur_user;

        $self->app->set_cur_user($fake_user);
    }

    throw Exception::Denied gettext("You don't have access to this page")
      unless $self->check_rights(@{$self->{'attrs'}{'check_rights'}});
}

TRUE;
