package API::Method::Statistics2;

use qbit;

use base qw(API::Method);

use Exception::DB;
use Exception::Validation::BadArguments;
use Exception::TooManyRequests;

use PiConstants qw($MOL_STAT_TYPES $STAT_AVAILABLE_CURRENCIES);

__PACKAGE__->model_accessors(
    bk_statistics => 'Application::Model::BKStatistics',
    memcached     => 'QBit::Application::Model::Memcached',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'api_statistics2',
            description => d_gettext('Rights to use statistics api'),
            rights      => {api_statistics2 => d_gettext('Right to call statistics get hand'),},
        }
    ]
);

sub get : METHOD :
  PARAMS(!period[], !field[], limits{}, dimension_field[], entity_field[], filter, dimension_filter, order_by, vat, stat_type, currency)
  : CHECK_RIGHTS('api_statistics2') : TITLE('Get statistics') {
    my ($self, %opts) = @_;

    my $qv = $self->app->validator->check(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                period          => {type => 'array',  all      => {}},
                field           => {type => 'array',  all      => {}},
                limits          => {type => 'scalar', optional => TRUE},
                dimension_field => {type => 'array',  optional => TRUE, all => {}},
                entity_field    => {type => 'array',  optional => TRUE, all => {}},
                filter           => {optional => TRUE},
                dimension_filter => {optional => TRUE},
                order_by         => {optional => TRUE},
                vat              => {optional => TRUE},
                currency         => {
                    optional => TRUE,
                    in       => $STAT_AVAILABLE_CURRENCIES,
                    type     => 'scalar',
                },
                stat_type => {
                    optional => TRUE,
                    in       => $MOL_STAT_TYPES,
                    type     => 'scalar',
                },
            }
        },
    );

    if ($qv->has_errors) {
        throw Exception::Validation::BadArguments
          join('; ',
            map {join(', ', @{$_->{'msgs'}}) . ' — ' . ($_->{'path'}[0] // '')} $qv->get_fields_with_error());
    }

    my @filter;
    if (defined $opts{'filter'}) {
        my $f = from_json($opts{'filter'});
        if ((ref $f eq 'ARRAY') && (@$f == 2) && ($f->[0] eq 'AND') && (ref $f->[1] eq 'ARRAY')) {
            push @filter, @{$f->[1]};
        } else {
            push @filter, $f;
        }
    }

    my %statistics_opts = (
        hash_transform(
            \%opts,
            ['stat_type', 'currency'],
            {field => 'fields', dimension_field => 'dimension_fields', entity_field => 'entity_fields'}
        ),
    );

    $statistics_opts{levels} = [{filter => ['AND', \@filter]}] if @filter;

    my $period;
    # period compare
    if (@{$opts{'period'}} == 4) {
        $period = [[$opts{'period'}->[0], $opts{'period'}->[1]], [$opts{'period'}->[2], $opts{'period'}->[3]]];
    } elsif (@{$opts{'period'}} == 1) {
        $period = $opts{'period'}->[0];
    } else {
        $period = $opts{'period'};
    }

    $statistics_opts{'period'}           = $period;
    $statistics_opts{'dimension_filter'} = from_json($opts{'dimension_filter'}) if defined($opts{'dimension_filter'});
    $statistics_opts{'limits'}           = from_json($opts{'limits'}) if defined($opts{'limits'});
    $statistics_opts{'order_by'}         = from_json($opts{'order_by'}) if defined($opts{'order_by'});

    return $self->bk_statistics->get_statistics2(%statistics_opts);
}

sub tree : PARAMS(stat_type) : METHOD : CHECK_RIGHTS('api_statistics2') : TITLE('Get statistics levels tree') {
    my ($self, %opts) = @_;

    my $qv = $self->app->validator->check(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                stat_type => {
                    optional => TRUE,
                    'in'     => $MOL_STAT_TYPES,
                    'type'   => 'scalar',
                },
            }
        },
    );

    if ($qv->has_errors) {
        throw Exception::Validation::BadArguments
          join('; ', map {join(', ', @{$_->{'msgs'}}) . ' — ' . $_->{'path'}[0]} $qv->get_fields_with_error());
    }

    return $self->bk_statistics->get_tree2(%opts);
}

TRUE;
