package Application::Model::API::HTTPLogger;

use qbit;
use Compress::Zlib;
use Fcntl qw(:flock SEEK_END);
use Time::HiRes qw(usleep);

use base qw(QBit::Application::Model);

our $STORE_KEY = 'http_logger_mode';
my $STORE_NEXT  = 0;
my $STORE_FRESH = 3600;
my $STORE_VALUE;

sub need_log_all {TRUE}

sub is_force_log {
    my ($self) = @_;

    my $time = curdate(oformat => 'sec');
    if ($STORE_NEXT <= $time) {
        $STORE_VALUE = $self->app->kv_store->get($STORE_KEY);
        $STORE_NEXT  = $time + $STORE_FRESH;
    }

    return $STORE_VALUE;
}

sub need_log {
    my ($self, $data) = @_;

    return
         $self->is_force_log
      || $self->need_log_all
      || $data->{error}
      || ($data->{status} && $data->{status} ne '200');
}

sub log {
    my ($self, $data) = @_;

    return unless $self->need_log($data);

    $data->{'time'}        = format_date(curdate(), '%Y-%m-%d %H:%M:%S');
    $data->{'name_module'} = ref($self);
    $data->{'error'}       = $self->canonical_error_message($data->{'error'});

    my $mod_name = ref($self) || 'main';
    $mod_name =~ s/^Application::Model:://g;
    $mod_name =~ s/::/./g;

    my $sub_name = $data->{'method'};
    unless ($sub_name) {
        my @call_info = caller(4);
        $sub_name = $call_info[3] || 'unknown';
        $sub_name =~ s/^.*:://;
    }
    $data->{'method'} = $sub_name;

    my $log_fname = $self->get_option('api_logs') . "/$mod_name.$sub_name.log";
    $self->write_log($log_fname, $data);
}

sub write_log {
    my ($self, $log_fname, $data) = @_;

    my $fh;
    open $fh, ">>", $log_fname or throw "can't open log file $log_fname";
    my $i = 0;
    while (not(flock $fh, LOCK_EX | LOCK_NB)) {
        throw "can't obtain lock $log_fname" if $i++ > 10000;    # 10sec
        usleep 1000;                                             # one millisecond
    }
    seek $fh, 0, SEEK_END;
    print $fh to_json($data) . "\n";
    close $fh;

    return;
}

sub canonical_error_message {
    my ($self, $error) = @_;

    if (!defined($error)) {
        return '';
    } elsif (blessed($error) && $error->isa('Exception')) {
        return $error->message();
    } else {
        return "$error";
    }
}

TRUE;
