package Application::Model::API::Yandex::AdFox;

use qbit;

use Digest::SHA;
use POSIX qw(floor);
use URI;

use base qw(QBit::Application::Model::API::HTTP Application::Model::API::HTTPLogger);

use Exception::API::AdFox;
use Exception::Validation::BadArguments;

=encoding UTF-8
=head1 get_adfox_user_info

Метод запрашивает информацию о пользователе AdFox у сервиса AdFox.

Пример возвращаемой структуры:

    {
      "userInfo": {
         "login": "test-01",
         "firstName": "John",
         "lastName": "Millow"
       }
    }

=cut

sub accessor {'api_adfox'}

# Overriding 'call' method here because we need a fixed order of url parameters for AdFox.
# AdFox lines up parameters in a certain way before checking the signature.
# We need to have parameters in the same order before signing.
sub call {
    my ($self, $path) = @_;

    my $uri = $self->get_signed_adfox_uri($path);
    return $self->get($uri);
}

sub check_signature {
    my ($self, $uri) = @_;

    my $uri_obj = URI->new($uri);
    my $path    = $uri_obj->path_query();
    my $time    = time();
    my $sign;

    # Here we believe that sign=... is always the last query parameter.
    # This is how signed urls are created at AdFox.
    # BUT passport can add extra parameters to the url after authorization so we ignore everything after sign=
    #
    ($path, $sign) = $path =~ /^(.*)[&?]sign=([^&]*)(?:$|&)/;

    $path //= '';
    $sign //= '';

    # Check against current time (in full hours) and against 1 hour ago.
    return ( $sign eq $self->get_signature($path, $time)
          || $sign eq $self->get_signature($path, $time - (60 * 60)) ? 1 : 0);
}

sub get_adfox_user_info {
    my ($self, %opts) = @_;

    throw Exception::API::AdFox gettext("Incorrect adfox id '%s'", ($opts{'adfox_id'} // 'undef'))
      if $opts{'adfox_id'} !~ /^[0-9]+$/;

    my $path = '/exportData.php?action=getUserInfo&adfox_id=' . $opts{'adfox_id'};
    my $json = $self->call($path);
    my $data = from_json($json);

    throw Exception::API::AdFox gettext("AdFox answer after decoding is not a hashref. JSON:\n %s", $json)
      unless ref($data) eq 'HASH';

    if ($data->{'error'}) {
        throw Exception::API::AdFox $data->{'error'}->{'msg'};
    }

    return $data->{'userInfo'};
}

sub get_signature {
    my ($self, $path, $time) = @_;

    my $salt = $self->get_option('salt');
    return Digest::SHA::hmac_sha256_hex(floor($time / (60 * 60)) . '|' . ($path // ''), $salt);
}

sub get_signed_adfox_uri {
    my ($self, $path) = @_;

    my $uri = $self->get_option('url');    # AdFox site
    throw Exception::Validation::BadArguments gettext("Url in config must not end with '/'") if $uri =~ /\/$/;

    my $sign = $self->get_signature($path, time());
    $uri .= ($path =~ /\?/ ? "$path&sign=$sign" : "$path?sign=$sign");

    return $uri;
}

TRUE;
