package Application::Model::API::Yandex::AdFoxGraphQL;

use qbit;
use base qw(QBit::Application::Model::API::HTTP Application::Model::API::HTTPLogger);

use Exception::API::AdFoxGraphQL;
use Exception::Validation::BadArguments::InvalidJSON;
use Exception::Validator::Fields;

use PiConstants qw(
  @ADFOX_PRESET_PAIDPRODUCTS
  @ADFOX_PRESET_ALL
  );

my %APP_TYPES = (
    1 => 'ANDROID',
    2 => 'IOS',
);
my $DEFAULT_MOBILE_APP_CAPTION  = 'Mobile app campaign';
my $DEFAULT_API_PATH            = 'api/internal/graphql/';
my $CREATE_USER_GRAPHQL_REQUEST = _req('
mutation ($yandexUid: Int64!, $mobileApps: [AnketaMobileAppInput!], $presets: [RegistrationPreset!]){
    %s {
        %s(params: {
            yandexUid: $yandexUid,
            mobileApps: $mobileApps
        },
        presets: $presets
        ) {
            id
            login
        }
    }
}' => AdfoxInternal => 'registerUser');

my $UPDATE_MOBILE_APP_STATUS_GRAPHQL_REQUEST = _req('
mutation($appId:Int!, $isActive:Boolean!){
    %s{
        %s(piAppId: $appId, isActive: $isActive)
    }
}' => AdfoxInternal => 'updateMobileAppStatus');

my $GET_BILLING_CLIENT_INFO_GRAPHQL_REQUEST = _req('
query ($adfoxAccount: String!, $passportUid: Int64) {
    %s {
        %s(adfoxAccount: $adfoxAccount, passportUid: $passportUid) {
            billingClientId,
            billingContractId,
            hasOtherAccounts,
            hasPaidServices,
            piAccounts,
            complexPiLinks,
            inWhitelist,
            needCloseContract,
            userId,
            hasSameUidInOtherAccount
        }
    }
}' => AdfoxInternal => 'billingClientInfo');

my $BIND_USER_GRAPHQL_REQUEST = _req('
mutation ($adfoxAccount: NonEmptyString!, $passportUid: Int64!, $presets: [RegistrationPreset!]) {
    %s {
        %s(adfoxAccount: $adfoxAccount, passportUid: $passportUid, presets: $presets) {
            id,
            login
        }
    }
}' => AdfoxInternal => 'bindClient');

my $SET_PRESETS_GRAPHQL_REQUEST = _req('
mutation ($adfoxUserId: Int!, $presets: [RegistrationPreset!]) {
    %s {
        %s(adfoxUserId: $adfoxUserId, presets: $presets)
    }
}' => AdfoxInternal => 'setPresets');

my $BIND_CLIENT_WBILLING_GRAPHQL_REQUEST = _req('
mutation ($adfoxAccount: NonEmptyString!, $passportUid: Int64!, $presets: [RegistrationPreset!], $billingClientId: Int!, $billingContractId: Int!) {
    %s {
        %s(
            adfoxAccount: $adfoxAccount,
            passportUid: $passportUid,
            presets: $presets,
            billingData: {
                billingClientId: $billingClientId,
                billingContractId: $billingContractId
            },
        ) {
            id,
            login
        }
    }
}' => AdfoxInternal => 'bindClient');

my $SET_BILLING_DATA_GRAPHQL_REQUEST = _req('
mutation ($passportUid: Int64!, $billingClientId: Int!, $billingContractId: Int!, $presets: [RegistrationPreset!]) {
  %s{
    %s(
        passportUid: $passportUid,
        presets: $presets,
        billingData: {
            billingClientId: $billingClientId,
            billingContractId: $billingContractId
        }
    )
  }
}' => AdfoxInternal => 'setBillingData');

my $UPDATE_PAGE_NAME_GRAPHQL_REQUEST = _req('
mutation ($pageId: Int!, $name: String!) {
    %s {
        %s(pageId: $pageId, name: $name) { name }
    }
}' => InternalMobileMediation => 'updatePageName');

my $UPDATE_BLOCK_NAME_GRAPHQL_REQUEST = _req('
mutation ($placeId: Int!, $name: String!) {
    %s {
        %s(placeId: $placeId, name: $name ) { name }
    }
}' => InternalMobileMediation => 'updatePlaceName');

my $DELETE_BLOCK_GRAPHQL_REQUEST = _req('
mutation ($placeIds: [Identifier!]!) {
    %s {
        %s(placeIds: $placeIds)
    }
}' => InternalMobileMediation => 'deletePlace');

my $START_STOP_PAGE_GRAPHQL_REQUEST = _req('
mutation ($pageId: Int!, $isActive: Boolean!) {
    %s {
        %s(pageId: $pageId, isActive: $isActive) { updatePageIdStatus }
    }
}' => InternalMobileMediation => 'updatePageIdStatus');

my %call_mocked_data = (
    $CREATE_USER_GRAPHQL_REQUEST->[0] => sub {
        my ($data) = @_;

        return to_json {
            data => {
                AdfoxInternal => {
                    registerUser => {
                        id    => 31415926,
                        login => 'mocked_adfox_login',
                    },
                },
            },
        };
    },
    $BIND_USER_GRAPHQL_REQUEST->[0] => sub {
        my ($data) = @_;

        return to_json {
            data => {
                AdfoxInternal => {
                    bindClient => {
                        id    => 31415926,
                        login => $data->{adfoxAccount},
                    },
                },
            },
        };
    },
    $SET_PRESETS_GRAPHQL_REQUEST->[0] => sub {
        my ($data) = @_;

        return to_json {data => {AdfoxInternal => {setPresets => \1,},},};
    },
);

sub accessor {'api_adfox_graphql'}

sub call {
    my ($self, %opts) = @_;

    my $check_fields = delete $opts{check_fields};
    my $check        = delete $opts{check};
    my ($query, $type_name, $field_name) = @{delete $opts{query}};
    my $variables = delete $opts{variables};
    $opts{':post'}                    = 1;
    $opts{':headers'}{'Content-Type'} = 'application/json';
    $opts{':content'}                 = to_json(
        {
            'query'     => $query,
            'variables' => $variables,
        }
    );

    my $response_str;
    if (   $self->get_option('stage', '') ne 'production'
        && $self->get_option('mock')
        && exists $call_mocked_data{$query})
    {
        $response_str = $call_mocked_data{$query}->($variables);
    }
    $response_str //= $self->SUPER::call($DEFAULT_API_PATH, %opts,);
    my $response;
    try {
        $response = from_json($response_str);
    }
    catch Exception::Validation::BadArguments::InvalidJSON with {
        my ($exception) = @_;
        throw Exception::API::AdFoxGraphQL($exception, undef, undef,
            sentry => {extra => {adfox_response => $response_str,},});
    };

    throw Exception::API::AdFoxGraphQL(
        join(' ', map {$_->{'message'} // 'undef'} @{$response->{'errors'}}),
        undef, undef,
        adfox_error_category =>
          join('_', ($response->{'errors'}[0]{category} // ''), ($response->{'errors'}[0]{code} // '')),
        sentry => {extra => {adfox_response => $response,},},
    ) if (ref($response->{'errors'}) eq 'ARRAY' && @{$response->{'errors'}});

    my $info = $response->{data}{$type_name}{$field_name};

    throw Exception::API::AdFoxGraphQL('Incorrect answer',
        undef, undef, sentry => {extra => {adfox_response => $response,},})
      if (
        !defined $info
        || $check_fields && (ref $info ne 'HASH'
            || grep {!defined $info->{$_}} @{$check_fields})
         );

    throw Exception::API::AdFoxGraphQL('Incorrect answer',
        undef, undef, sentry => {extra => {adfox_response => $response,},})
      if (ref $check eq 'CODE' && $check->($info));

    return $info;
}

# Ручка создания пользователя AdFox (изначально для mobile mediation, теперь ещё и для common offer),
# с возможностью регистрации мобильных приложений
# (сейчас без приложений, иначе возникают транзакционные коллизии между ПИ и AdFox)
#
# На вход принимает обязательный идентификатор пользователя (паспортовый),
# набор пресетов и необязательный список приложений
#
# Возвращает структуру, содержащую идентификатор и логин созданного пользователя AdFox

sub create_user {
    my ($self, %opts) = @_;

    $self->app->validator->check(
        'data'     => \%opts,
        'template' => {
            'extra'  => TRUE,
            'fields' => {
                'mobile_apps' => {
                    'all' => {
                        'check' => sub {
                            my ($qv, $m_app) = @_;

                            my $type     = $m_app->{'type'};
                            my $store_id = $m_app->{'store_id'};

                            throw Exception::Validator::Fields gettext('Invalid Bundle ID "%s"', $store_id)
                              unless get_bundle_id($store_id, $type);
                        },
                        'fields' => {
                            'store_id' => {
                                'len_min' => 3,
                                'type'    => 'scalar',
                            },
                            'store_url' => {
                                'len_min' => 4,
                                'type'    => 'scalar',
                            },
                            'type' => {
                                'in'   => [sort keys %APP_TYPES],
                                'type' => 'scalar',
                            },
                        },
                        'type' => 'hash',
                    },
                    'optional' => TRUE,
                    'size_min' => 0,
                    'type'     => 'array',
                },
                'user_id' => {'type' => 'int_un'},
                'presets' => {
                    'all' => {
                        'in'   => \@ADFOX_PRESET_ALL,
                        'type' => 'scalar',
                    },
                    'type' => 'array',
                },
            },
            'type' => 'hash',
        },
        'throw' => TRUE,
    );

    my $user_id     = delete $opts{user_id};
    my $mobile_apps = delete $opts{mobile_apps};
    my $presets     = delete $opts{presets};

    return $self->call(
        %opts,
        check_fields => [qw(id login)],
        query        => $CREATE_USER_GRAPHQL_REQUEST,
        variables    => {
            yandexUid  => "$user_id",    # to str PI-13203
            presets    => $presets,
            mobileApps => [
                map {
                    {
                        storeId  => $_->{store_id},
                        storeUri => $_->{store_url},
                        appName  => $DEFAULT_MOBILE_APP_CAPTION,
                        platform => $APP_TYPES{$_->{type}},
                    }
                  } @$mobile_apps
            ],
        },
    );
}

# Ручка актуализации состояния мобильного приложения в AdFox
# с возможностью регистрации мобильных приложений
# (сейчас без приложений, иначе возникают транзакционные коллизии между ПИ и AdFox)
#
# На вход принимает идентификатор приложения и флаг состояния

sub update_mobile_app_status {
    my ($self, %opts) = @_;

    my $active = delete $opts{active} ? \1 : \0;
    my $app_id = delete $opts{app_id};

    $self->call(
        %opts,
        check     => sub {$_[0] xor $$active},
        query     => $UPDATE_MOBILE_APP_STATUS_GRAPHQL_REQUEST,
        variables => {
            isActive => $active,
            appId    => int $app_id,
        },
    );

    return TRUE;
}

# Ручка получения биллинговой информации пользователя AdFox
#
# На вход принимает логин и пароль польщователя AdFox
#
# Возвращает структуру, содержащую параметры:
#    * billingClientId   - client_id пользователя AdFox в балансе
#    * billingContractId - номер договора
#    * hasOtherAccounts  - флаг того, что на этом договоре висят ещё другие пользователи AdFox
#    * hasPaidServices   - флаг того, что подключены платные услуги
#    * piAccounts        - список аккаунтов ПИ, к которому привязан данный аккаунтов
#    * complexPiLinks    - флаг того, что этот пользователь AdFox связан с другим
#                          пользователем AdFox посредством связи с пользователем ПИ
#    * inWhitelist       - пользователю разрешено заполнять единую оферту
#    * needCloseContract - флаг того, что необходимо закрывать договор
#    * userId            - идентификатор пользователя AdFox
#    * otherAccountWithSameUid - yandex_uid, с которым связан данный аккаунт AdFox, если он не равен переданному uid

sub get_billing_client_info {
    my ($self, %opts) = @_;

    @_ = ($self, '[FILTERED]');    # Защита от сохранения пароля в стектрейсе sentry

    my $login = delete $opts{adfox_login};
    my $psw   = uri_escape(delete $opts{adfox_psw} // '');
    my $uid   = delete $opts{uid};

    $opts{':headers'}{'Authorization'} = "password $psw";

    return $self->call(
        %opts,
        check_fields => [
            qw(billingClientId billingContractId hasOtherAccounts hasPaidServices piAccounts complexPiLinks inWhitelist needCloseContract userId hasSameUidInOtherAccount)
        ],
        query     => $GET_BILLING_CLIENT_INFO_GRAPHQL_REQUEST,
        variables => {adfoxAccount => $login, ($uid ? (passportUid => $uid) : ())},
    );
}

# Ручка связывания пользователей ПИ и AdFox
#
# На вход принимает обязательный идентификатор пользователя (паспортовый),
# набор пресетов и логин пользователя AdFox
#
# Возвращает структуру, содержащую идентификатор и логин связанного пользователя AdFox

sub bind_user {
    my ($self, %opts) = @_;

    $self->app->validator->check(
        'data'     => \%opts,
        'template' => {
            'fields' => {
                'adfox_login' => {
                    'len_min' => 1,
                    'type'    => 'scalar',
                },
                'presets' => {
                    'all' => {
                        'in'   => \@ADFOX_PRESET_ALL,
                        'type' => 'scalar',
                    },
                    'type' => 'array',
                },
                'user_id'      => {'type' => 'int_un',},
                'billing_data' => {
                    'optional' => TRUE,
                    'type'     => 'hash',
                    'fields'   => {
                        'client_id'   => {'type' => 'int_un',},
                        'contract_id' => {'type' => 'int_un',},
                    }
                },
            },
            'type' => 'hash',
        },
        'throw' => TRUE,
    );

    my $login        = delete $opts{adfox_login};
    my $user_id      = delete $opts{user_id};
    my $presets      = delete $opts{presets};
    my $billing_data = delete $opts{billing_data};

    return $self->call(
        %opts,
        check_fields => [qw(id login)],
        query        => $billing_data ? $BIND_CLIENT_WBILLING_GRAPHQL_REQUEST : $BIND_USER_GRAPHQL_REQUEST,
        variables    => {
            adfoxAccount => $login,
            passportUid  => "$user_id",    # to str
            presets      => $presets,
            (
                $billing_data
                ? (
                    billingClientId   => $billing_data->{client_id} + 0,
                    billingContractId => $billing_data->{contract_id} + 0,
                  )
                : ()
            ),
        },
    );
}

# Ручка выставления пользователю AdFox набора пресетов
#
# На вход принимает идентификатор пользователя AdFox и набор пресетов

sub set_presets {
    my ($self, %opts) = @_;

    $self->app->validator->check(
        'data'     => \%opts,
        'template' => {
            'fields' => {
                'adfox_id' => {'type' => 'int_un',},
                'presets'  => {
                    'all' => {
                        'in'   => \@ADFOX_PRESET_ALL,
                        'type' => 'scalar',
                    },
                    'type' => 'array',
                },
            },
            'type' => 'hash',
        },
        'throw' => TRUE,
    );

    my $adfox_id = delete $opts{adfox_id};
    my $presets  = delete $opts{presets};

    return $self->call(
        %opts,
        check     => sub {!$_[0]},
        query     => $SET_PRESETS_GRAPHQL_REQUEST,
        variables => {
            adfoxUserId => $adfox_id + 0,    # to int
            presets     => $presets,
        },
    ) ? TRUE : FALSE;
}

=head3
    Ручка связывания пользователя AdFox и свежесозданного договора на платные услуги AdFox

    $api->set_billing_data(client_id => 111, contract_id => 111, user_id => 111);

    На вход принимает обязательные: ID клиента Баланса, ID договора, ID пользователя (паспортовый)

    Возвращает smth
=cut

sub set_billing_data {
    my ($self, %opts) = @_;

    $self->app->validator->check(
        'data'     => \%opts,
        'template' => {
            'fields' => {
                'user_id'      => {'type' => 'int_un',},
                'billing_data' => {
                    'type'   => 'hash',
                    'fields' => {
                        'client_id'   => {'type' => 'int_un',},
                        'contract_id' => {'type' => 'int_un',},
                    }
                },
                'presets' => {
                    'all' => {
                        'in'   => \@ADFOX_PRESET_ALL,
                        'type' => 'scalar',
                    },
                    'type'     => 'array',
                    'optional' => TRUE
                },
            },
            'type' => 'hash',
        },
        'throw' => TRUE,
    );

    my $user_id      = delete $opts{user_id};
    my $billing_data = delete $opts{billing_data};
    my $presets      = delete $opts{presets};

    return $self->call(
        %opts,
        query     => $SET_BILLING_DATA_GRAPHQL_REQUEST,
        variables => {
            billingClientId   => $billing_data->{client_id} + 0,
            billingContractId => $billing_data->{contract_id} + 0,
            passportUid       => "$user_id",                                # to str
            presets           => $presets // \@ADFOX_PRESET_PAIDPRODUCTS,
        },
    );
}

sub update_page_name {
    my ($self, %opts) = @_;

    my ($page_id, $new_name) = delete @opts{qw(page_id new_name)};

    $self->call(
        %opts,
        query     => $UPDATE_PAGE_NAME_GRAPHQL_REQUEST,
        variables => {
            pageId => $page_id + 0,
            name   => $new_name,
        },
    );
}

sub update_block_name {
    my ($self, %opts) = @_;

    my ($place_id, $new_name) = delete @opts{qw(place_id new_name)};

    $self->call(
        %opts,
        query     => $UPDATE_BLOCK_NAME_GRAPHQL_REQUEST,
        variables => {
            placeId => $place_id + 0,
            name    => $new_name,
        }
    );
}

sub delete_block {
    my ($self, %opts) = @_;

    my $place_id = delete $opts{place_id};

    $self->call(
        %opts,
        query     => $DELETE_BLOCK_GRAPHQL_REQUEST,
        variables => {placeIds => [$place_id + 0]},
    );
}

sub start_stop_page {
    my ($self, %opts) = @_;

    my ($page_id, $is_started) = delete @opts{qw(page_id is_started)};

    $self->call(
        %opts,
        query     => $START_STOP_PAGE_GRAPHQL_REQUEST,
        variables => {
            pageId   => $page_id + 0,
            isActive => $is_started ? \1 : \0,
        },
    );
}

sub _req {
    my ($request, $type, $field) = @_;
    return [sprintf($request, $type, $field), $type, $field];
}

1;
