package Application::Model::API::Yandex::AdFoxRTD;

use qbit;
use Hash::Util qw(lock_keys);
use Utils::Logger qw(INFO);

use base qw(QBit::Application::Model::API::HTTP Application::Model::API::HTTPLogger);

use Exception::Validation::BadArguments;
use Exception::API::AdFoxRTD;

=encoding UTF-8

=head1 DESCRIPTION

L<https://wiki.yandex-team.ru/adfox/components/rtd/>

AdFoxRTD (Report Task Daemon) - это API для получения каких-либо данных из Adfox,
реализованное с помощью очереди.

Сначала нужно поставить задачу и получить task_id, потом нужно ходить в ручку
и выяснять статус этой задачи, а когда задача перейдет в статус SUCCESS, то
с помощью еще одной ручки можно получить результат.

=cut

my %ALL_STATES = map {$_ => 1} qw(
  PENDING
  RECEIVED
  STARTED
  SUCCESS
  FAILURE
  REVOKED
  RETRY
  );

sub accessor {'api_adfox_rtd'}

=head2 add_task_builder_basic

    my $task_id = $app->api_adfox_rtd->builder_basic( $params );

=cut

sub add_task_builder_basic {
    my ($self, $params) = @_;

    return $self->_call(
        method => 'builder_basic',
        params => $params,
    );
}

=head2 get_task_state

    my $task_state = $app->api_adfox_rtd->get_task_state('1362355c-afa3-4656-91a0-a62922e5d229'); # 'SUCCESS'

=cut

sub get_task_state {
    my ($self, $task_id) = @_;

    my $state = $self->_call(
        method => 'get_task_state',
        params => [$task_id],
    );

    if (!exists($ALL_STATES{$state})) {
        my $url = $self->get_option('url');
        throw Exception::API::AdFoxRTD "Unknown state '$state' from $url get_task_state";
    }

    return $state;
}

=head2 get_task_result

    my $task_result = $app->api_adfox_rtd->get_task_result('1362355c-afa3-4656-91a0-a62922e5d229');

=cut

sub get_task_result {
    my ($self, $task_id, $with_error) = @_;

    return $self->_call(
        ($with_error ? (error => TRUE,) : ()),
        method => 'get_task_result',
        params => [$task_id],
    );
}

=head2 add_task_and_get_result

    my $result = $app->api_adfox_rtd->add_task_and_get_result(
        method => 'slow_queue_builder_basic',
        params => {
            select => [
                'week_id',
            ],
            where => {
                date => [
                    '2018-09-05',
                    '2018-09-05',
                ],
            },
        },
        retries => 10,
    );

=cut

sub add_task_and_get_result {
    my ($self, %opts) = @_;

    my $method  = $opts{method};
    my $params  = $opts{params};
    my $retries = $opts{retries},

      my $task_id = $self->_call(
        method => $method,
        params => $params,
      );

    INFO "Created Task ID: $task_id";

    my $success    = 0;
    my $task_state = '';
    my $i          = 0;

    while ($i < $retries) {
        $i++;

        $task_state = $self->get_task_state($task_id);

        INFO "Got Task ID $task_id state: $task_state";

        if ($task_state eq 'SUCCESS') {
            $success = 1;
            last;
        }

        if ($task_state eq 'FAILURE') {
            last;
        }

        sleep 1;
    }

    if ($success) {
        my $task_result = $self->get_task_result($task_id);
        return $task_result;
    } else {
        my %extra = (
            last_state => $task_state,
            retries    => $i,
            task_id    => $task_id,
        );
        try {
            $extra{error} = $self->get_task_result($task_id, TRUE);
        }
        catch {
            # Поймать
            my ($e) = @_;

            # Добавить инфу
            $e->{sentry}{extra}{add_task_and_get_result} = \%extra;

            # Отправить дальше
            throw $e;
        };
        throw Exception::API::AdFoxRTD
          "After $i retries can't get result for Task ID $task_id. Last state: $task_state",
          undef, undef,
          sentry => {extra => {add_task_and_get_result => \%extra},};
    }

}

=head2 get_mobile_mediation_stat

    my $stat = $app->api_adfox_rtd->get_mobile_mediation_stat( '2020-01-21' );

=cut

sub get_mobile_mediation_stat {
    my ($self, $date) = @_;

    # Пример данных
    #
    #    {
    #        correct => 1,
    #        date => "2020-01-21",

    #        monetizer_id => 18,
    #        monetizer_name => "MoPub",

    #        rsya_block_id_mm => "R-M-499075-1",
    #        potential_impressions_mm => 2,
    #        impressions_mm => 0,

    #        calculated_revenue_by_event_original_1000000_mm => 0,
    #        calculated_revenue_by_event_1000000_mm => 0,
    #        campaign_cpm_currency_name => "RUB"
    #    }

    my %fields;
    my $total_data;
    my $data;
    foreach my $hour (0 .. 23) {
        $data = $self->add_task_and_get_result(
            method => 'slow_queue_builder_basic',
            params => {
                select => [
                    'date',                                            'monetizer_id',
                    'monetizer_name',                                  'rsya_block_id_mm',
                    'potential_impressions_mm',                        'impressions_mm',
                    'calculated_revenue_by_event_original_1000000_mm', 'calculated_revenue_by_event_1000000_mm',
                    'campaign_cpm_currency_name',                      'geo_id',
                ],
                where => {
                    date                => [$date, $date,],
                    hour                => $hour,
                    is_mobile_mediation => 1,
                },
                having => [
                    {
                        field    => 'potential_impressions_mm',
                        operator => '>',
                        value    => 0,
                    }
                ],
            },
            retries => 100,
        );

        my $i = 0;
        %fields = map {$_ => $i++} @{$data->{fields}} unless %fields;

        foreach my $row (@{$data->{table}}) {
            my $key = join '\t',
              (
                $row->[$fields{date}],
                $row->[$fields{monetizer_id}],
                $row->[$fields{monetizer_name}],
                $row->[$fields{rsya_block_id_mm}],
                $row->[$fields{campaign_cpm_currency_name}],
                $row->[$fields{geo_id}],
              );

            if (defined $total_data->{$key}) {
                my $current = $total_data->{$key};
                $current->[$fields{correct}] = 0 unless $row->[$fields{correct}];
                foreach (
                    qw(
                    potential_impressions_mm
                    impressions_mm
                    calculated_revenue_by_event_original_1000000_mm
                    calculated_revenue_by_event_1000000_mm
                    )
                  )
                {
                    $current->[$fields{$_}] += $row->[$fields{$_}];
                }
                $total_data->{$key} = $current;
            } else {
                $total_data->{$key} = $row;
            }

        }
    }

    $data->{table} = [values %$total_data];
    return $data;
}

sub _call {
    my ($self, %opts) = @_;

    my $method     = $opts{method};
    my $params     = $opts{params};
    my $need_error = $opts{error};

    my $response_str = $self->call(
        '',
        ':post'    => 1,
        ':content' => to_json(
            {
                id      => 1,
                jsonrpc => '2.0',
                method  => $method,
                params  => $params,
            }
        ),
    );

    my $response;
    try {
        $response = from_json($response_str);
    }
    catch Exception::Validation::BadArguments::InvalidJSON with {
        my ($exception) = @_;
        throw Exception::API::AdFoxRTD $exception, undef, undef, sentry => {extra => {response_str => $response_str,},};
    };

    unless (ref($response) eq 'HASH'
        && ($need_error ? exists($response->{error}) : exists($response->{result})))
    {
        my $url = $self->get_option('url');
        throw Exception::API::AdFoxRTD "Unexpected answer from $url $method: $response_str", undef, undef,
          sentry => {extra => {response => $response, response_str => $response_str,},};
    }

    return $need_error ? $response->{error} : $response->{result};
}

TRUE;
