package Application::Model::API::Yandex::Balalayka;

use qbit;

use base qw(QBit::Application::Model::API::HTTP Application::Model::API::HTTPLogger);

use Exception::API::Payoneer;
use Exception::API::Payoneer::Blocked;
use Exception::API::Payoneer::CannotRegister;
use Exception::API::Payoneer::Inactive;
use Exception::API::Payoneer::InvalidURL;
use Exception::API::Payoneer::InvalidVerb;
use Exception::API::Payoneer::NotEnabled;
use Exception::API::Payoneer::PayeeNotFound;
use Exception::API::Payoneer::Prohibited;
use Exception::API::Payoneer::Unauthorized;
use Exception::API::Payoneer::Validation;

sub accessor {'api_balalayka'}

my @PAYONEER_LOGIN_LINK_OPTIONS = qw(client_session_id redirect_url redirect_time language_id);
my @NUMERIC_OPTIONS             = qw(redirect_time language_id);
my %PAYONEER_CURRENCIES         = map {$_ => TRUE,} qw(EUR USD);
my %ERRORS                      = (
    10000 => 'Exception::API::Payoneer::Unauthorized',
    10001 => 'Exception::API::Payoneer::Prohibited',
    10002 => 'Exception::API::Payoneer::Validation',
    10003 => 'Exception::API::Payoneer::InvalidURL',
    10004 => 'Exception::API::Payoneer::InvalidVerb',
    10005 => 'Exception::API::Payoneer::PayeeNotFound',
    10006 => 'Exception::API::Payoneer::NotEnabled',
    10007 => 'Exception::API::Payoneer::Blocked',
    10009 => 'Exception::API::Payoneer::Inactive',
    10111 => 'Exception::API::Payoneer::CannotRegister',
);

sub call {
    my ($self, $method, %opts) = @_;

    my $debug = $self->get_option('debug');
    $opts{':headers'}{'Content-Type'} = 'application/json';

    my $response;
    try {
        my $content = $self->SUPER::call($method, %opts);
        $response = from_json $content;
        ldump $response if $debug;
    }
    catch Exception::API::HTTP with {
        my ($e) = @_;

        my $response = $e->{response};
        ldump $response if $debug;
        my $content = $response->decoded_content();
        my $data    = eval {from_json $content} // {};
        my $error   = $data->{errors}[0];
        my $ex_name = $ERRORS{$error->{code} // 0};
        if ($ex_name) {
            throw $ex_name->new($response, $error->{description}, $error->{hint},
                sentry => {extra => {content => $content},},);
        } else {
            $e->{sentry}{extra}{content} = $content if $content;
            throw $e;
        }
    };

    my $errors      = $response->{errors} // [];
    my $data        = $response->{data};
    my $error       = $data->{code};
    my $description = $data->{description} // '';
    if (@$errors || $error || $description ne 'Success') {
        throw Exception::API::Payoneer gettext('Payoneer request is not success'), undef, undef,
          sentry => {extra => {content => $response},};
    }

    return $data;
}

sub _get_payoneer_program_id {
    my ($self, $currency) = @_;

    throw Exception::API::Payoneer gettext('Currency is incorrect') unless $PAYONEER_CURRENCIES{$currency};

    my $program_id = $self->get_option('payoneer_program_id')->{$currency};
    throw Exception::API::Payoneer gettext('Payoneer program ID is not set for "%s"', $currency) unless $program_id;

    return $program_id;
}

sub get_payoneer_login_link {
    my ($self, $currency, $payee_id, %opts) = @_;

    my %options = hash_transform(\%opts, \@PAYONEER_LOGIN_LINK_OPTIONS);
    foreach (@NUMERIC_OPTIONS) {
        $options{$_} += 0 if exists $options{$_};
    }
    delete @opts{@PAYONEER_LOGIN_LINK_OPTIONS};
    my $response = $self->call(
        'proxy/payoneer/getloginlink/',
        ':post'    => TRUE,
        ':content' => to_json(
            {
                payee_id   => $payee_id,
                program_id => $self->_get_payoneer_program_id($currency),
                (keys %options ? (options => \%options) : ()),
            }
        ),
        %opts,
    );

    my $link = $response->{login_link};
    throw Exception::API::Payoneer gettext('Cannot get Payoneer login-link'), undef, undef,
      sentry => {extra => {content => $response},}
      unless $link;

    return $link;
}

sub get_payoneer_payee_status {
    my ($self, $currency, $payee_id, %opts) = @_;

    my $response;
    my $status;
    $response = $self->call(
        'proxy/payoneer/getpayeestatus/',
        ':post'    => TRUE,
        ':content' => to_json(
            {
                payee_id   => $payee_id,
                program_id => $self->_get_payoneer_program_id($currency),
            }
        ),
        %opts,
    );
    $status = $response->{status} // '';

    my $result;
    if ($status eq 'INACTIVE') {
        $result = FALSE;
    } elsif ($status eq 'ACTIVE') {
        $result = TRUE;
    } else {
        throw Exception::API::Payoneer gettext('Incorrect Payoneer payee status'), undef, undef,
          sentry => {extra => {content => $response},};
    }

    return $result;
}

sub check_tvm_service {
    my ($self) = @_;

    my $result;
    try {
        $self->get_payoneer_payee_status("USD", 1000);
    }
    catch {
        my ($e) = @_;
        if ($e->isa('Exception::API::HTTP') && $e->{response}->code == 401) {
            $result = TRUE;
        }
    };

    return $result;
}

1;
