package Application::Model::API::Yandex::FormatSystem;

use qbit;

use Exception::API::FormatSystem;
use Exception::Validation::BadArguments;
use Exception::NotFound;

use base qw(QBit::Application::Model::API::HTTP Application::Model::API::HTTPLogger);

use PiConstants qw($FORMAT_SYSTEM);

__PACKAGE__->model_accessors(memcached => 'QBit::Application::Model::Memcached');

sub accessor {'api_format_system'}

sub validate {
    my ($self, %opts) = @_;

    my $path   = 'schemas/pcode-7012/validate';
    my $locale = $self->get_option('locale');

    my $key = to_json({hash_transform(\%opts, [qw(design site role)]), lang => $locale}, pretty => TRUE);
    my $data = $self->memcached->get(format_system => $key);

    unless ($data) {
        $data = $self->call(
            $path,
            ':post'    => 1,
            ':content' => to_json($opts{design}),
            'role'     => $opts{role},
            'site'     => $FORMAT_SYSTEM->{site_versions}->{$opts{site}} // $opts{site},
            'lang'     => $locale,
        );
        try {
            $self->memcached->set(format_system => $key => $data, 60 * 60 * 24);
        };
    }
    return $data;
}

sub formats {
    my ($self, %opts) = @_;

    my $path = 'schemas/pcode-7012/formats/' . ($opts{format} // '');

    return $self->call(
        $path,
        'role' => $opts{role},
        (defined $opts{form} ? ('form' => $opts{form}) : ()),
        'site' => $FORMAT_SYSTEM->{site_versions}->{$opts{site}} // $opts{site},
        'lang' => $self->get_option('locale'),
    );
}

sub validate_native {
    my ($self, %opts) = @_;

    return $self->call(
        'native-master/validate',
        ':post'    => 1,
        ':content' => to_json(
            {
                template => $opts{template},
                name     => 'nativeDesign',
            }
        ),
    );
}

sub get_design_limits {
    my ($self, %opts) = @_;

    my $path = "default-formats";

    my $data = $self->call(
        $path,
        site => $FORMAT_SYSTEM->{site_versions}->{$opts{site}} // $opts{site},
        ':memcached' => 60 * 60 * 1,
    );
    return $data->{limit};
}

sub call {
    my ($self, $path, %opts) = @_;
    my $caller_sub_name = (split /::/, (caller(1))[3])[-1] // '';
    my $data;
    $opts{':headers'} = {
        Accept         => 'application/json',
        'Content-Type' => 'application/json',
        %{$opts{':headers'} // {}}
    };
    try {
        my $result = $self->SUPER::call($path, %opts);
        throw Exception::NotFound 'empty MSF reply' unless length $result;
        utf8::decode($result);
        $data = from_json($result);
    }
    catch {
        my ($exception) = @_;
        throw Exception::API::FormatSystem $exception, undef, undef,
          sentry => {fingerprint => ['FormatSystem', $caller_sub_name]};
    };
    return $data;
}

TRUE;
