package Application::Model::API::Yandex::HBF;

use qbit;

use base qw(QBit::Application::Model::API::HTTP);

use Exception::HBF::NotFound;

sub accessor {'api_hbf'}

=encoding UTF-8

=head2 get_elements_from_macros

Метод разворачивает макрос в список хостов/сетей/айпишников. Метод ходит в ручку `macros` сервиса `hbf` (host base firewall).

Пример:

    my @hosts = $app->api_hbf->get_hosts_from_macros('_C_PARTNER_RTB_FRONTENDS_PROD_');

После этого в @hosts будут 2 элемента:

    (
        'pi-rtb01e.yandex.ru',
        'pi-rtb01f.yandex.ru',
    )

Но в качестве элемента ответа может быть не только hostname, но и сеть (например '87.250.235.240/28' или
'2a02:6b8:b010:603e::/64') или ip адрес.

Метод бросает исключеня в случае каких-либо ошибок:

 * Exception::HBF::NotFound - если сервис `hbf` отвечает что не знает про такой макрос
 * Exception::API::HTTP - если по какой-то причине не получилось получить запрос от сервиса `hbf` (сервис 500-ит, пропал доступ, etc)
 * Так же возможны другие исключения (например, если сервис отвеил 200, но не валидным json-ом)

Пример работы с ручкой с помощью curl:

    $ curl --silent "http://hbf.yandex.net/macros/_C_PARTNER_RTB_FRONTENDS_PROD_?format=json" | jq .
    [
      "pi-rtb01e.yandex.ru",
      "pi-rtb01f.yandex.ru"
    ]

=cut

sub get_hosts_from_macros {
    my ($self, $macros) = @_;

    throw Exception::Validation::BadArguments 'Must specify macros' unless $macros;

    my @hosts;

    try {
        my $method = sprintf('macros/%s', $macros);
        my $json = $self->call(
            $method,
            format => 'json',

            # Если не указать этот параметр, то макрос _QLOUDNETS_ разворачивается в `1478@2a02:6b8:c00::/40`,
            # если параметр указан, то префикс `1478@` пропадает, в ответе находится сеть `2a02:6b8:c00::/40`
            trypo_format => 'none',
        );

        my $data = from_json $json;
        @hosts = @$data;
    }
    catch {
        my ($e) = @_;

        if (ref($e) eq 'Exception::API::HTTP' && $e->message() =~ /^400 Bad Request/) {
            throw Exception::HBF::NotFound 'Can\'t get hosts from macros ' . $macros;
        } else {
            throw $e;
        }
    };

    return @hosts;
}

1;
