
=encoding UTF-8

=head1 DESCRIPTION

HTTPMOL - это модель для работы с http сервиом "Мастер Отчетов на Логин".

=cut

package Application::Model::API::Yandex::HTTPMOL;

use base qw(QBit::Application::Model);

use Exception::API::HTTPMOL;
use Exception::API::HTTPMOL::TooManyGroupBy;

sub accessor {'api_http_mol'}

use qbit;
use HTTP::Tiny;
use Utils::Logger qw(), {logger => 'Application.Model.API.Yandex.HTTPMOL',};

=head2 get_data

Низкоуровневая ручка по работе с матером отчетов.
На входе нужно передать стуктуру с данными, которая будет отправлена в МОЛ.

Метод возвращает структуру с ответом МОЛ в том случае если не было ошибки.
Если была ошибка, то будет брошено исключение.

Эта ручка никак не валидирует данные, которые в нее переданы и ручка
ничего не знает про права/роли.

Дока про ручку с которой работает этот метод - L<https://wiki.yandex-team.ru/bannernajakrutilka/master-report-pi/>

Пример:

    my $data = $app->api_http_mol->get_data(
        countable_fields => [
            "Shows",
            "Clicks",
            "Cost",
        ],
        currency         => "RUB",
        date_from        => 20160701,
        date_to          => 20160701,
        filters_pre      => {
            RegionName => {
                not_like => [
                    "крым",
                ],
            },
        },
        group_by         => [
            "RegionID",
            "RegionName",
        ],
        limits           => {
            limit  => 2,
            offset => 0,
        },
    );

Это вернет что-то вроде:

    {
        data            => [
            [
                0,
                "",
                572,
                0,
                0,
            ],
            [
                1,
                "",
                500085,
                1145,
                0,
            ],
        ],
        duration        => 0.227,
        error_text      => "",
        header          => [
            "RegionID",
            "RegionName",
            "Shows",
            "Clicks",
            "Cost",
        ],
        host-aggregator => "bspstat_1",
        stat_time       => 20160722171826,
        status          => 0,
        total_rows      => 804,
        totals          => {
            Clicks => 200661,
            Cost   => 0,
            Shows  => 124255667,
        },
    }

=cut

sub get_data {
    my ($self, %opts) = @_;

    my $body = to_json(\%opts, canonical => TRUE);

    utf8::encode($body) if utf8::is_utf8($body);

    my $id = $self->_get_random_string();
    INFO("Request $id: " . $body);

    my $timeout_seconds = 600;

    # Используется HTTP::Tiny, а не QBit::Application::Model::API::HTTP
    # чтобы ускорить работу и чтобы не писать логи в базу
    my $response = HTTP::Tiny->new(timeout => $timeout_seconds)->request(
        'POST',
        $self->get_option('url') . '/export/master-report-pi.cgi',
        {
            headers => {'Content-Type' => 'application/json'},
            content => $body,
        }
    );

    if ($response->{'status'} ne '200') {
        my $message;

        if ($response->{'status'} eq '599' && (($response->{'content'} // '') =~ /^Timed out /)) {
            $message = sprintf('MOL server timeout after %s seconds', $timeout_seconds);
        } else {
            $message = sprintf('Got not 200 status code: %s', $response->{'status'});
        }

        throw Exception::API::HTTPMOL $message,
          sentry => {extra => {mol_request_id => $id}, fingerprint => ['HTTPMOL', $response->{'status'}]};
    }

    utf8::decode($response->{'content'});
    my $parsed_response;

    try {
        $parsed_response = from_json($response->{'content'});
    }
    catch {
        my $message = 'Response is not a valid json.';
        throw Exception::API::HTTPMOL $message, sentry =>
          {extra => {mol_response_content => $response->{'content'}}, fingerprint => ['HTTPMOL', 'Bad response']};
    };

    if ($parsed_response->{'status'} == 0) {
        INFO(
            "Response $id: "
              . to_json(
                {
                    status     => $parsed_response->{'status'},
                    duration   => $parsed_response->{'duration'},
                    total_rows => $parsed_response->{'total_rows'},
                },
                canonical => TRUE
              )
        );
    } else {
        my $message = "Response $id: "
          . to_json(
            {
                status     => $parsed_response->{'status'},
                duration   => $parsed_response->{'duration'},
                error_text => $parsed_response->{'error_text'},
            },
            canonical => TRUE
          );
        if ($parsed_response->{'error_text'} =~ /Too many fields in "group_by"/) {
            INFO($message);
            throw Exception::API::HTTPMOL::TooManyGroupBy;
        } else {
            throw Exception::API::HTTPMOL sprintf('Got error form master report: %s', $parsed_response->{'error_text'}),
              sentry => {
                fingerprint => ['HTTPMOL', 'Bad response status', $parsed_response->{'status'}],
                extra       => {
                    message => $message,
                    status  => $parsed_response->{'status'}
                }
              };
        }
    }

    return $parsed_response;
}

sub _get_random_string {
    my ($self) = @_;

    my @c = ("a" .. "z", "A" .. "Z", 0 .. 9);

    # Удивительно, но без этого rand() возвращал одно и то же
    srand();

    my $string;
    $string .= $c[rand(@c)] for (1 .. 12);

    return $string;
}

TRUE;
