package Application::Model::API::Yandex::MediaStorage;

use qbit;

use base qw(
  QBit::Application::Model::API::Yandex::MediaStorage
  Application::Model::API::HTTPLogger
  );

use MIME::Base64 qw(decode_base64 encode_base64);
use Digest::SHA1;

use Exception::Validation::BadArguments;

sub accessor {'api_media_storage'}

__PACKAGE__->model_accessors(partner_db => 'Application::Model::PartnerDB::Statistics');

=encoding UTF-8

=head1 save

Метод загружает файл в медийный сторадж и записывает его в таблицу mds. Метод
возвращает публичный урл файла.

    my $url = $app->api_media_storage->save(
        file_name => 'the_file.swf',
        base_64_content => "Li4u\n\n",   # Это строка '...' в base64
    );

Mds переписывает файлы с одинаковым именем, поэтому в качестве имени файла
используется хеш от содержимого.

В случае возникновения ошибок метод выбрасывает исключение.

Пример публичного урла:

    http://storage.mdst.yandex.net/get-partner/1184/bq46WwYsbQ158HDCbm1iSGtAy0Y.swf

=cut

sub save {
    my ($self, %opts) = @_;

    throw Exception::Validation::BadArguments gettext('Expected "file_name"')       unless $opts{'file_name'};
    throw Exception::Validation::BadArguments gettext('Expected "base_64_content"') unless $opts{'base_64_content'};
    throw Exception::Validation::BadArguments gettext('"file_name" is too long') if length($opts{'file_name'}) > 256;

    my $dt = curdate(oformat => 'db_time');

    my $content = decode_base64($opts{'base_64_content'});
    throw Exception::Validation::BadArguments gettext("Can't decode base64 string") unless length($content);
    throw Exception::Validation::BadArguments gettext('File is too big') if length($content) > 524288;

    my $digest = Digest::SHA1->new->add($self->get_option('cur_user')->{'id'} . $dt . $content)->b64digest;
    $digest =~ s{\+}{-}g;
    $digest =~ s{/}{_}g;

    my $extension;
    $opts{'file_name'} =~ /([^.]*)$/;

    if (defined $1) {
        $extension = ".$1";
        $extension = lc($extension);
    } else {
        $extension = "";
    }

    my $url = $self->get_link_with_upload("$digest$extension", $content);

    $self->partner_db->mds->add(
        {
            url       => $url,
            user_id   => $self->get_option('cur_user')->{'id'},
            upload_dt => $dt,
            file_name => $opts{'file_name'},
        }
    );

    return $url;
}

TRUE;
