
=encoding UTF-8

=cut

package Application::Model::API::Yandex::MediaStorage::S3;

use base qw(QBit::Application::Model);

use PiConstants qw($DEFAULT_HTTP_TIMEOUT);

use qbit;

use Net::INET6Glue::INET_is_INET6;
use LWP::UserAgent;
use HTTP::Headers;
use HTTP::Date;
use MIME::Base64 qw(encode_base64);
use URI::Escape qw(uri_escape_utf8);
use Digest::HMAC_SHA1;
use Digest::MD5;
use Fcntl qw(:seek);
use URI;
use XML::Simple qw(XMLin);

use Exception::API::MediaStorage::S3;

my $S3_CUSTOM_HEADER_PREFIX = 'x-amz-';
my $S3_UPLOAD_CHUNK_SIZE    = 4096;

sub accessor {'api_media_storage_s3'}

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    $self->{__UA__} = LWP::UserAgent->new(
        timeout => $self->get_option('timeout', $DEFAULT_HTTP_TIMEOUT),
        ssl_opts => {SSL_ca_path => '/etc/ssl/certs/'}
    );
}

=head2 request

Пример входных параметров

my %opts = (
    method  => 'GET', # 'PUT', 'HEAD', 'DELETE'
    key     => 'get_publisher_stat__2017-04-27.tsv',
    headers => HTTP::Headers->new(),
    request_content => $send_this_scalar_as_content,
    request_content_file => 'read_this_file_by_chunks_and_send_as_content.txt',
    response_content_file => 'put_response_content_into_this_file.txt',
);

=cut

sub request {
    my ($self, %opts) = @_;

    $opts{key} //= '';
    $opts{headers} //= HTTP::Headers->new;

    my $ua  = $self->{__UA__};
    my $key = $self->get_option('bucket_name') . '/' . $opts{key};

    my $ctx = Digest::MD5->new();
    if (defined $opts{request_content_file}) {
        open(my $fh, '<:raw', $opts{request_content_file})
          or throw Exception::API::MediaStorage::S3 "Failed to open $opts{request_content_file}: $!";
        $ctx->addfile($fh);
        seek($fh, 0, SEEK_SET)
          or throw Exception::API::MediaStorage::S3 "Failed to seek $opts{request_content_file}: $!";
        $opts{request_content} = sub {
            $fh->read(my $buf, $S3_UPLOAD_CHUNK_SIZE);
            $buf
        };
        $opts{headers}->header('Content-MD5' => $ctx->b64digest . '==');
    } elsif (defined $opts{request_content}) {
        $ctx->add($opts{request_content});
        $opts{headers}->header('Content-MD5' => $ctx->b64digest . '==');
    }

    my $headers = $self->_add_auth_header($opts{headers}, $opts{method}, $key);

    my $uri = URI->new($self->get_option('url'));
    $uri->path($key);
    $uri->query_form($opts{query_form} // {});

    my $request = HTTP::Request->new($opts{method}, $uri, $headers, $opts{request_content},);

    my $response = $ua->request($request, $opts{response_content_file});

    if ($response->is_error) {
        my $http_code = $response->code;
        my $error     = $self->_parse_response($response->content);
        my $code      = $error->{Code};
        my $message   = $error->{Message};
        my $resource  = $error->{Resource};

        my $exception_message = sprintf(
            '%s "%s" return %s(%s): %s',
            $opts{method} // '',
            $opts{key}    // '',
            $code         // '',
            $http_code    // '',
            $message      // '',
        );

        throw Exception::API::MediaStorage::S3 $exception_message, sentry => {
            extra => {
                http_method => $opts{method},
                http_code   => $http_code,
                s3_code     => $code,
                s3_key      => $opts{key},
                s3_message  => $message,
            },
        };
    }

    return $response;
}

=head2 list

=cut

sub list {
    my ($self, $prefix) = @_;

    my @list;

    my $parsed;
    do {
        my $response = $self->request(
            method     => 'GET',
            query_form => {$list[-1] ? ('marker' => $list[-1]) : (), $prefix ? ('prefix' => $prefix) : (),}
        );

        $parsed = $self->_parse_response($response->content);

        push @list, (map {$_->{Key}} @{$parsed->{Contents}});
    } while ($parsed->{IsTruncated} eq 'true');

    return \@list;
}

=head2 put

=cut

sub put {
    my ($self, $key, $content_type, $scalar) = @_;

    my $response = $self->request(
        method          => 'PUT',
        key             => $key,
        request_content => $scalar,
        headers         => HTTP::Headers->new('Content-Type' => $content_type),
    );

    return 1;
}

=head2 put_file

=cut

sub put_file {
    my ($self, $key, $content_type, $filename) = @_;

    my $response = $self->request(
        method               => 'PUT',
        key                  => $key,
        request_content_file => $filename,
        headers              => HTTP::Headers->new('Content-Type' => $content_type),
    );

    return 1;
}

=head2 get

=cut

sub get {
    my ($self, $key) = @_;

    my $response = $self->request(
        method => 'GET',
        key    => $key,
    );

    return {
        content      => $response->decoded_content,
        content_type => ($response->header('Content-Type') // undef),
    };
}

=head2 get_file

=cut

sub get_file {
    my ($self, $key, $filename) = @_;

    my $response = $self->request(
        method                => 'GET',
        key                   => $key,
        response_content_file => $filename,
    );

    return {content_type => ($response->header('Content-Type') // undef),};
}

=head2 delete

=cut

sub delete {
    my ($self, $key) = @_;

    my $response = $self->request(
        method => 'DELETE',
        key    => $key,
    );

    return 1;
}

sub _parse_response {
    my ($self, $xml) = @_;

    return XMLin($xml, KeyAttr => [], ForceArray => ['Contents']);
}

#Mostly copypaste from Net::Amazon::S3
sub _add_auth_header {
    my ($self, $headers, $method, $path) = @_;

    my $aws_access_key_id     = $self->get_option('access_key_id');
    my $aws_secret_access_key = $self->get_option('access_secret_key');

    if (not $headers->header('Date')) {
        $headers->header(Date => time2str(time));
    }

    my $canonical_string = $self->_canonical_string($method, $path, $headers);
    my $encoded_canonical = $self->_encode($aws_secret_access_key, $canonical_string);
    $headers->header(Authorization => "AWS $aws_access_key_id:$encoded_canonical");

    return $headers;
}

# generate a canonical string for the given parameters.  expires is optional and is
# only used by query string authentication.
sub _canonical_string {
    my ($self, $method, $path, $headers, $expires) = @_;
    my %interesting_headers = ();
    while (my ($key, $value) = each %$headers) {
        my $lk = lc $key;
        if (   $lk eq 'content-md5'
            or $lk eq 'content-type'
            or $lk eq 'date'
            or $lk =~ /^$S3_CUSTOM_HEADER_PREFIX/)
        {
            $interesting_headers{$lk} = trim($value);
        }
    }

    # these keys get empty strings if they don't exist
    $interesting_headers{'content-type'} //= '';
    $interesting_headers{'content-md5'}  //= '';

    # just in case someone used this.  it's not necessary in this lib.
    $interesting_headers{'date'} = ''
      if $interesting_headers{'x-amz-date'};

    # if you're using expires for query string auth, then it trumps date
    # (and x-amz-date)
    $interesting_headers{'date'} = $expires if $expires;

    my $buf = "$method\n";
    foreach my $key (sort keys %interesting_headers) {
        if ($key =~ /^$S3_CUSTOM_HEADER_PREFIX/) {
            $buf .= "$key:$interesting_headers{$key}\n";
        } else {
            $buf .= "$interesting_headers{$key}\n";
        }
    }

    # don't include anything after the first ? in the resource...
    $path =~ /^([^?]*)/;
    $buf .= "/$1";

    # ...unless there any parameters we're interested in...
    if ($path =~ /[&?](acl|torrent|location|uploads|delete)($|=|&)/) {
        $buf .= "?$1";
    } elsif (my %query_params = URI->new($path)->query_form) {
        #see if the remaining parsed query string provides us with any query string or upload id
        if ($query_params{partNumber} && $query_params{uploadId}) {
            #re-evaluate query string, the order of the params is important for request signing, so we can't depend on URI to do the right thing
            $buf .= sprintf("?partNumber=%s&uploadId=%s", $query_params{partNumber}, $query_params{uploadId});
        } elsif ($query_params{uploadId}) {
            $buf .= sprintf("?uploadId=%s", $query_params{uploadId});
        }
    }

    return $buf;
}

# finds the hmac-sha1 hash of the canonical string and the aws secret access key and then
# base64 encodes the result (optionally urlencoding after that).
sub _encode {
    my ($self, $aws_secret_access_key, $str, $urlencode) = @_;
    my $hmac = Digest::HMAC_SHA1->new($aws_secret_access_key);
    $hmac->add($str);
    my $b64 = encode_base64($hmac->digest, '');
    if ($urlencode) {
        return $self->_urlencode($b64);
    } else {
        return $b64;
    }
}

sub _urlencode {
    my ($self, $unencoded) = @_;
    return uri_escape_utf8($unencoded, '^A-Za-z0-9_-');
}

1;
