package Application::Model::API::Yandex::TVM;

use qbit;

use base qw(QBit::Application::Model::API::HTTP);
use Exception::Validation::BadArguments;
use Exception::API;
use Exception::API::HTTP;
use File::Slurp qw(read_file);
use Utils::Logger qw(INFOF INFO);
use PiConstants qw($TVM_HTTP_HEADER_NAME);

sub accessor {'api_tvm'}

sub init {
    my ($self) = @_;

    $self->SUPER::init();
    $self->_set_new_auth_token();

    return TRUE;
}

sub call_ping {
    my ($self) = @_;

    return $self->call_command("ping");
}

sub get_service_ticket {
    my ($self, $tvm_dst_alias, $tvm_src) = @_;
    $tvm_src //= $self->get_option('self_tvm_id');

    my $response_json = $self->call_command('tickets', dsts => $tvm_dst_alias, src => $tvm_src);
    my $response = from_json($response_json);

    if ($ENV{SPOIL_TVM_TICKET}) {
        $response->{$tvm_dst_alias}->{ticket} .= 'xxx' if $response;
    }

    return $response ? $response->{$tvm_dst_alias}->{ticket} : undef;
}

sub check_ticket {
    my ($self, $ticket, $tvm_dst) = @_;
    $tvm_dst //= $self->get_option('self_tvm_id');
    my $response;
    try {
        $response = from_json(
            $self->call_command(
                'checksrv',
                headers => {$TVM_HTTP_HEADER_NAME => $ticket},
                dst     => $tvm_dst
            )
        );
    }
    catch {
        my ($exception) = @_;
        INFOF("CHECK_TICKET(%s): %s", $ticket // '<undef>', $exception->message);
    };
    return $response;
}

sub call_command {
    my ($self, $command, %params) = @_;

    my $result;
    try {
        $result = $self->_call_command($command, %params);
    }
    catch Exception::API::HTTP with {
        my ($exception) = @_;
        my $code = $exception->{response}->code;
        if ($code == 401 && $self->_set_new_auth_token) {
            $result = $self->_call_command($command, %params);
        }
        # while daemon is restarting try to connect & set new token (token is changed on restart)
        elsif ($code == 500 && ($exception->message =~ /connection refused/i)) {
            $self->_set_new_auth_token;
            $result = $self->_call_command($command, %params, attempts => 30);
        } else {
            throw Exception::API $exception, sentry => {fingerprint => ['TVM', 'call error', $code]};
        }
    }
    catch {
        my ($exception) = @_;
        throw Exception::API $exception, sentry => {fingerprint => ['TVM', 'other call error']};
    };
    return $result;
}

sub _call_command {
    my ($self, $command, %params) = @_;

    my $uri = sprintf('%s/%s', $self->get_option('url'), $command);
    my $tvm_auth_token = $self->get_option('auth_token');

    my $headers = delete $params{headers} || {};
    my $attempts = delete $params{attempts} // 1;

    my $response = $self->request(
        method     => 'GET',
        uri        => $uri,
        headers    => {'Authorization' => $tvm_auth_token, %$headers},
        url_params => \%params,
        attempts   => $attempts,
    );

    return $response;
}

sub _set_new_auth_token {
    my ($self) = @_;

    my $tvm_auth_token_path = $self->get_option('auth_token_path');

    my $token = $ENV{TVMTOOL_LOCAL_AUTHTOKEN} // (-e $tvm_auth_token_path ? read_file($tvm_auth_token_path) : '');

    $token =~ s/\s//g;

    unless ($token eq $self->get_option('auth_token')) {
        $self->set_option('auth_token', $token);
        return TRUE;
    }

    return FALSE;
}

1;
