package Application::Model::API::Yandex::YaMoney;

use qbit;

use base qw(QBit::Application::Model);

use PiConstants qw($DEFAULT_HTTP_TIMEOUT);

use Encode;
use Digest::SHA1 qw(sha1);
use MIME::Base64;
use XML::Simple;

# Чтобы включить дебаг вывод, нужно раскомментировать строку ниже
# use IO::Socket::SSL qw(debug3);;

use Exception::Validation::BadArguments;
use Exception::API::YaMoney;

sub accessor {'api_yamoney'}

=encoding UTF-8

=head2 is_account_identificated

Взаимодействует с сервером Я.Денег.

Возвращает true значение если выполняются все условия:

 * счет является валидным номером счета Я.Денег
 * счет принадлежит указанному пользователю
 * пользователь прошел идентификацию

    ->is_account_identificated(
        account    => '41003430873044', # номер кошелька в Яндекс.Деньгах
        lastname   => 'Партнерская',
        firstname  => 'Сеть',
        middlename => 'Яндекса',
        passport_s => 'AZ',     # серия паспорта
        passport_n => '123456', # номер паспорта
    );

иначе возвращает false значение.

=cut

sub is_account_identificated {
    my ($self, %opts) = @_;

    my $iddoc_hash = $self->_iddoc_hash($opts{'lastname'}, $opts{'firstname'}, $opts{'middlename'}, $opts{'passport'},);

    my $xml = qq{<?xml version="1.0" encoding="windows-1251"?>
<ID_DATA>
<IDDOC_HASH>$iddoc_hash</IDDOC_HASH>
</ID_DATA>};

    my $answer_xml = $self->_post(
        ACT_CD     => 'idDataCheck',
        VERSION    => '2.50',
        DSTACNT_NR => $opts{'account'},
        ID_DATA    => $xml,
    );

    my $answer_xml_data;
    try {
        $answer_xml_data = XMLin($self->_utf2cp1251($answer_xml));
    };

    if ($answer_xml_data->{code} eq '0') {
        # Все ок — счет принадлежит пользователя и пользователь прошел идентификацию
        return TRUE;
    } elsif ($answer_xml_data->{code} == 2) {
        # Ошибка сервера. Сбой. Неизвестно, поставлен ли запрос на обработку.
        throw Exception::API::YaMoney 'Yandex.Money server is saying that it is broken.',
          sentry => {fingerprint => ['YaMoney', 'Server broken']};
    } elsif ($answer_xml_data->{code} == 3) {
        # Неуспешно. Обработка завершена, результат - неуспешно. Деталировка ошибки – в коде ошибки (атрибут code тэга error).
        return FALSE;
    } else {
        # Такого вообще не должно быть. Это означает что мы не получили ответ от Я.Деньги или ответ не является правлиьным xml
        throw Exception::API::YaMoney $answer_xml, sentry => {fingerprint => ['YaMoney', 'bad answer']};
    }
}

=begin comment _is_valid_account_number

    $self->_is_valid_account_number(41001123456740);

Метод возвращает true/false значение в зависимости от того, является ли указанный
скаляр валидным номером счета в Яндекс.Деньгах. Метод не взаимодействует со внешними
системами.

Здесь реализован "Алгоритм проверки действительности строки, представляющей номер счета" из
документа "Привязка банковских карт к виртуальному счету в системе 'Яндекс.Деньги', HTTP-транспорт"

=end comment

=cut

sub _is_valid_account_number {
    my ($self, $maybe_a_number) = @_;

    return FALSE unless defined($maybe_a_number);
    return FALSE if $maybe_a_number =~ /\D/;
    return FALSE if length($maybe_a_number) > 32;
    return FALSE if length($maybe_a_number) < 11;

    my $n = substr($maybe_a_number, 0, 1);
    return FALSE unless $n;
    return FALSE if length($maybe_a_number) < $n + 4;
    my $z = substr($maybe_a_number, length($maybe_a_number) - 2, 2);
    return FALSE if $z eq "00";
    my $x = substr($maybe_a_number, 1,      $n);
    my $y = substr($maybe_a_number, 1 + $n, length($maybe_a_number) - 3 - $n);
    return FALSE if length($y) > 20;
    return FALSE unless $self->_account_number_redundancy($x, $y) eq $z;

    return TRUE;
}

=begin comment _account_number_redundancy

Тут реализована функция AccountNumberRedundancy из документа
"Привязка банковских карт к виртуальному счету в системе «Яндекс.Деньги», HTTP-транспорт"

Метод получает два числа и возвращает третье.

Метод необходим для работы _is_valid_account_number.

=end comment

=cut

sub _account_number_redundancy {
    my ($self, $x, $y) = @_;

    my $result = 0;

    my $a = 70;

    for (my $i = 0; $i < 30; $i++) {

        my $value = 0;

        if ($i < 20) {
            # проходим y
            if (length($y) > $i) {
                $value = substr($y, length($y) - 1 - $i, 1);
            }
        } else {
            # проходим x
            if (length($x) > ($i - 20)) {
                $value = substr($x, length($x) - 1 - ($i - 20), 1);
            }
        }

        $result = ($result + (($value || 10) * $a) % 99) % 99;

        $a = ($a * 13) % 99;

    }
    $result = $result % 99 + 1;

    return sprintf("%02i", $result);
}

=begin comment _iddoc_hash

    my $iddoc_hash = $self->_iddoc_hash(
        'Фамилия',
        'Имя',
        'Отчество',
        '1245 123456', # серия и номер паспорта
    );

Метод возвращает хэш персональных данных пользователя (строка в base64).

Правило формирование возвращаемого значения:

 * ФИО и номер паспорта нормализуются.
  * Нормализация ФИО - это замена всех букв Ё на Е и приведение к верхнему регистру.
  * Нормализация номера паспорта - удаление всех разделителей (в т.ч. пробела между серией и номером), удаление символов # и №.
    Латинские символы, для которых есть аналог кирилици перобразоываются в кирилицу. Полный набор заменяемых символов: ABEKMOPCTXH.
 * После этого производится конкатенация ФИО и номера паспорта в виде Ф_И_О_N, т.е. разделяя символом _.
 * Полученная строка приводится к массиву байт кодировкой windows-1251
 * Вычисляется хэш SHA1
 * Приводится обратно к строке Base64.

=end comment

=cut

sub _iddoc_hash {
    my ($self, $last_name, $first_name, $middle_name, $passport_info) = @_;

    foreach ($last_name, $first_name, $middle_name, $passport_info) {
        $_ //= '';
    }

    foreach ($last_name, $first_name, $middle_name) {
        s/^\s+|[^a-zA-Zа-яА-ЯёЁ\- ]|\s+$//g;
        s/ +/_/g;
    }

    my $concated = join "_", ($last_name, $first_name, $middle_name);

    $concated = uc($concated);
    $concated =~ s/Ё/Е/g;
    $passport_info = uc($passport_info);
    $passport_info =~ tr/ABEKMOPCTXH/АВЕКМОРСТХН/;
    $passport_info =~ s/[^\dа-яА-Яa-zA-Z]//g;
    $concated .= "_$passport_info";

    my $converted = $self->_utf2cp1251($concated);
    my $sha1      = sha1($converted);
    my $base64    = encode_base64($sha1, '');

    return $base64;
}

=begin comment _utf2cp1251

    $self->_utf2win('фыва'); # ôûâà;

На входе строка в кодировке utf8, на выходе строка в кодировке cp1251.

Символы которых нет в cp1251 будут преобразованы в xml сущности, например:

    ☢ => &#9762;

=cut

sub _utf2cp1251 {
    my ($self, $original_string) = @_;

    my $string = $original_string;
    Encode::_utf8_off($string);  # Иначе падаем с ошибкой 'Cannot decode string with wide characters'
    Encode::from_to($string, 'utf-8', 'cp1251', Encode::FB_HTMLCREF);

    return $string;
}

=begin comment _post

B<Параметры:> 1) % с данными для передачи в Я.Деньги

B<Возвращаемое значение:> 1) $ с ответом Я.Денег

Пример использвования:

    LogDump $self->_post(
        ACT_CD      => 'idDataCheck',
        VERSION     => '2.50',
        TR_NR       => 1,
        DSTACNT_NR  => 41001122521533,
        ID_DATA     => $converted,
    );

=end comment

=cut

sub _post {
    my ($self, %data) = @_;

    my $ctx = new IO::Socket::SSL::SSL_Context(
        SSL_use_cert    => 1,
        SSL_verify_mode => 0x00,
        SSL_key_file    => '/etc/partners-ssl/yamoney_privkey.pem',
        SSL_cert_file   => '/etc/partners-ssl/yamoney.pem',
    );
    IO::Socket::SSL::set_default_context($ctx);

    my $ua = LWP::UserAgent->new(timeout => $self->get_option('timeout', $DEFAULT_HTTP_TIMEOUT));

    my $res = $ua->post($self->get_option('url'), \%data);

    if ($res->is_success()) {
        return $res->decoded_content();
    } else {
        throw Exception::API::YaMoney $res->status_line(), sentry => {fingerprint => ['YaMoney', 'failed request']};
    }
}

TRUE;
