package Application::Model::ActionLog;

use qbit;

use Utils::JSON qw(fix_type_for_complex);
use base qw(Application::Model::DBManager::Base RestApi::DBModel);

__PACKAGE__->abstract_methods(qw(accessor db_table_name));

sub get_product_name {
    my ($self) = @_;
    my $model = $self->_get_model_of_action();

    return gettext('Action logs for "%s"', $model->get_product_name());
}

sub get_structure_model_accessors {
    return {
        users      => 'Application::Model::Users',
        partner_db => 'Application::Model::PartnerDB',
    };
}

sub get_structure_model_fields {
    my ($self) = @_;

    my $model_fields = {
        id        => {db => TRUE, default => TRUE, type => 'number', pk  => 1},
        public_id => {db => TRUE, db_expr => 'id', type => 'string', api => 1},
        dt        => {db => TRUE, default => TRUE, type => 'string', api => 1},
        user_id   => {db => TRUE, default => TRUE, type => 'number'},
        login     => {
            depends_on => ['user_id', 'users.login'],
            get        => sub {
                return $_[0]->{'users'}{$_[1]->{'user_id'}}{'login'} // '';
            },
            type => 'string',
            api  => 1
        },
        old_multistate      => {db => TRUE, default => TRUE, type => 'number', api => 1},
        old_multistate_name => {
            depends_on => ['old_multistate'],
            get        => sub {
                my $accessor = $_[0]->model->_action_model_accessor();
                $_[0]->model->app->$accessor->get_multistate_name($_[1]->{'old_multistate'});
            },
            type => 'string',
            api  => 1
        },
        new_multistate      => {db => TRUE, default => TRUE, type => 'number', api => 1},
        new_multistate_name => {
            depends_on => ['new_multistate'],
            get        => sub {
                my $accessor = $_[0]->model->_action_model_accessor();
                $_[0]->model->app->$accessor->get_multistate_name($_[1]->{'new_multistate'});
            },
            type => 'string',
            api  => 1
        },
        action           => {db => TRUE, default => TRUE,     type => 'string', api => 1},
        comment          => {db => TRUE, type    => 'string', api  => 1},
        available_fields => {
            get => sub {
                return $_[0]->model->get_available_fields();
            },
            api      => 1,
            type     => 'complex',
            fix_type => \&fix_type_for_complex,
        }
    };

    my $table               = $self->_get_table();
    my $model               = $self->_get_model_of_action();
    my @model_pk            = @{$table->{'elem_table_pk'}};
    my %action_log_map      = map {("elem_${_}" => $_)} @model_pk;
    my @action_log_model_pk = keys(%action_log_map);
    my $fields              = $model->get_model_fields();
    foreach my $field (@action_log_model_pk) {
        $model_fields->{$field} = clone($fields->{$action_log_map{$field}});

        # new keys
        $model_fields->{$field}{'api'} = 1;

        # remove keys
        delete($model_fields->{$field}{'pk'});
        delete($model_fields->{$field}{'need_check'});

        unless ($field eq 'elem_id') {
            $model_fields->{$action_log_map{$field}} = {
                db      => TRUE,
                db_expr => $field,
                type    => 'string',
                api     => 1,
            };
        }
    }

    $model_fields->{'elem_public_id'} = {
        depends_on => \@action_log_model_pk,
        type       => 'string',
        get        => sub {
            my ($self, $obj) = @_;
            my $accessor = $self->model->_action_model_accessor();
            my %model_obj = map {($action_log_map{$_} => $obj->{$_})} @action_log_model_pk;
            $self->model->app->$accessor->public_id(\%model_obj);
        },
        api => 1,
    };

    if ($table->{'with_opts'}) {
        $model_fields->{'opts'} = {db => TRUE, type => 'string'};
        $model_fields->{'data_opts'} = {
            depends_on => [qw(opts)],
            get        => sub {
                return eval {from_json($_[1]->{'opts'})} // {};
            },
            type => 'complex',
            api  => 1
        };
    }

    return $model_fields;
}

sub get_structure_model_filter {
    my ($self) = @_;

    my $model_filter = {
        db_accessor => 'partner_db',
        fields      => {
            id      => {type => 'number', label => d_gettext('ID')},
            dt      => {type => 'date',   label => d_gettext('Time of action')},
            action  => {type => 'text',   label => d_gettext('Action')},
            comment => {type => 'text',   label => d_gettext('Comment')},
            user_id => {type => 'number', label => d_gettext('User ID')},
            users   => {
                type           => 'subfilter',
                model_accessor => 'users',
                field          => 'user_id',
                fk_field       => 'id',
                label          => d_gettext('Login'),
            },
        }
    };

    my $table  = $self->_get_table();
    my $model  = $self->_get_model_of_action();
    my $filter = $model->get_model_filter_fields();
    foreach my $elem (@{$table->{'elem_table_pk'}}) {
        my $elem_field = clone($filter->{$elem});
        delete($elem_field->{'db_filter'});
        if ($elem_field->{'type'} eq 'publicid') {
            $elem_field->{'type'} = 'text';
            delete($elem_field->{'regexp'});
        }
        $model_filter->{'fields'}{"elem_${elem}"} = $elem_field;

        unless ($elem eq 'id') {
            my $elem_field_alias = clone($elem_field);
            $elem_field_alias->{'db_filter'} = sub {
                return ["elem_${elem}" => $_[1]->[1] => \$_[1]->[2]];
            };

            $model_filter->{'fields'}{$elem} = $elem_field_alias;
        }
    }

    return $model_filter;
}

sub get_available_fields {
    my ($self) = @_;

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);
    # Don't check right for login

    return \%fields;
}

sub api_available_fields {
    my ($self) = @_;
    my @fields = $self->SUPER::api_available_fields();
    #Don't check right for login
    push @fields, 'login';

    return @fields;
}

sub api_available_actions { }

sub query_filter {
    my ($self, $filter) = @_;

    my $model = $self->_get_model_of_action();

    $filter->and({user_id => $self->get_option('cur_user', {})->{'id'}})
      unless $self->check_rights($model->get_right('view_action_log'));

    return $filter;
}

sub init {
    my ($self) = @_;

    $self->SUPER::init();
    $self->model_fields($self->get_structure_model_fields());
    $self->model_filter($self->get_structure_model_filter());
}

sub related_models {
    my ($self) = @_;

    return {
        users => {
            accessor => 'users',
            filter   => sub {
                +{id => array_uniq(map {$_->{'user_id'}} @{$_[1]})};
            },
            key_fields => ['id'],
        },
    };
}

sub get_table_elem_keys {
    my ($self) = @_;

    my $table    = $self->_get_table();
    my @model_pk = @{$table->{'elem_table_pk'}};

    return \@model_pk;
}

sub have_opts {
    my ($self) = @_;

    return $self->_get_table()->{'with_opts'};
}

sub _action_model_accessor {
    my ($self) = @_;

    my %exceptions = (moderation_queue_action_log => 'moderation',);

    my $accessor = $self->accessor();

    if (exists($exceptions{$accessor})) {
        $accessor = $exceptions{$accessor};
    } else {
        $accessor =~ s/_action_log$//;
    }

    return $accessor;
}

sub _get_model_of_action {
    my ($self) = @_;

    my $accessor = $self->_action_model_accessor();

    if ($self->app->can($accessor)) {
        return $self->app->$accessor;
    }

    return undef;
}

sub _get_table {
    my ($self) = @_;
    my $db_table_name = $self->db_table_name();
    return $self->partner_db->$db_table_name;
}

TRUE;
